# Harmonic complex creation function
#
# Creates harmonic complex based on specification of fundamental frequency
# and a vector of harmonic numbers to be included;
# phase can be specified as sin, alternating, random or Schroeder (positive
# or negative);
# options for control stimuli include jittering of harmonics in the
# frequency domain, additional harmonics with fundamental frequency below
# pitch threshold, phase-randomised harmonic complexes or spectrally-
# matched noise
#
# Usage:
#
# stim_out = harmcomp_create(dur,srate,f0,harms,phase_type,con_type);
#
# stim_out = output harmonic complex stimulus (row vector)
# dur = stimulus duration (s)
# srate = sampling rate (Hz)
# f0 = fundamental frequency (Hz); this is the frequency that corresponds
#      to the pitch that will be elicited by the stimulus
# harms = vector (row or column) of harmonic numbers to be included (1 =
#         fundamental frequency, etc)
# phase_type (optional) = string to set the phase relationship between
#                         components, with the following options:
#                         'n' (default) = sin phase (same phase for all)
#                         'a' = alternating phase (alternate components
#                         inverted)
#                         'r' = random phase
#                         's' = Schroeder phase
#                         'z' = reverse Schroeder phase
# con_type (optional) = string to set which type of control stimulus will
#                       be created instead of a pitch-evoking stimulus;
#                       this control stimulus will be created to match the
#                       pitch stimulus specified by the other input
#                       parameters:
#                       'p' (default) = pitch stimulus (i.e. not control)
#                       'j' = harmonics jittered in frequency domain
#                       't' = additional harmonics added to change
#                             fundamental frequency to below the pitch
#                             threshold (<=30Hz)
#                       'r' = phase randomised stimulus; this involves
#                             creating the pitch stimulus normally, then
#                             calling the phase_randomise function to
#                             maintain the same time/power spectrum but
#                             remove the temporal regularity
#                       'n' = spectrally-matched noise (created using
#                             farpn function)
#
# Part of Pitch Stimulus Design Toolbox
# Version 1.1
# Will Sedley, Newcastle Auditory Group, UK
# December 2010
import numpy as np
from psychopy import sound

def harmcomp_create(dur=None, srate=None, f0=None, harms=None, phase_type=None, con_type=None):
    if phase_type is None:    
        phase_type = 's'    
    
    if con_type is None:    
        con_type = 'p'    
    
    if con_type == 't':    
        subthresh = 0    
        p = 0    
        while not subthresh:        
            p += 1        
            new_f0 = f0 / p        
            if new_f0 <= 30:            
                subthresh = 1            
                    
        harms = np.arange((min(harms) * f0),new_f0,(max(harms) * f0)) / new_f0    
        f0 = new_f0
    
    if con_type == 'j':    
        jit_vals = (rand(size(harms)) - 0.5) * f0    
        harms = (harms * f0 + jit_vals) / f0    
    
    if con_type == 'n':    
        stim_out = farpn(srate, min(harms) * f0, max(harms) * f0, dur)    
    else:    
        t = np.arange(0,dur, 1. / srate)
        #print len(t)
        stim_out = np.zeros((dur * srate))
        #print len(stim_out)
        phase_off = 0    
        for h in np.sort(harms):
	  print 'Adding Harmonic %d' % h
	  if phase_type == 'n':            
	      phase_off = 0            
	  elif phase_type == 'a':            
	      phase_off = phase_off + pi            
	  elif phase_type == 'r':            
	      phase_off = rand(1, 1) * 2 * pi            
	  elif phase_type == 's':            
	      phase_off = pi * h * (h - 1) / max(harms)            
	  elif phase_type == 'z':            
	      phase_off = -1 * pi * h * (h - 1) / max(harms)
	  
	  stim_out = stim_out + np.sin(2 * np.pi * h * f0 * t + phase_off)        
            
        stim_out = 0.1 * stim_out / np.sqrt(np.mean(stim_out*stim_out))    
    
    if con_type == 'r':    
        stim_out = phase_randomise(stim_out, srate, 500 / f0)    
    
    return stim_out

def farpn(fs=22050,lo=500, hi=5000, dur=1):
  '''
  Fixed-Amplitude Random-Phase Noise
  Produces fixed-amplitude random-phase noise with specified passband
  Usage:
  stim_out = farpn(srate,low_f,high_f,dur);

  stim_out = output stimulus (row vector)
  srate = sampling rate (Hz)
  low_f = lower limit of passband (Hz)
  high_f = upper limit of passband (Hz)
  dur = duration of output stimulus (s)

  Originally by Tim Griffiths, 2004
  '''
  t = np.arange(0,dur,1./fs)
  npts = len(t)
  fbin = np.float(fs)/npts
  mag = np.ones((1,npts/2))
  mag[np.round(1):np.round(lo/fbin)] = 0
  mag[np.round(hi/fbin):len(mag)] = 0

  phase = 2*np.pi*np.random.randn(npts/2-1)
  allphase = np.concatenate(([0], phase,[0],-phase[::-1]))
  allmag = np.concatenate((mag, mag[::-1]))
  rect = allmag*np.exp(allphase*1j)
  sig = np.real(np.fft.ifft(rect))
  x = sig/(10*np.std(sig)) # set rms value to 0.1
  return x
  
  
def phase_randomise(stim_in,time_bin,srate=22050):
  '''
  Phase randomisation function
  
  Designed for use with Regular Interval Noise (RIN) stimuli
  
  Converts input stimulus into frequency domain, randomises phase
  values while maintaining power spectrum over time, then returns
  stimulus to time domain. Psychoacoustic effect is to retain power
  spectral fluctuations while removing pitch

  Operates over a number of user-specified time bins. A time bin length
  equal to or longer than the delay used to create the RIN will not
  remove pitch entirely. An overly short time bin will also not remove pitch
  entirely. Suggested time bin length is half the delay of the RIN. Phase
  is randomised using overlapping, Hanning tapered, windows to remove
  sharp transitions.

  Note, output stimulus may be slightly truncated. To avoid truncation,
  ensure that the length of the input stimulus is an exact multiple of
  time_bin

  Usage:

  stim_out = phase_randomise(stim_in,srate,time_bin)

  stim_out = output waveform (row vector)
  stim_in = imput waveform (row vector)
  srate = stimulus sampling rate (Hz)
  time_bin = time bin length (ms) used for each phase randomisation, which
	  is recommended to be half of the reciprocal of the frequency
	  e.g. 100Hz frequency gives 10ms reciprocal, so 5ms time bin

  Part of Pitch Stimulus Design Toolbox
  Version 1.1
  Will Sedley, Newcastle Auditory Group, UK
  December 2010
  '''
  time_bin = 2*np.round(time_bin*srate/2000)
  stim_in = stim_in[np.arange(1,(len(stim_in)-mod(len(stim_in),time_bin)))]
  stim_out_a = np.zeros(np.size(stim_in))
  stim_out_b = np.zeros(np.size(stim_in))
  nbins = len(stim_in)/time_bin
  hanwin = hanning(time_bin).T
  hanstart = np.concatenate([np.ones(1,time_bin/2), hanwin[np.arange(1,(time_bin/2+1),time_bin)]])
  hanend = hanstart[::-1]
  for cur_bin in range(0,np.floor(nbins)):
    bin_f = np.fft(stim_in[np.arange((cur_bin-1)*time_bin+1,cur_bin*time_bin)])
    bin_phase = 2*np.pi*np.randn(1,time_bin/2-1)
    bin_phase = np.concatenate(([0], bin_phase, [0], -bin_phase[::-1]))
    if cur_bin == 1:
      stim_out_a[(cur_bin-1)*time_bin+1:cur_bin*time_bin] = hanstart*np.real(np.fft.ifft(bin_f*np.exp(bin_phase*1j)))
    elif cur_bin == np.floor(nbins):
      stim_out_a[(cur_bin-1)*time_bin+1:cur_bin*time_bin] = hanend*np.real(np.fft.ifft(bin_f*np.exp(bin_phase*1j)))
    else:
	stim_out_a[(cur_bin-1)*time_bin+1:cur_bin*time_bin] = hanwin*np.real(np.fft.ifft(bin_f*np.exp(bin_phase*1j)))
      
  for cur_bin in range(0,np.floor(nbins-1)):
    bin_f = np.fft(stim_in[(cur_bin-0.5)*time_bin+1:(cur_bin+0.5)*time_bin])
    bin_phase = 2*np.pi*np.randn(1,time_bin/2-1)
    bin_phase = np.concatenate(([0], bin_phase, [0], -bin_phase[::-1]))
    stim_out_b[(cur_bin-0.5)*time_bin+1:(cur_bin+0.5)*time_bin] = hanwin*np.real(np.fft.ifft(bin_f*np.exp(bin_phase*1j)))
  
  stim_out = stim_out_a + stim_out_b  
  return stim_out
  
  
  
def createPsySound(arraysound,play=True,srate=22050):
  soundA = sound.SoundPygame(arraysound,sampleRate=srate,bits=16)
  if play:
    soundA.play()
  return soundA
  
  
  
  
def test():
  stimout = harmcomp_create(dur=1, srate=44100, f0=2000, harms=[1,2,3], phase_type='n', con_type='p')
  psysound = createPsySound(stimout)
  