##########################################################################
#                                                                        #
#    copyright (c) 2004 Royal Belgian Institute for Natural Sciences     #
#                       and contributors                                 #
#                                                                        #
##########################################################################

""" APABibrefStyle Class
    Based on the APA (American Psychological Association) Bibliography style
    as described here:
    http://www.english.uiuc.edu/cws/wworkshop/bibliography_style_handbookapa.htm
    http://owl.english.purdue.edu/workshops/hypertext/apa/sources/reference.html
    http://www.westwords.com/guffey/apa.html
"""


# Zope stuff
from Globals import InitializeClass
from App.Dialogs import MessageDialog
from Products.CMFCore.utils import getToolByName

# CMFBibliographyAT stuff
from Products.CMFBibliographyAT.config import REFERENCE_TYPES

# Bibliolist stuff
from base import BibrefStyle
from Products.ATBiblioStyles.interface import IBiblioStylesTool


class APABibrefStyle(BibrefStyle):
    """ specific formatter to process input in APA format
    """
    
    meta_type = "APA Bibref Style"
    
    def __init__(self, id = 'APA', title = "APA bibliography reference style"):
        """ initializes only id and title
        """
        self.id = id
        self.title = title
        
    def formatDictionary(self, refValues):
        """ formats a bibref dictionnary
        """
        formatted_entry = ''
        uid = refValues.get('UID')
        
        entry_type = refValues.get('meta_type')
        if not entry_type:
            entry_type = refValues.get('ref_type')+'Reference'
            
        if entry_type in REFERENCE_TYPES:
        
            formatted_entry += '<div style="padding-bottom: 4pt;text-align:justify;text-indent:-2em; padding-left: 2em;">'
            
            # authors
            authors = refValues.get('authors')
            if authors == []:
                pass
            elif len(authors) == 1:
                formatted_entry += '%s ' % self.formatAuthor(authors[0])
            else:
                if len(authors)<7:
                    for author in authors[0:-1]:
                        formatted_entry += '%s, ' % self.formatAuthor(author)
                    formatted_entry += '& %s' % self.formatAuthor(authors[-1])
                else:
                    for author in authors[0:7]:
                        formatted_entry += '%s, ' % self.formatAuthor(author)
                    formatted_entry += 'et al.'
                    
            if refValues.get('editor_flag'):
                if len(authors) == 1:
                    formatted_entry += ', editor'
                else:    
                    formatted_entry += ', editors'
                    
            formatted_entry += ' '
            
            # year
            if entry_type == 'ArticleReference':
                formatted_entry += '(%s' % refValues.get('publication_year')
                if refValues.get('publication_month'):
                    formatted_entry += ', %s' % refValues.get('publication_month')
                formatted_entry += ').'
            else:
                formatted_entry += '(%s).' % refValues.get('publication_year')
                
                # title
            entry_title = refValues.get('title')
            if entry_title and entry_title[-1] not in '.?!': entry_title += '.'
            title_link = refValues.get('title_link')
            if entry_type == 'ArticleReference' or entry_type[:2] == 'In':
                if title_link:
                    entry_title = '<a UID="%s" href="%s">%s</a>' % (uid, title_link, entry_title)
                formatted_entry += ' %s' % entry_title
            else:
                if title_link:
                    formatted_entry += ' <a UID="%s" href="%s"><i>%s</i></a>' % (uid, title_link, entry_title)
                else:
                    formatted_entry += ' <i>%s</i>' % entry_title
                    
                    # 'In' part
            if entry_type[:2] == 'In':
                formatted_entry += ' In <i>%s</i>' % refValues.get('booktitle')
                
                # series
            if refValues.get('series'):
                formatted_entry += ', %s' % refValues.get('series')
                
                # Pages, Edition & Volume
            if refValues.get('volume') or refValues.get('edition') or (refValues.get('pages') and entry_type[:2] == 'In'):
                formatted_entry += ' ('
                if refValues.get('pages'):
                    formatted_entry += 'pp. %s ' % refValues.get('pages')
                if refValues.get('edition'):
                    formatted_entry += '%s ed. ' % refValues.get('volume')
                if refValues.get('volume'):
                    formatted_entry += 'vol. %s ' % refValues.get('volume')
                formatted_entry = formatted_entry.strip() + ')'
            formatted_entry += '.'
            
            # address & publisher
            if refValues.get('address'):
                formatted_entry += ' %s' % refValues.get('address')
                if  refValues.get('publisher'):
                    formatted_entry += ':'
                else:
                    formatted_entry += '.'
            if refValues.get('publisher'):
                formatted_entry += ' %s.' % refValues.get('publisher')
                
                # Journal & pages
            if refValues.get('journal'):
                formatted_entry += ' <i>%s</i>' % refValues.get('journal')
                if refValues.get('pages'):
                    formatted_entry += ' %s.' % refValues.get('pages')
                    
            if formatted_entry.strip()[-1] == ',':
                formatted_entry = formatted_entry.strip()[:-1] + '.'
                
                # url
            url = refValues.get('publication_url')
            if url:
                formatted_entry += ' Retrieved from <a href="%s">%s</a>.' %(url, url)
                
            formatted_entry += '</div>'
            
        return formatted_entry
        
        # due to a misprint in former style versions this has to be here for compatibility
    formatDictionnary = formatDictionary
    
    def formatAuthor(self, author):
        """ formats a single author for this format """
        bs_tool = getToolByName(self, 'portal_bibliostyles')
        last = author.get('lastname')
        initials = bs_tool.getGivenNameInitials(author, dot_after_initials=True, keep_hyphens=True)
        if initials and last:
            result = '%s, %s' % (last, initials)
        elif last:
            result = '%s' % last
        else:
            return ''
            
        url = author.get('homepage')
        if url:
            result = '<a href="%s">%s</a>' %(url, result)
        return result
        
        # Class instanciation
InitializeClass(APABibrefStyle)


def manage_addAPABibrefStyle(self, REQUEST=None):
    """ """
    try:
        self._setObject('APA', APABibrefStyle())
    except:
        return MessageDialog(
            title='BiblioList tool warning message',
            message='The bibref style you attempted to add already exists.',
            action='manage_main')
    return self.manage_main(self, REQUEST)
