# -*- coding: iso-8859-1 -*-

"""
ATSchemaEditorNG
License: see LICENSE.txt
$Id: ParentManagedSchema.py 52127 2007-10-21 11:42:29Z naro $
"""

from Globals import InitializeClass
from AccessControl import ClassSecurityInfo
from Acquisition import ImplicitAcquisitionWrapper
from Products.CMFCore.permissions import View
from Products.Archetypes.Schema import ManagedSchema
from interfaces import IParentManagedSchema, ISchemaEditor
from Products.CMFCore.utils import getToolByName
from Products.Archetypes.utils import shasattr
from Products.Archetypes.utils import capitalize
from Products.Archetypes.utils import DisplayList
from util import create_signature
from Products.ATSchemaEditorNG.SchemaEditor import SchemaEditorError
import config
from util import LOG, INFO

class ManagedSchemaBase:
    """ mix-in class for AT content-types whose schema is managed by
        the parent container and retrieved through acquisition.
    """

    __implements__ = (IParentManagedSchema,)

    security = ClassSecurityInfo()  
    isToolManaged = False

    def _wrap_schema(self, schema):
        return ImplicitAcquisitionWrapper(schema , self)

    def updateSchemaFromEditor(self):
        self.__atse_should_update__ = True

    security.declareProtected(View, 'Schema')
    def Schema(self, schema_id=None):
        """ Retrieve schema from parent object. The client class should
            override the method as Schema(self) and then call his method
            of the baseclass with the corresponding schema_id.
        """

        # Schema() seems to be called during the construction phase when there is
        # no acquisition context. So we return the default schema itself.
        # ATT: mind the difference btn 'hasattr' and 'shasattr' (latter strips acq)
        if not hasattr(self, 'aq_parent') or self.id.find('atse-ptypedummy')>=0:
            if shasattr(self, 'schema'):
                return self._wrap_schema(self.schema)
            else:
                raise ValueError('Instance has no "schema" attribute defined')
            
        provider = self.lookup_provider()
        if not ISchemaEditor.isImplementedBy(provider):
            raise SchemaEditorError, 'Provider -- %s -- can not be recognized as Archetypes Schema Editor! Perhaps you are using ParentManagedSchema derived objects with Tool. A tool can only manage ToolManagedSchema (ParentOrToolManagedSchema) derived objects.' % str(provider).replace('<', '').replace('>','')

        if not self.lookup_provider().atse_isSchemaRegistered(self.portal_type):
            return self._wrap_schema(self.schema)

        # If we're called by the generated methods we can not rely on
        # the id and need to check for portal_type
        if not schema_id:
            schema_id = self.portal_type
            
        # Otherwise get the schema from the parent through
        # acquisition and assign it to a volatile attribute for performance
        # reasons
        self._v_schema = getattr(self, '_v_schema', None)
        if self._v_schema is None:

            # looking for changes in the schema held by the object
            LOG('ATSchemaEditorNG', INFO, 'Looking up changes via lookupChanges() - _v_schema is None')
            self._v_schema = self._lookupChanges(schema_id)

            if not shasattr(self, '_md'):
                self.initializeArchetype()

            for field in self._v_schema.fields():

                ##########################################################
                # Fake accessor and mutator methods
                ##########################################################

                # XXX currently only honoring explicitly specified mutators
                #     or accessors if the method exists on the object.  if
                #     the methods are autogenerated the specified names will
                #     not be honored.

                name = field.getName()

                def atse_get_method(self=self, name=name, *args, **kw):
                    return self.getField(name).get(self, **kw)

                def atse_getRaw_method(self=self, name=name, *args, **kw):
                    return self.getField(name).getRaw(self, **kw)

                # workaround for buggy widget/keyword AT template that
                # uses field.accessor as catalog index name *grrrr*
                # XXX find another way to fix that
                if name != 'subject':
                    accessor_name = getattr(field, 'accessor', None)
                    if accessor_name and shasattr(self, accessor_name):
                        pass # the accessor exists, we're cool
                    else:
                        v_name = '_v_%s_accessor' % name
                        accessor_name = v_name
                        setattr(self, v_name, atse_get_method)
                    field.accessor = accessor_name
                    
                    # the edit accessor and the regular accessor can be the
                    # same in most cases, but not for ReferenceFields
                    # XXX any other cases?
                    
                    # test for special cases where there is no
                    # getter with conformant naming scheme
                    con_name = capitalize(field.getName())
                    dogenerate = not shasattr(self, 'get%s' % con_name)

                    edit_accessor_name = getattr(field, 'edit_accessor', None)
                    if edit_accessor_name and shasattr(self, edit_accessor_name) and not dogenerate:
                        pass # the edit_accessor_name exists, we're cool
                    else:
                        edit_v_name = '_v_%s_edit_accessor' % name
                        edit_accessor_name = edit_v_name
                        field_type = str(field.type).upper()
                        if field_type == 'REFERENCE':
                            setattr(self, edit_accessor_name, atse_getRaw_method)
                        else:
                            setattr(self, edit_accessor_name, atse_get_method)
                    field.edit_accessor = edit_accessor_name

                def atse_set_method(value, self=self, name=name, *args, **kw):
                    if name != 'id':
                         self.getField(name).set(self, value, **kw)
                        
                    # saving id directly (avoiding unicode problems)
                    else: self.setId(value)

                mutator_name = getattr(field, 'mutator', None)
                if mutator_name and shasattr(self, mutator_name):
                    pass # the mutator exists, we're cool
                else:
                    v_name = '_v_%s_mutator' % name
                    mutator_name = v_name
                    setattr(self, v_name, atse_set_method)
                field.mutator = mutator_name

                # Check if we need to update our own properties
                try:
                    value = field.get(self)
                except:
                    field.set(self, field.default)

        return self._wrap_schema(self._v_schema)

    def _lookupChanges(self, atse_schema_id):
        """ Checks if schema has changed """
        provider = self.lookup_provider()
       
        # only refresh schema if editor told us
        # to do so. If not refreshing from editor we
        # take schema from where we saved it
        if getattr(self, '__atse_should_update__', None) != True:
            LOG('ATSchemaEditorNG', INFO, 'Editor did not trigger update')
            if not shasattr(self, '_schema_protected_from_fs_changes'):
                LOG('ATSchemaEditorNG', INFO, 'No saved version of schema found - taking from FS')
                self._schema_protected_from_fs_changes = self.schema.copy()

            if not config.ALWAYS_SYNC_SCHEMA_FROM_DISC == True:
                return self._schema_protected_from_fs_changes
       
        if config.ALWAYS_SYNC_SCHEMA_FROM_DISC == True:
            
            # looking if schema has changed
            atse_schema = provider.atse_getSchemaById(atse_schema_id)
            object_schema = self.schema

            if create_signature(atse_schema) != create_signature(object_schema):
                LOG('ATSchemaEditorNG', INFO, 'Schema <%s> changed - refreshing from disk' % atse_schema_id)
                provider.atse_reRegisterSchema(atse_schema_id, object_schema)

        self.__atse_should_update__ = False
        LOG('ATSchemaEditorNG', INFO, 'Updating schema from editor for %s' % self.getId())

        # keeping schemata in sync
        self._schema_protected_from_fs_changes = provider.atse_getSchemaById(atse_schema_id).copy()
        self.schema = self._schema_protected_from_fs_changes.copy()
        return self._schema_protected_from_fs_changes

    def vocabularyProxy(self, **kwargs):
        """ """ 
        instance = kwargs.get('content_instance')
        field = kwargs.get('field')
        provider = self.lookup_provider()
        name = provider.atse_getVocabularyScriptName(instance.portal_type, field)
        
        method = getattr(instance, name, None)
        if method and callable(method):
            kw = {'content_instance' : instance,
                  'field' : field}
            value = method(**kw)
            return value
        else:
            return DisaplyList(())


class ParentManagedSchema(ManagedSchemaBase):
    """ base class for content-types whose schema is managed by the parent folder """

    isToolManaged = False

    def lookup_provider(self):
        """ return schema provider instance """
        parent = self.aq_parent

        # test if parent is a temp folder
        if parent.__class__.__name__ == 'TempFolder':

            # see CMFPlone.FactoryTool
            obj = parent.__getitem__(self.id)

            # get the parent folder from intended_parent_folder that will provide schema definitions
            return obj.aq_parent

        # we need to include support for portal_factory
        if parent.__class__.__name__ == 'FactoryTool':
            return parent.aq_parent

        return parent

InitializeClass(ParentManagedSchema)

class AcquisitionManagedSchema(ManagedSchemaBase):
    """ base class for content-types whose schema is managed by a
    provider somewhere up the acquisition tree """
   
    isToolManaged = False
   
    def lookup_provider(self):
        """ return schema provider instance """
        app = self.getPhysicalRoot()
        
        def get_aq_provider(obj):
            parent = obj.aq_parent
            if parent is app:
                raise SchemaEditorError(self.translate('atse_no_provider',
                                                       default="No Schema provider found"))
            if ISchemaEditor.isImplementedBy(parent):
                return parent
            else:
                return get_aq_provider(parent)

        return get_aq_provider(self)

InitializeClass(AcquisitionManagedSchema)
    
class ToolManagedSchema(ManagedSchemaBase):
    """ base class for content-types whose schema is managed by a global tool"""

    isToolManaged = True

    def lookup_provider(self):
        """ return schema provider instance """
        
        tool = getToolByName(self, config.TOOL_NAME, None)
        if not tool:
            raise LookupError('Tool %s for managing schema not found!' % TOOL_NAME)
        return tool

InitializeClass(ToolManagedSchema)

class ParentOrToolManagedSchema(ManagedSchemaBase):
    """ Schema provider retrieved from ATSE in context or tool.
    If there is no SchemaEditor provider found in aq path
    then the lookup is redirected to the tool. """

    isToolManaged = True

    def lookup_provider(self):
        """ First searches for parent provider then checks tool """

        provider = None
        try:
            provider = self.aq_parent
            if not ISchemaEditor.isImplementedBy(provider):
                if provider.__class__.__name__ == 'FactoryTool':
                    return provider.aq_parent
                raise SchemaEditorError, ''

        # We redirect lookup per default to the tool
        except SchemaEditorError:
            tool = getToolByName(self, config.TOOL_NAME, None)
            if not tool:
                raise LookupError('Tool %s for managing schema not found!' % TOOL_NAME)
            provider = tool

        return provider

InitializeClass(ParentOrToolManagedSchema)

