# PyAlgoTrade
# 
# Copyright 2011 Gabriel Martin Becedillas Ruiz
# 
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
# 
#   http://www.apache.org/licenses/LICENSE-2.0
# 
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""
.. moduleauthor:: Gabriel Martin Becedillas Ruiz <gabriel.becedillas@gmail.com>
"""

from pyalgotrade.utils import dt
from pyalgotrade.utils import csvutils
from pyalgotrade.barfeed import membf
from pyalgotrade import dataseries
from pyalgotrade import bar

import datetime
import pytz

# Interface for csv row parsers.
class RowParser:
	def parseBar(self, csvRowDict):
		raise Exception("Not implemented")

	def getFieldNames(self):
		raise Exception("Not implemented")

	def getDelimiter(self):
		raise Exception("Not implemented")

# Interface for bar filters.
class BarFilter:
	def includeBar(self, bar_):
		raise Exception("Not implemented")

class DateRangeFilter(BarFilter):
	def __init__(self, fromDate = None, toDate = None):
		self.__fromDate = fromDate
		self.__toDate = toDate

	def includeBar(self, bar_):
		if self.__toDate and bar_.getDateTime() > self.__toDate:
			return False
		if self.__fromDate and bar_.getDateTime() < self.__fromDate:
			return False
		return True

# US Equities Regular Trading Hours filter
# Monday ~ Friday
# 9:30 ~ 16 (GMT-5)
class USEquitiesRTH(DateRangeFilter):
	timezone = pytz.timezone("US/Eastern")

	def __init__(self, fromDate = None, toDate = None):
		DateRangeFilter.__init__(self, fromDate, toDate)

		self.__fromTime = datetime.time(9, 30, 0)
		self.__toTime = datetime.time(16, 0, 0)

	def includeBar(self, bar_):
		ret = DateRangeFilter.includeBar(self, bar_)
		if ret:
			# Check day of week
			barDay = bar_.getDateTime().weekday()
			if barDay > 4:
				return False

			# Check time
			barTime = dt.localize(bar_.getDateTime(), USEquitiesRTH.timezone).time()
			if barTime < self.__fromTime:
				return False
			if barTime > self.__toTime:
				return False
		return ret

class BarFeed(membf.Feed):
	"""A CSV file based :class:`pyalgotrade.barfeed.BarFeed`.

	.. note::
		This is a base class and should not be used directly.
	"""

	def __init__(self, frequency, maxLen=dataseries.DEFAULT_MAX_LEN):
		membf.Feed.__init__(self, frequency, maxLen)
		self.__barFilter = None
		self.__dailyTime = datetime.time(23, 59, 59)

	def getDailyBarTime(self):
		"""Returns the time to set to daily bars when that information is not present in CSV files. Defaults to 23:59:59.

		:rtype: datetime.time.
		"""

		return self.__dailyTime

	def setDailyBarTime(self, time):
		"""Sets the time to set to daily bars when that information is not present in CSV files.

		:param time: The time to set.
		:type time: datetime.time.
		"""

		self.__dailyTime = time

	def setBarFilter(self, barFilter):
		self.__barFilter = barFilter

	def addBarsFromCSV(self, instrument, path, rowParser):
		# Load the csv file
		loadedBars = []
		reader = csvutils.FastDictReader(open(path, "r"), fieldnames=rowParser.getFieldNames(), delimiter=rowParser.getDelimiter())
		for row in reader:
			bar_ = rowParser.parseBar(row)
			if bar_ != None and (self.__barFilter is None or self.__barFilter.includeBar(bar_)):
				loadedBars.append(bar_)

		self.addBarsFromSequence(instrument, loadedBars)

class GenericRowParser(RowParser):
	def __init__(self, timezone):
		self.__timezone = timezone
		self.__haveAdjClose = False

	def barsHaveAdjClose(self):
		return self.__haveAdjClose

	def __parseDate(self, dateString):
		datetime_format = "%Y-%m-%d %H:%M:%S"
		ret = datetime.datetime.strptime(dateString, datetime_format)
		# Localize the datetime if a timezone was given.
		if self.__timezone:
			ret = dt.localize(ret, self.__timezone)
		return ret

	def getFieldNames(self):
		# It is expected for the first row to have the field names.
		return None

	def getDelimiter(self):
		return ","

	def parseBar(self, csvRowDict):
		dateTime = self.__parseDate(csvRowDict["Date Time"])
		close = float(csvRowDict["Close"])
		open_ = float(csvRowDict["Open"])
		high = float(csvRowDict["High"])
		low = float(csvRowDict["Low"])
		volume = float(csvRowDict["Volume"])

		adjClose = csvRowDict["Adj Close"]
		if len(adjClose) > 0:
			adjClose = float(adjClose)
			self.__haveAdjClose = True
		else:
			adjClose = None

		return bar.BasicBar(dateTime, open_, high, low, close, volume, adjClose)

class GenericBarFeed(BarFeed):
	"""A BarFeed that loads bars from CSV files that have the following format:
	::

		Date Time,Open,High,Low,Close,Volume,Adj Close
		2013-01-01 00:59:59,13.51001,13.56,13.51,13.56,273.88014126,13.51001

	:param frequency: The frequency of the bars.
	:param timezone: The default timezone to use to localize bars. Check :mod:`pyalgotrade.marketsession`.
	:type timezone: A pytz timezone.
	:param maxLen: The maximum number of values that the :class:`pyalgotrade.dataseries.bards.BarDataSeries` will hold.
		If not None, it must be greater than 0.
		Once a bounded length is full, when new items are added, a corresponding number of items are discarded from the opposite end.
	:type maxLen: int.

	.. note::
		* It is ok if the **Adj Close** column is empty.
		* Valid **frequency** parameter values are:

		 * pyalgotrade.barfeed.Frequency.MINUTE 
		 * pyalgotrade.barfeed.Frequency.HOUR
		 * pyalgotrade.barfeed.Frequency.DAY
	"""

	def __init__(self, frequency, timezone = None, maxLen=dataseries.DEFAULT_MAX_LEN):
		BarFeed.__init__(self, frequency, maxLen)
		self.__timezone = timezone
		self.__haveAdjClose = False

	def barsHaveAdjClose(self):
		return self.__haveAdjClose

	def addBarsFromCSV(self, instrument, path, timezone = None):
		"""Loads bars for a given instrument from a CSV formatted file.
		The instrument gets registered in the bar feed.
		
		:param instrument: Instrument identifier.
		:type instrument: string.
		:param path: The path to the CSV file.
		:type path: string.
		:param timezone: The timezone to use to localize bars. Check :mod:`pyalgotrade.marketsession`.
		:type timezone: A pytz timezone.
		"""

		if timezone is None:
			timezone = self.__timezone
		rowParser = GenericRowParser(timezone)
		BarFeed.addBarsFromCSV(self, instrument, path, rowParser)

		if rowParser.barsHaveAdjClose():
			self.__haveAdjClose = True
		elif self.__haveAdjClose == True:
			raise Exception("Previous bars had adjusted close and these ones doesn't have.")

