# Second Music Player example program from Python Fit examples
# copyright 2005, John H. Roth Jr.
# Released under the terms of the GNU General Public License, version 2 or above

# This contains slightly modified versions of several support classes
# from MusicPlayer.py

import copy
import re
import sys
import types
from fit.Parse import Parse
from fit.Utilities import em

class EnglishListTypeAdapter(object):
    parseRE = re.compile(r",|\sor\s|\sand\s")
    def parse(self, aString):
        aList = self.parseRE.split(aString)
        result = [x.strip() for x in aList
                  if len(x.strip()) != 0]
        return result

    def toString(self, aList):
        aList = [x.strip() for x in aList]
        aString = ", ".join(aList)
        lastComma = aString.rfind(",")
        if lastComma == -1:
            return aString
        begin = aString[:lastComma]
        end = aString[lastComma+1:]
        result = "%s and %s" % (begin.strip(), end.strip())
        return result

class Song(object):
    _typeDict = {}
    _baseTypeDict = {}
    
    def __init__(self, names, fields):
        Song._typeDict = self._baseTypeDict.copy()
        nameList = names.split("\t")
        fieldList = fields.split("\t")
        convertSpec = "SSSSIIIIIS"
        i = 0
        while i < 10:
            name = nameList[i]
            value = fieldList[i]
            conv = convertSpec[i]
            name = name.replace(" ", "")
            if name == "Time":
                name = "Seconds"
            if conv == "I":
                self._typeDict[name] = "Int"
                value = int(value)
            else:
                self._typeDict[name] = "String"
            setattr(self, name, value)
            i += 1

    def getTrack(self):
        return "%s of %s" % (self.TrackNumber, self.TrackCount)
    _baseTypeDict["Track"] = "String"
    Track = property(getTrack)

    def getTime(self):
        return TimeInSeconds(self.Seconds).inMinutesAndSeconds()
    _baseTypeDict["Time"] = "String"
    Time = property(getTime)

    def fields(self, fieldList):
        resultList = []
        for fieldName in fieldList:
            fieldName = fieldName.title()
            fieldValue = getattr(self, fieldName, "unknown field name")
            resultList.append("%s: %s" % (fieldName, fieldValue))
        return resultList

    def getField(self, fieldName):
        return getattr(self, fieldName.title(), "unknown field name")    

class TimeInSeconds(object):
    def __init__(self, operand):
        seconds = self._extractSeconds(operand, "")
        self._typeErrorIfNotImplemented(seconds)
        self.seconds = seconds

    def _extractSeconds(self, other, side):
        if isinstance(other, TimeInSeconds):
            seconds = other._seconds
        elif isinstance(other, int):
            seconds = other
        elif isinstance(other, types.StringTypes):
            parts = [int(x) for x in other.split(":")]
            if len(parts) == 2:
                seconds = parts[0] * 60 + parts[1]
            elif len(parts) == 1:
                seconds = int(seconds)
            else:
                raise ValueError("Value must contain at most one ':'")
        else:
            seconds = NotImplemented
        return seconds

    def _typeErrorIfNotImplemented(self, result):
        if result is NotImplemented:
            raise TypeError("Operand must be integer, string or TimeInSeconds")

##    def inSeconds(self):
##        return "%s" % self._seconds

    def __str__(self):
        return "%d:%02d" % divmod(self._seconds, 60)

##    inMinutesAndSeconds = __str__
##
##    def addSeconds(self, anInt):
##        self._seconds += anInt
##        return self._seconds
##
##    def subtractSeconds(self, anInt):
##        result = self._seconds - anInt
##        self._seconds = max(0, result)
##        return result

    def __add__(self, other):
        seconds = self._extractSeconds(other, "Right ")
        if seconds is NotImplemented:
            return NotImplemented
        return TimeInSeconds(self._seconds + seconds)

    def __radd__(self, left):
        seconds = self._extractSeconds(left, "Left ")
        if seconds is NotImplemented:
            return NotImplemented
        return TimeInSeconds(self._seconds + seconds)

    def __sub__(self, right):
        seconds = self._extractSeconds(right, "Right ")
        if seconds is NotImplemented:
            return NotImplemented
        result = self._seconds - seconds
        self._valueErrorIfNegative(result)
        return TimeInSeconds(result)

    def _valueErrorIfNegative(self, result):
        if result < 0:
            raise ValueError("Result is not allowed to be negative")

    def __rsub__(self, left):
        seconds = self._extractSeconds(left, "Left ")
        if seconds is NotImplemented:
            return NotImplemented
        result = seconds - self._seconds
        self._valueErrorIfNegative(result)
        return TimeInSeconds(result)

    def __eq__(self, other):
        seconds = self._extractSeconds(other)
        if seconds is NotImplemented:
            return NotImplemented
        return self._seconds == seconds

    def __ne__(self, other):
        seconds = self._extractSeconds(other)
        if seconds is NotImplemented:
            return NotImplemented
        return self._seconds != seconds

    def __lt__(self, other):
        seconds = self._extractSeconds(other)
        if seconds is NotImplemented:
            return NotImplemented
        return self._seconds < seconds

    def __gt__(self, other):
        seconds = self._extractSeconds(other)
        if seconds is NotImplemented:
            return NotImplemented
        return self._seconds > seconds

    def __le__(self, other):
        seconds = self._extractSeconds(other)
        if seconds is NotImplemented:
            return NotImplemented
        return self._seconds <= seconds

    def __ge__(self, other):
        seconds = self._extractSeconds(other)
        if seconds is NotImplemented:
            return NotImplemented
        return self._seconds >= seconds
