# RunnerImplementation.
# copyright 2005, John H. Roth Jr.
# released under the GNU General Public License, version 2 or later
# Independently written work-alike for HTMLRunner and FolderRunner from
# Rick Mugridge's FitLibrary

"""
usage for FileRunner
python FileRunner [options] in-file out-file [rename-file]
\t+ turns option on, - turns option off
\ta <module> Application Configuration module
\tb <option> Option to pass to Application Configuration Module
\tc - CSS formatting mode
\td <encoding> - default if input encoding cannot be determined
\te - force strict compliance with FIT 1.1 specification
\ti <encoding> - input encoding regardless of <meta> tag
\to <encoding> - output encoding
\tq <options> - print results to console
\tr is recursive mode: process all subdirectories
\ts - add SetUp and TearDown files
\tv - option processing results
\tx - write xml statistics file - must be in multi-file mode.
\tin-file input file, directory or file list
\tout-file. Output file or directory
\trenames-file is optional
"""


try:
    False
except:
    False = 0
    True = 1

try:
    bool(0)
except:
    def bool(predicate):
        if predicate:
            return True
        else:
            return False

import copy
import os, os.path
import re
import stat
import sys
import time

from fit.Counts import Counts
from fit import FitGlobal
from fit.Fixture import Fixture
from fit.Options import Options
from fit.Parse import Parse
from fit.Utilities import em
from fit import Variations

# This is an adapter class; there is one instance bound to the module
# level identifier conMsg. It's replaced by the testing framework to
# capture console messages.

class ConsoleMessageHandler(object):
    def __init__(self):
        self.verbose = False

    def setVerbose(self, verbose):
        self.verbose = verbose
   
    def err(self, msg):
        if msg[-1] != "\n":
            msg += "\n"
        sys.stderr.write(msg)

    def notes(self, msg):
        if self.verbose:
            self.err(msg)

    def stats(self, msg):
        self.err(msg)

    def debug(self, msg):
        if msg[-1] != "\n":
            msg += "\n"
        sys.stderr.write(msg)

conMsg = ConsoleMessageHandler()

class FileSystemAdapter(object):
    def abspath(self, path): return os.path.abspath(path)
    def basename(self, path): return os.path.basename(path)
    def dirname(self, path): return os.path.dirname(path)
    def exists(self, path): return os.path.exists(path)
    def isdir(self, path): return os.path.isdir(path)
    def isfile(self, path): return os.path.isfile(path)
    def join(self, *paths): return os.path.join(*paths)
    def listdir(self, path): return os.listdir(path)
    def mkdir(self, path): return os.mkdir(path)
    def makedirs(self, path): return os.makedirs(path)
    def open(self, path, mode): return open(path, mode)
    def split(self, path): return os.path.split(path)
    def splitext(self, path): return os.path.splitext(path)
#    def stat(self, path): return os.stat(path)
# following returns wkd mon dd hh:mm:ss yyyy
    def timeUpdated(self, path): return time.ctime(os.stat(path).st_mtime)

fsa = FileSystemAdapter()

class AbstractFileHandler(object):
    _fileName = ""

    def setFileName(self, fileName):
        self._fileName = fileName
    def getFileName(self):
        return self._fileName
    fileName = property(getFileName, setFileName)

    def getAbsPath(self):
        return fsa.abspath(self._fileName)

    def setEncodingDefaults(self, default, override):
        self.defaultEncoding = default
        self.inputEncodingOverride = override
        
    def verifyFileName(self):
        pass

class InputFileHandler(AbstractFileHandler):
    _useWikiTag = False
    inputEncodingOverride = ""
    defaultEncoding = "Windows-1252"
    
    def __init__(self):
        self.encoding = ""
        self.encodingType = ""
        
    def verifyFileName(self):
        return fsa.isfile(self._fileName)

    def getEncoding(self):
        return self.encoding, self.encodingType

    def getParseTree(self):
        file = open(self._fileName, "r")
        text = file.read()
        file.close()
        decodedText = self._findEncoding(text)
        return self._parseText(decodedText)
    
    def _parseText(self, text):
        tag = ("table", "tr", "td")
        if text.find("<wiki>") != -1:
            tag = ("wiki", "table", "tr", "td")
        return Parse(text, tag)

    def getUpdateTime(self):
        return fsa.timeUpdated(self._fileName)

    def _findEncoding(self, text):
        encoding = ""
        encodingType = ""
        if text[:3] == "\xef\xbb\xbf":
            encoding, encodingType = "utf-8", "BOM"
        elif text[:2] in ("\xfe\xff", "\x00<"):
            encoding, encodingType = "utf-16be", "BOM"
        elif text[:2] in ("\xff\xfe", "<\x00"):
            encoding, encodingType = "utf-16le", "BOM"
        elif text[:4] in ("\xff\xfe\x00\x00", "<\x00\x00\x00"):
            encoding, encodingType = "utf-32le", "BOM"
        elif text[:4] in ("\x00\x00\xfe\xff", "\x00\x00\x00<"):
            encoding, encodingType = "utf-32ge", "BOM"
        if not encoding and self.inputEncodingOverride:
            encoding, encodingType = self.inputEncodingOverride, "Forced"
        if not encoding:
            encoding, encodingType = self._encodingFromMetaTag(text)
        if not encoding and self.defaultEncoding:
            encoding, encodingType = self.defaultEncoding, "Options" # ???
        if not encoding:
            encoding, encodingType = "Windows-1252", "default"
        self.encoding = encoding
        self.encodingType = encodingType
        decodedText = text.decode(encoding) # needs a try/except block.
        return decodedText

    metaRE = re.compile(r"<meta[ ]+http-equiv=[\"\']content-type[\"\']"
                        "[ ]+content=[\"\']text/html;[ ]+charset=(.+?)"
                        "[\"\'][ ]*/?>")
    def _encodingFromMetaTag(self, text):
        lc = text.lower()
        match = self.metaRE.search(lc)
        if match is None:
            return "", ""
        result = match.group(1)
        return result, "metaTag"

class SetUpTearDownStack(object):
    def __init__(self):
        self.theStack = [NullSetupFileHandler()]

    def push(self, handler):
        handler.setUpStack = self.theStack
        self.theStack.append(handler)

    def pop(self):
        self.theStack[-1].setUpStack = None
        del self.theStack[-1]
        return self.theStack[-1]
        
class SetUpTearDownFileHandler(InputFileHandler):
    def __init__(self):
        self.setUpStack = None # !!! dependency inserted by parent
        self._firstParseTree = None
        self._lastParseTree = None
        super(SetUpTearDownFileHandler, self).__init__()
        
    def verifyFileName(self):
        head, tail = fsa.split(self._fileName)
        if not fsa.isdir(head):
            return
        self._dirPath = head
        self._firstParseTree = self._fetchFile("SetUp")
        self._firstParseTree = self._attachTrees(
            copy.deepcopy(self.setUpStack[-2]._firstParseTree),
            self._firstParseTree, retain="first")
        self._lastParseTree = self._fetchFile("TearDown")
        self._lastParseTree = self._attachTrees(self._lastParseTree,
            copy.deepcopy(self.setUpStack[-2]._lastParseTree),
            retain="last")
        return

    def _fetchFile(self, fileName):
        text = self._fetchOneFile(fileName, ".htm")
        text = text or self._fetchOneFile(fileName, ".html")
        if text == "":
            return None
        return self._parseText(text)

    def _fetchOneFile(self, fileName, suffix):
        fileName = fsa.join(self._dirPath, fileName + suffix)
        if not fsa.isfile(fileName):
            return ""
        inFile = open(fileName, "r")
        text = inFile.read()
        inFile.close()
        return text

    def getParseTree(self, mainTree):
        resultTree = mainTree
        resultTree = self._attachTrees(copy.deepcopy(self._firstParseTree),
                                       resultTree, retain="last")
        resultTree = self._attachTrees(resultTree,
                        copy.deepcopy(self._lastParseTree),
                        retain = "first")
        return resultTree

    def _attachTrees(self, firstTree, lastTree, retain="last"):
        if firstTree is None or lastTree is None:
            return firstTree or lastTree
        firstTree1stNode = firstTree
        firstTreeLastNode = firstTree.last()
        head1, body1 = self._splitHeaderFromBody(firstTree.leader)
        head2, body2 = self._splitHeaderFromBody(lastTree.leader)
        if retain == "last" and head2 != "":
            firstTree.leader = self._makeNewLeader(head2, body1)
        lastTree.leader = body2
        trailer1 = self._removeEndBody(firstTree.last().trailer)
        lastTree.leader = self._makeNewLeader(trailer1, lastTree.leader)
        firstTreeLastNode.more = lastTree
        firstTreeLastNode.trailer = ""
        return firstTree

    def _splitHeaderFromBody(self, leader):
        lcLeader = leader.lower()
        bodyOffset = lcLeader.find("<body")
        if bodyOffset == -1:
            return "", leader
        endBodyTag = lcLeader.find(">", bodyOffset) + 1
        return leader[:endBodyTag], leader[endBodyTag:]

    def _makeNewLeader(self, head, body):
        if head == "":
            return body
        if body == "":
            return head
        return head + "\n<br>" + body

    def _removeEndBody(self, trailer):
        lcTrailer = trailer.lower()
        endBodyTag = lcTrailer.find("</body")
        if endBodyTag == -1:
            endBodyTag = lcTrailer.find("</html")
        if endBodyTag == -1:
            return trailer
        return trailer[:endBodyTag]

class NullSetupFileHandler(SetUpTearDownFileHandler):
    def verifyFileName(self):
        return

class AcceptanceTestOut(AbstractFileHandler):
    textOut = ""
    encoding = ""
    encodingType = ""

    def verifyFileName(self):
        head, tail = fsa.split(self._fileName)
        if not head:
            return False
        return fsa.isdir(head)

    def setEncoding(self, encoding):
        self.encoding, self.encodingType = encoding

    def acceptParseTree(self, parseTree):
        self._addCSSStuffToHeader(parseTree) # moved from Fixture
        textOut = parseTree.toString()
        fixedText = self._fixMetaTag(textOut, self.encoding,
                                     self.encodingType)
        self.text = fixedText.encode(self.encoding)

    def write(self):
        fileObj = open(self.fileName, "w")
        fileObj.write(self.text)
        fileObj.close()

    fitLink = '<link rel="stylesheet" href="FIT.css" type="text/css">\n'

    def _addCSSStuffToHeader(self, table):
        if table.leader is None:
            table.leader = ""
        lc = table.leader.lower()
        if lc.find("fit.css") != -1:
            return
        endHead = lc.find("</head>")
        if endHead == -1:
            table.leader = "%s%s%s%s" % (
                "<html><head>\n<title>FIT Acceptance Test Dummy Header</title>\n",
                self.fitLink,
                "</head><body>",
                table.leader)
            # XXX need to scan to the end of the table chain and add the
            #     correct ending markup: </body></html>
            return
        table.leader = "%s%s%s%s" % (table.leader[:endHead], "\n",
                                     self.fitLink,
                                     table.leader[endHead:])
        return

    metaRE = re.compile(r"<meta[ ]+http-equiv=[\"\']content-type[\"\']"
                        "[ ]+content=[\"\']text/html;[ ]+charset=(.+?)"
                        "[\"\'][ ]*/?>", re.I)
    def _fixMetaTag(self, text, encodingName, encodingType):
        match = self.metaRE.search(text)
        if encodingName in ("utf-16be", "utf-16le"):
            if match is None:
                return text
            else:
                return self._removeMetaEncodingTag(text, match)
        if match == None:
            text = self._addMetaEncodingTag(text, encodingName, None)
            return text
        result = match.groups(1)
        if  encodingName == result:
            return
        text = self._removeMetaEncodingTag(text, match)
        text = self._addMetaEncodingTag(text, encodingName, match)
        return text

    def _removeMetaEncodingTag(self, text, match):
        begin, end = match.span()
        return text[:begin] + text[end:]

    headRE = re.compile(r"<head( [^>]*?>|>)", re.I)
    htmlRE = re.compile(r"<html( [^>]*?>|>)", re.I)

    def _addMetaEncodingTag(self, text, encoding, match):
        newTag = ('<meta http-equiv="Content-Type" '
                  'content="text/html; charset=%s">' % encoding)
        if match is not None:
            insertAt = match.start()
        else:
            match = self.headRE.search(text)
            if match is None:
                match = self.htmlRE.search(text)
            if match is None:
                insertAt = 0
            else:
                insertAt = match.end()
        newText = text[:insertAt] + newTag + text[insertAt:]
        return newText

class StatNode(object):
    def __init__(self, fileName, counts, summary):
        self.fileName = fileName
        self.counts = counts
        self.summary = summary

    def outputAsXML(self, parts, indent):
        parts.append("%s<result><Name>%s</Name>\n" % (indent, self.fileName))
        self.outputCounts(parts, self.counts, indent + "   ")
        self.outputSummary(part, self.summary, indent + "   ")
        parts.append("%s</result>\n" % indent)

    def outputCounts(self, parts, counts, indent):
        parts.append("%s<counts>\n" % indent)
        parts.append('%s   <right>%i</right>\n' % (indent, counts.right))
        parts.append('%s   <wrong>%i</wrong\n>' % (indent, counts.wrong))
        parts.append('%s   <ignores>%i</ignores>' % (indent, counts.ignores))
        parts.append('%s   <exceptions>%i</exceptions>' % (indent, counts.exceptions))
        parts.append("%s<counts>\n" % indent)

    def outputSummary(self, parts, summary, indent):
        items = self.summary.items()
        items.sort()
        parts.append("%s<summary>\n" % indent)
        for key, value in items:
            parts.append('%s   <item key="%s">%s</item>\n' %
                         (indent, key, value))
        parts.append('%s</summary>\n' % indent)

class StatDirectory(object):
    def __init__(self, shortName, upLink = None):
        self._fileDict = {}
        self._dirDict = {}
        self._upLink = upLink
        self._dirName = shortName

    def reportStats(self, fileName, counts, summary):
        shortName = fsa.basename(fileName)
        newNode = StatNode(shortName, counts, summary)
        self._fileDict[shortName] = newNode

    def reportDirectory(self, fileName):
        shortName = fsa.basename(fileName)
        newDir = StatDirectory(shortName, self)
        self._dirDict[shortName] = newDir
        return newDir

    def getParent(self):
        return self._upLink

    def outputAsXML(self, parts, indent):
        nextLevelIndent = indent + "   "
        parts.append("%s<Directory>\n%s<Name>%s</Name>\n" %
                     (indent, nextLevelIndent, self._dirName))
        fileList = self._fileDict.items()
        fileList.sort()
        for key, value in fileList:
            value.outputAsXML(parts, nextLevelIndent)
        dirList = self._dirDict.items()
        dirList.sort()
        for key, value in dirList:
            value.outputAsXML(parts, nextLevelIndent)
        parts.append("%s</Directory>\n" % indent)

class StatCollector(object):
    def __init__(self, statFileName):
        head, shortName = fsa.split(statFileName)
        self._treeRoot = StatDirectory(shortName)
        self._statFileName = statFileName
        self._dirName = ""
        self.currentDir = self._treeRoot
        self.outDirName = ""

    def setDirectoryNames(self, dirName, outDirName):
        self._dirName = dirName
        self._outDirName = outDirName
        head, sourceDirectory = fsa.split(self._dirName)
        self._treeRoot._dirName = sourceDirectory

    def reportStats(self, fileName, counts, summary):
        self.currentDir.reportStats(fileName, counts, summary)

    def reportDirectory(self, fileName):
        newDir = self.currentDir.reportDirectory(fileName)
        self.currentDir = newDir

    def endOfDirectory(self):
        self.currentDir = self.currentDir.getParent()

    def writeXMLFile(self):
        counts = Counts()
        parts = ['<?xml version="1.0"?>\n',
                 "<testResults><Name>%s</Name>\n" % self._dirName]
        self._treeRoot.outputAsXML(parts, "   ")
        parts.append("</testResults>\n")
        xmlText = "".join(parts)
        outText = xmlText.encode("UTF-8")
        outPath = fsa.join(self._outDirName, self._statFileName)
        outFile = open(outPath, "w")
        outFile.write(outText)
        outFile.close()

class NullStatCollector(StatCollector):
    def __init__(self): pass
    def setDirectoryNames(self, dirName, outDirName): pass
    def reportStats(self, fileName, counts, summary): pass
    def reportDirectory(self, fileName): pass
    def endOfDirectory(self): pass
    def writeXMLFile(self): pass

class ConsoleTotals(object):
    def __init__(self, testTotals, summaryTotals, file, recursion):
        self.testTotals = testTotals
        self.summaryTotals = summaryTotals
        if file is False:
            self.directoryProcess = "n"
        elif recursion is False:
            self.directoryProcess = 'd'
        else:
            self.directoryProcess = "r"

    def folderTitle(self, directoryName):
        self.directoryName = directoryName
        self.directoryNamePrinted = False
        if (self.directoryProcess == 'r' and
            self.summaryTotals == "f"):
                conMsg.stats("Processing Directory: %s" % directoryName)
                self.directoryNamePrinted = True

    def fileResult(self, fileName, counts):
        if self.testTotals == "n":
            return
        if self.testTotals == "e":
            if not counts.isError():
                return
            if (self.directoryProcess == 'r' and
                self.summaryTotals != 'f' and
                self.directoryNamePrinted is False):
                    conMsg.stats("Processing Directory: %s" % self.directoryName)
                    self.directoryNamePrinted = True
                
        if self.directoryProcess == 'n':
            fileName = ""
        conMsg.stats("%s %s" % (counts.toString(), fsa.basename(fileName)))

    def directoryTotal(self, counts):
        directoryName = self.directoryName
        if self.summaryTotals == 'f' and self.directoryProcess == 'r':
            conMsg.stats("Total this Directory: %s" % counts.toString())

    def finalTotal(self, counts):
        if self.summaryTotals != "n":
            conMsg.stats("Total tests Processed: %s" % counts.toString())

class FileRunner(object):
    InjectedSetUpFileHandler = SetUpTearDownFileHandler

    def __init__(self):
        self._inFileHandler = InputFileHandler()
        self._parseTree = None
        self._acceptanceTestOut = AcceptanceTestOut()
        self.fixture = Fixture()
        self.command = "FileRunner"
        self._runAllFilesInDirectory = False
        self._runListOfFiles = False
        self.statList = []
        self.setUpStack = SetUpTearDownStack()
        self.setUpFileHandler = NullSetupFileHandler()
        FitGlobal.Environment = "Batch"

    def parms(self, optv):
        self.options = Options(optv, self._parmDict)
        opts = self.options
        conMsg.setVerbose(opts.verbose)
        errMsgs = opts.eMsgs
        fileList = opts.posParms
        if not opts.isValid:
            for msg in errMsgs:
                conMsg.err(msg)
            self._usage()
            return False

        if opts.standardMode is True:
            if opts.useCSS is True:
                conMsg.err("Standards mode requested. CSS output suppressed\n")
                opts.useCSS = False

        parmEditOK = True
        
        if 2 <= len(fileList) <= 3:
            self.inDir = fsa.abspath(fileList[0])
            self.outDir = fsa.abspath(fileList[1])
            if fsa.isdir(self.inDir):
                self._runAllFilesInDirectory = True
            elif fsa.isfile(self.inDir):
                self._runAllFilesInDirectory = False
                root, ext = fsa.splitext(self.inDir)
                if ext == ".txt":
                    self._runListOfFiles = True
                else:
                    self._runListOfFiles = False
            else:
                conMsg.err("'%s' does not exist!\n" % self.inDir)
                self._runAllFilesInDirectory = False
                parmEditOK = False
                
            if self._runAllFilesInDirectory:
                if not fsa.isdir(self.outDir):
                    conMsg.err("'%s' is not a directory\n" % self.outDir)
                    parmEditOK = False
            else:
                if fsa.isdir(self.outDir): # directory must exist!
                    self.outFileName = fsa.join(self.outDir,
                                       fsa.basename(self.inDir))
                else:
                    self.outFileName = self.outDir
                    self.outDir = fsa.dirname(self.outDir)
                    if not fsa.isdir(self.outDir):
                        conMsg.err("'%s' does not exist!\n"
                                         % self.outDir)
                        parmEditOK = False
            if len(fileList) > 2:
                self._loadRenameFile(fileList[2])
        else:
            conMsg.err("Wrong number of positional parameters\n")
            parmEditOK = False

        if not self._runAllFilesInDirectory and opts.collectStats:
            conMsg.err("Cannot collect stats on single file\n")
            opts.collectStats = False
            
        self.statCollector = NullStatCollector()
        if opts.collectStats is True:
            self.statCollector = StatCollector("FitStatistics.xml")
            self.statCollector.setDirectoryNames(self.inDir, self.outDir)

        opts.filePrintLevel = opts.quietMode[0]
        opts.folderPrintLevel = opts.quietMode[1]
        if self._runAllFilesInDirectory:
        # do some editing to make sure summary print options are sane.
            if opts.recursive is False:
                if opts.folderPrintLevel == "f":
                    opts.folderPrintLevel = "t"
            else:
                if opts.filePrintLevel == "y":
                    opts.folderPrintLevel = "a"
        else:
            opts.folderPrintLevel = "n"
        conMsg.notes("Summary print level in effect. "
                    "Files: '%s' Folders: '%s'\n" %
                   (opts.filePrintLevel, opts.folderPrintLevel))
        self.consoleTotals = ConsoleTotals(opts.filePrintLevel,
                                           opts.folderPrintLevel,
                                           self._runAllFilesInDirectory,
                                           opts.recursive)

        if parmEditOK:
            return True
        self._usage()
        return False

    _parmDict = {"c": ("bo", "useCSS"),
                 "d": ("eo", "defaultEncoding"),
                 "e": ("bo", "standardMode"),
                 "i": ("eo", "inputEncodingOverride"),
                 "o": ("eo", "outputEncodingForce"),
                 "q": ("qo", "quietMode"),
                 "r": ("bo", "recursive"),
                 "s": ("bo", "doSetUp"),
                 "v": ("bo", "verbose"),
                 "x": ("bo", "collectStats"), # should this be an f?
                 }

    def _usage(self):
        text = __doc__.replace("FileRunner", self.options.runner)
        conMsg.err(text)        

    def _loadRenameFile(self, renameFileName):
        self._renameFileName = renameFileName
        try:
            self.fixture.loadFixtureRenamesFromFile(self._renameFileName)
        except:
            parmEditOK = False
            conMsg.notes("Rename file '%s' failed to load" %
                               self._renameFileName)

    def _errMsg(self, prior, result, message):
        if result:
            return prior and True
        conMsg.err(message)
        return False

    def _createFitCSSFile(self, outDir):
        cssFileName = fsa.join(outDir, "FIT.css")
        try:
            cssFile = open(cssFileName, 'rt')
            cssFile.close()
            return
        except:
            pass
        cssFile = open(cssFileName, "wt")
        cssFile.write(cssFileContent)
        cssFile.close()
        return

    # main entry point.
    def run(self):
        FitGlobal.CommandOptions = self.options
        # XXX remove next phrase when refactoring done
        if self.options.standardMode: 
            Variations.runMode = "Standard"
        else:
            Variations.runMode = "Batch"
        inDir = self.inDir
        outDir = self.outDir
        if self._runAllFilesInDirectory:
            result = self.runTestsInDirectory(inDir, outDir)
            self.consoleTotals.finalTotal(result)
        else:
            outFileName = self.outFileName
            if self.options.useCSS:
                self._createFitCSSFile(outDir)
            self._inFileHandler = InputFileHandler()
            self._inFileHandler.fileName = inDir
            self._acceptanceTestOut = AcceptanceTestOut()
            self._acceptanceTestOut.fileName = outFileName
            self._setUpFileHandler(inDir)
            result = self._doOneFile(self._inFileHandler,
                                     self._acceptanceTestOut)
        return result.wrong + result.exceptions

    def _setUpFileHandler(self, inDir):    
            self.setUpFileHandler = self.InjectedSetUpFileHandler()
            self.setUpStack.push(self.setUpFileHandler)
            self.setUpFileHandler.fileName = fsa.join(inDir,
                                                          "SetUp.htm")
            self.setUpFileHandler.verifyFileName()

    def runTestsInDirectory(self, inDir, outDir):
        if self.options.useCSS:
            self._createFitCSSFile(outDir)
        rollUpPageCounts = Counts()
        thisDirPageCounts = Counts()
        dirList = fsa.listdir(inDir)
        dirList.sort()
        self._setUpFileHandler(inDir)
        listOfFiles = []
        listOfDirectories = []
        for item in dirList:
            inFileName = fsa.join(inDir, item)
            # isfile, isdir and stat don't understand the current directory.
            if fsa.isfile(inFileName):
                listOfFiles.append(item)
            else:
                listOfDirectories.append(item)
        if listOfFiles:
            fileRollUp = self._doFilesInDirectory(inDir, outDir, listOfFiles)
            rollUpPageCounts.tally(fileRollUp)

        if self.options.recursive and len(listOfDirectories):
            for aDir in listOfDirectories:
                newInDir = fsa.join(inDir, aDir)
                newOutDir = fsa.join(outDir, aDir)
                if fsa.isdir(newOutDir):
                    pass
                elif fsa.exists(newOutDir):
                    # XXX warning message goes here!
                    newOutDir = outDir #reuse current output dir.
                else:
                    fsa.mkdir(newOutDir)
                self.statCollector.reportDirectory(aDir) # ??? here ???
                rolledUpCounts = self.runTestsInDirectory(newInDir,
                                                          newOutDir)
                self.statCollector.endOfDirectory()
                rollUpPageCounts.tally(rolledUpCounts)
        self.statCollector.writeXMLFile()
        rollUpPageCounts.tally(thisDirPageCounts)
        self.setUpFileHandler = self.setUpStack.pop()
        return rollUpPageCounts
    
    def _doFilesInDirectory(self, inDir, outDir, listOfFiles):
        self.consoleTotals.folderTitle(fsa.basename(inDir))
        pageCounts = Counts()
        for aFile in listOfFiles:
            if not self._isFileAcceptable(aFile):
                continue
            inFileName = fsa.join(inDir, aFile)
            outFileName = fsa.join(outDir, aFile)
            self._inFileHandler = InputFileHandler()
            self._inFileHandler.fileName = inFileName
            self._acceptanceTestOut = AcceptanceTestOut()
            self._acceptanceTestOut.fileName = outFileName
            self._doOneFile(self._inFileHandler, self._acceptanceTestOut)

            self.statCollector.reportStats(aFile, self.fixture.counts,
                                 self.fixture.summary)
            pageCounts.tallyPageCounts(self.fixture.counts)
        self.consoleTotals.directoryTotal(pageCounts)
        return pageCounts

    def _isFileAcceptable(self, fileName):
        head, ext = fsa.splitext(fileName)
        if ext.lower() not in (".html", ".htm"):
            return False
        if head in ("index", "SetUp", "TearDown"):
            return False
        return True
        
    def _doOneFile(self, inFileHandler, outFileHandler):
        inFileHandler.setEncodingDefaults(
                              self.options.defaultEncoding,
                              self.options.inputEncodingOverride)
        inFileHandler.verifyFileName() # should not fail.
        self._acceptanceTestOut.verifyFileName() # should not fail.
        self.fixture = Fixture()

        self.fixture.summary["input update"] = inFileHandler.getUpdateTime()
        self.fixture.summary["output file"] = outFileHandler.getAbsPath()
        self._parseTree = inFileHandler.getParseTree()
        self._parseTree = self.setUpFileHandler.getParseTree(self._parseTree)
#        self.fixture.doTables(self._parseTree, env=Variations.runMode)
        self.fixture.doTables(self._parseTree)
        inputEncoding = inFileHandler.getEncoding()
        if self.options.outputEncodingForce:
            inputEncoding = (self.options.outputEncodingForce,
                             "OutputOverride")
        outFileHandler.setEncoding(inputEncoding)
        outFileHandler.acceptParseTree(self._parseTree)
        outFileHandler.write()
        self.consoleTotals.fileResult(inFileHandler.fileName,
                                      self.fixture.counts)
        return self.fixture.counts

cssFileContent = """
/**********
 Used in FIT
**********/

.fit_pass {
  background-color: #cfffcf;
  }

.fit_fail {
  background-color: #ffcfcf;
  }

.fit_error {
  background-color: #ffffcf;
  }

.fit_ignore {
  background-color: #efefef;
  }

.fit_stacktrace {
  font-size: 0.7em;
  }

.fit_label {
  font-style: italic;
  color: #c08080;
  }

.fit_grey {
  color: #808080;
  }

.fit_green {
  color: #80c080;
  }
"""