"""
Repository info logic.
"""
import os
import subprocess
from pygitfoo import raw_data_operations

__author__ = 'lmiranda'


class Repository(object):
    """
    Provides info about a repository.
    """

    def __init__(self, path):
        """
        Initiates with info about a repository located in path.

        :param path: The path where the repository is located.
        """

        if not os.path.exists(path):
            raise OSError("Path does not exist")

        if not os.path.isdir(path):
            raise OSError("Path is not a directory")

        path = os.path.abspath(path)

        self.name = os.path.split(path)[1]
        self.path = path
        
    def tag(self):
        """
        List all tags.
        """
        
        raw_tags = _system_command("git tag", self.path)

        return raw_data_operations.process_tags(raw_tags)

    def branch(self):
        """
        Lists local branches.
        """

        raw_branches = _system_command("git branch", self.path)

        return raw_data_operations.process_branches(raw_branches)

    def current_branch(self):
        """
        Gets current branch.
        """

        raw_branches = _system_command("git branch", self.path)

        return raw_data_operations.process_current_branch(raw_branches)

    def ls_remote(self):
        """
        Gets a list of tags in the remote.
        """

        raw_tags = _system_command("git ls-remote", self.path)

        return raw_data_operations.process_ls_remote(raw_tags)


class Git(object):
    """
    Provides info about git software installed in the system.
    """

    @staticmethod
    def version():
        """
        Returns the version of the installed git.
        """

        ret = _system_command("git --version")
        return ret.split(' ')[-1].replace('\n', "")


def _system_command(command, cwd=None):
    """
    Runs a command in the specified directory.

    :param command: The command to be executed.
    :param cwd: The directory were the command should be executed.
    """

    p = subprocess.Popen(command.split(' '), stdout=subprocess.PIPE, cwd=cwd)
    out, err = p.communicate()
    if err:
        raise Exception(str(err))

    return out