# -*- coding: utf-8 -*-
'''
This is the API reference for PyGtkImageView which is a set of Python
bindings for the `GTK+ <http://www.gtk.org>`__ widget GtkImageView.

The latest releases of both these bindings and GtkImageView itself can
always be found at http://trac.bjourne.webfactional.com.

PyGtkImageView contains a simple but full-featured image viewer widget
similar to the image viewer panes in `gThumb
<http://gthumb.sourceforge.net/>`__ or `Eye of GNOME
<http://www.gnome.org/gnome-office/eog.shtml>`__. The main class in
the module is `ImageView`.

Documentation
=============
This API documentation can be found in the ``./docs`` directory. HTML
and PDF documentation is generated by the bash script ``makedocs.sh``.
To build the documentation, you need to have the following tools
installed:

* `epydoc <http://epydoc.sourceforge.net>`__ 3.0 beta (install from
  source)
* latex (the packages tetex-bin and tetex-extra in Ubuntu)
* docutils

Then just run the script::

    $ ./makedocs.sh

The source for the documentation is found in the
``./docs/gtkimageview.py`` file.

Keep in mind that this documentation is **not** built from the library
source. This is because while epydoc does support generating
documentation from extension modules, the result is generally very
poor (GTK+ and PyGTK is partially to blame for that).

Epydoc has no support for extracting documentation for signals, which
is why signals are here documented as methods prefixed with sig_*. For
example, the description for the signal "pixbuf-changed" is found in
the method description `ImageView.sig_pixbuf_changed`.

The examples should make it clear how these signals are supposed to be
used.

For that reason, epydoc cannot automatically discern which GTK+
classes GtkImageView's classes subclass. Instead, see the class
descriptions for inheritance information.

:author: `Björn Lindqvist <bjourne@gmail.com>`__
:requires: Python 2.2+, GtkImageView, PyGTK
:version: 1.0.0
:license: LGPL
:copyright: |copy| 2007 Björn Lindqvist
:todo: Figure out how to best handle the DrawSettings struct.
:todo: Make the epydoc generated HTML documentation a little nicer.
:todo: Fix the PDF generation so that images can be used both in the
    HTML and in the PDF documentation.
:todo: Think out a good way to partially update the pixbuf.
:todo: Slap it up on PyPI.

.. |copy| unicode:: 0xA9 .. copyright sign
'''

#: Enumeration value to use a single color for transparent parts.
TRANSP_COLOR = 1
#: Enumeration value to use a light and dark gray grid for transparent
#: parts.
TRANSP_GRID = 2
#: Enumeration value to use the widgets background color for
#: transparent parts.
TRANSP_BACKGROUND = 3

class IImageTool:
    '''
    IImageTool is an interface that defines how `ImageView` interacts
    with objects that acts as tools. ImageView delegates many of its
    most important tasks (such as drawing) to its tool which carries
    out all the hard work. The PyGtkImageView package comes with two
    tools; `ImageToolDragger` and `ImageToolSelector`, but by
    implementing your own tool it is possible to extend ImageView to
    do stuff its author (thats me) didn't imagine.

    GtkImageView uses `ImageToolDragger` by default, as that tool is
    he most generally useful one. However, it is trivial to make it
    use another tool.

    .. python::

       view = gtkimageview.ImageView()
       tool = gtkimageview.ImageToolSelector(view)
       view.set_tool(tool)

    Using the above code makes the view use the selector tool instead
    of the default dragger tool.
    '''
    def button_press(self, ev_button):
        pass
    def button_release(self, ev_button):
        pass
    def motion_notify(self, ev_motion):
        pass
    def pixbuf_changed(self, reset_fit):
        '''
        This method is called by the view whenever its pixbuf or its
        tool changes. That is, when any of the methods
        `ImageView.set_pixbuf()` or `ImageView.set_tool()` is invoked.
        If the `reset_fit` parameter is ``True``, then the tool is
        free to treat the pixbuf as completely new. See also
        `ImageView.sig_pixbuf_changed`.

        :param reset_fit: whether the view is resetting its fit mode
            or not
        '''

    def paint_image(self, draw_settings, drawable):
        '''
        Called whenever the image view decides that any part of the
        image it shows needs to be redrawn.

        :param draw_settings: a set of draw settings to use for this
            draw
        :param drawable: a ``gtk.gdk.Drawable`` to draw the image data
            on
        '''

    def cursor_at_point(self, x, y):
        '''
        Ask the tool what cursor it wants displayed.

        :param x: the mouse pointers x-coordinate
        :param y: the mouse pointers y-coordinate
        :return: an appropriate ``gtk.gdk.Cursor``
        '''

class ImageToolDragger(IImageTool):
    '''
    ImageToolDragger is the default image tool for `ImageView`. Its
    only feature is that it can drag the image around.
    '''
    def __init__(self, view):
        '''
        Creates a new dragger tool.

        :param view: an `ImageView`
        '''

class ImageToolSelector(IImageTool):
    '''
    ImageToolSelector is a tool for selecting areas of an image. It is
    useful for cropping an image, for example. The tool is an
    implementor of the `IImageTool` inteface which means that it can
    be plugged into an `ImageView` object by using the
    `ImageView.set_tool()` method.

    ImageToolSelector changes the default display of the ImageView. It
    darkens down the unselected region of the image which provides a
    nice effect and makes it clearer what part of the image that is
    currently selected. Unfortunately, this effect is somewhat
    incompatible with how `ImageNav` behaves and that widget will show
    the image without darkening it.

    The tool also changes the default behaviour of the mouse. When an
    ImageToolSelector is set on an ImageView, mouse presses do not
    "grab" the image and you cannot scroll by dragging. Instead mouse
    presses and dragging is used to resize and move the selection
    rectangle. When the mouse drags the selection rectangle to the
    border of the widget, the view autoscrolls which is a convenient
    way for a user to position the selection.
    
    Please note that ImageToolSelector draws the image in two layers.
    One darkened and the selection rectangle in normal luminosity.
    Because it uses two draw operations instead one one like
    `ImageToolDragger` does, it is significantly slower than that
    tool. Therefore, it makes sense for a user of this library to set
    the interpolation to ``gtk.gdk.INTERP_NEAREST`` when using this
    tool to ensure that performance is acceptable to the users of the
    program.

    Zoom bug
    ========
    There is a small bug in ImageToolSelector that becomes apparent
    when the zoom factor is greater than about 30. The edge of the
    selection rectangle may in that case intersect a pixel:

    The bug is caused by bug `389832
    <http://bugzilla.gnome.org/show_bug.cgi?id=389832>`__ in
    `gdk-pixbuf
    <http://developer.gnome.org/doc/API/2.0/gdk-pixbuf/>`__. There is
    no way to solve this bug on ImageView's level (but if someone
    knows how, I'd really like to know).

    :group Signals: sig_selection_changed
    :group Read-write properties: get_selection, set_selection
    '''

    def __init__(self, view):
        '''
        Creates a new selector tool for the specified view with default
        values. The default values are:

        * selection : (0, 0) - [0, 0]

        :param view: an `ImageView`
        '''

    def sig_selection_changed(cls):
        '''
        The selection-changed signal is emitted when the selection
        rectangle on the selector is moved or resized. It is inteded
        to be used by applications that wants to print status
        information. For example:

        .. python::

           def sel_changed_cb(selector):
               rect = selector.get_selection()
               if not rect.width or not rect.height):
                   print 'No selection'
               else:
                   fmt = 'The selection is %d, %d - %d, %d'
                   print fmt % (rect.x, rect.y, rect.width, rect.height)
           ...
           selector.connect('selection-changed', sel_changed_cb)
        '''
    
    def get_selection(self):
        '''
        Returns the current selection rectangle in image space
        coordinates. If either the width or the height of the
        rectangle is zero, then noting is selected and the selection
        should be considered inactive.

        :see: `sig_selection_changed` for an example
        :return: the selection rectangle
        '''

    def set_selection(self, rect):
        '''
        Sets the selection rectangle for the tool. Setting this
        attribute will cause the widget to immidiately repaint itself
        if its view is realized.

        This method does nothing under the following circumstances:

        * If the views pixbuf is ``None``.
        * If `rect` is wider or taller than the size of the pixbuf.
        * If `rect` equals the current selection rectangle.

        If the selection falls outside the pixbufs area, its position
        is moved so that it is within the pixbuf.

        Calling this method causes the `sig_selection_changed` signal
        to be emitted.

        The default selection is (0,0) - [0,0].

        :param rect: selection rectangle in image space coordinates
        '''


class ImageNav:
    '''
    ImageNav is a popup window that shows a downscaled preview of the
    pixbuf that `ImageView` is showing. The user can drag around a
    rectangle which indicates the current view of the image.

    This class is used by `ImageScrollWin` itself. It is probably not
    very useful for clients of this library.

    ImageNav has the same keybindings that ImageView has. All
    keypresses that it receives are passed along to the view.

    ImageNav is a subclass of ``gtk.Window``.

    :see: `ImageScrollWin`
    '''

    def __init__(self, view):
        '''
        Creates a new image navigator for showing thumbnails of the
        view. The default values are:

        * pixbuf : ``None``
        '''

    def get_pixbuf(self):
        '''
        Returns the downscaled pixbuf of the views pixbuf that the
        image navigator shows, or ``None`` if that pixbuf has not been
        created yet.

        The pixbuf is by default ``None``.

        :return: the pixbuf in the image window that this navigator
            shows, or ``None`` if it has not been created.
        '''

    def grab(self):
        pass

    def release(self):
        pass

    def show_and_grab(self, center_x, center_y):
        '''
        Show the navigator centered around the point (`center_x`,
        `center_y`) and grab mouse and keyboard events. The grab
        continues until a left mouse button release event is received
        which causes the widget to hide.

        :param center_x: x coordinate of center position
        :param center_y: y coordinate of center position
        '''

class ImageScrollWin:
    '''
    ImageScrollWin provides a widget similar in appearance to
    ``gtk.ScrollableWindow`` that is more suitable for displaying
    `ImageView`'s.

    In particular, this widget draws a crosshair icon in the bottom
    right corner. Pressing that icon brings up an `ImageNav` which
    shows a thumbnailed overview of the viewed pixbuf.

    ImageScrollWin is a subclass of ``gtk.Table``.
    
    :see: `GtkScrolledWindow
        <http://developer.gnome.org/doc/API/2.0/gtk/GtkScrolledWindow.html>`__
        the GTK widget that ImageScrollWin mimics
    :see: `ImageNav`        
    '''
    
    def __init__(self, view):
        '''
        Creates a new ImageScrollWin containing the view.

        The widget is built using four subwidgets arranged inside a
        ``gtk.Table`` with two columns and two rows. Two scrollbars,
        one navigator button (the decorations) and one `ImageView`.

        When the ImageView fits inside the window, the decorations are
        hidden.
        '''

class ImageView:
    '''
    ImageView is a full-featured general purpose image viewer widget
    for GTK. It provides a scrollable, zoomable pane in which a pixbuf
    can be displayed.

    Keybindings
    ===========
    When focused, GtkImageView responds to the following keybindings:

    +----------------------------------------------------+----------------------+--------------------------------------+
    |Keys                                                |Corresponding function|Description                           |
    +====================================================+======================+======================================+
    |``GDK_KP_Add``, ``GDK_equal``, ``GDK_plus``         |`zoom_in()`           |Causes the widget to zoom in one step |
    +----------------------------------------------------+----------------------+--------------------------------------+
    |``GDK_KP_Subtract``, ``GDK_minus``                  |`zoom_out()`          |Causes the widget to zoom out one step|
    +----------------------------------------------------+----------------------+--------------------------------------+
    |``GDK_1``                                           |`set_zoom()`          |Sets zoom to 100%                     |
    +----------------------------------------------------+----------------------+--------------------------------------+
    |``GDK_2``                                           |`set_zoom()`          |Sets zoom to 200%                     |
    +----------------------------------------------------+----------------------+--------------------------------------+
    |``GDK_3``                                           |`set_zoom()`          |Sets zoom to 300%                     |
    +----------------------------------------------------+----------------------+--------------------------------------+
    |``GDK_x``                                           |`set_fitting()`       |Sets fitting to ``True``              |
    +----------------------------------------------------+----------------------+--------------------------------------+
    |``GDK_Page_Up``, ``GDK_Up`` + ``GDK_SHIFT_MASK``    |n/a                   |Scroll the view half a page upwards   |
    +----------------------------------------------------+----------------------+--------------------------------------+
    |``GDK_Page_Down``, ``GDK_Down`` + ``GDK_SHIFT_MASK``|n/a                   |Scroll the view half a page downwards |
    +----------------------------------------------------+----------------------+--------------------------------------+
    |``GDK_Left`` + ``GDK_SHIFT_MASK``                   |n/a                   |Scroll the view half a page leftwards |
    +----------------------------------------------------+----------------------+--------------------------------------+
    |``GDK_Right`` + ``GDK_SHIFT_MASK``                  |n/a                   |Scroll the view half a page rightwards|
    +----------------------------------------------------+----------------------+--------------------------------------+

    Mouse actions
    =============
    When focused, ImageView responds to the following mouse actions:

    +-----------------------------------------+-----------------------------------------+
    |Mouse gesture                            |Description                              |
    +=========================================+=========================================+
    |Mouse wheel scroll + ``GDK_CONTROL_MASK``|Increase or decrease the zoom of the view|
    +-----------------------------------------+-----------------------------------------+

    Coordinate systems
    ==================

    Operations on ImageView are executed in three different 2d
    coordinate systems:

    * **Image coordinates**: each coordinate represents a pixel in the
      image. The range of valid coordinates goes from
      *(0,0)-(p.w,p.h)*, where *p.w* and *p.h* is the width and height
      of the image.
    * **Widget coordinates**: each coordinate represents a pixel in
      the image view widgets coordinate system. The range of valid
      coordinates goes from *(0,0)-(a.w,a.h)* where *a.w* and *a.h* is
      the allocated width and height of the widget. Naturally, these
      coordinates are only valid for as long as the widget is
      realized.
    * **Zoom coordinates**: this coordinate system is the most
      frequently used coordinate system in ImageView. The range of
      valid coordinates goes from *(0,0)-z(p.w,p.h)* where *p.w* and
      *p.h* are the width and height of the image and *z* is the
      current zoom of the view. In other words, this coordinate system
      is simply the image coordinate system scaled.

    Settings
    ========
    ImageView has a few settings that can be configured by users of
    the library. For example, when showing transparent images it may
    in certain cases be better to draw alpha transparent parts using
    the widgets background color instead of the default checkerboard:

    .. python::
       view.set_transp(gtkimageview.TRANSP_BACKGROUND)

    When the window that is showing the widget is fullscreened, other
    settings has to be tweaked to make the view look as good as
    possible:

    .. python::
       view.set_show_cursor(False)
       view.set_show_frame(False)
       view.set_black_bg(True)

    Naturally, you should reset these settings again when the view
    leaves fullscreen mode.

    Updating the image data
    =======================
    ImageView aggresively caches the scaled image data. This behaviour
    is most often beneficial and makes the widget very fast. For a
    concrete example, try opening a very large image (4000x2000 pixels
    or so) in ImageView. The widget will spend some time bilinearly
    downsampling the image. Then try minimizing and unminimizing the
    window. The image will reappear immidately because the view has
    cached it.

    However, this feature means that a client application must signal
    to the view when it changes the pixels on the pixbuf that the view
    shows. The right way to do that is to use `ImageView.set_pixbuf()`
    with the old pixbuf as the argument. In particular, code that
    merely tries to update the view by requesting that it should be
    redrawn will not work.

    .. python::

       # Do some operation on the pixbuf data here..
       view.queue_draw_area(10, 10, 50, 50)     # Incorrect!

    In future versions of PyGtkImageView a method will be added that
    marks an area of the pixbuf as dirty and forces the view to flush
    its cache.
    
    Example
    =======

    This is the minimal code needed for using ImageView:

    .. python::
       import gtk
       import gtk.gdk
       import gtkimageview

       win = gtk.Window()
       view = gtkimageview.ImageView()
       pixbuf = gtk.gdk.pixbuf_new_from_file("tests/gnome_logo.jpg")
       view.set_pixbuf(pixbuf)
       win.add(view)
       win.show_all()
       gtk.main()

    Note that because the example doesn't use `ImageScrollWin` many
    nice features aren't available.

    ImageView is a subclass of ``gtk.Widget``.
    
    :group Signals: sig_pixbuf_changed, sig_zoom_changed
    :group Read-only properties: get_viewport, get_draw_rect, get_check_colors
    :group Write-only properties: set_offset, set_transp
    :group Read-write properties: get_fitting, set_fitting,
        get_pixbuf, set_pixbuf, get_zoom, set_zoom,
        get_black_bg, set_black_bg, get_tool, set_tool,
        get_interpolation, set_interpolation,
        get_show_cursor, set_show_cursor, get_show_frame, set_show_frame
    :group Zooming: zoom_in, zoom_out    
    '''

    def __init__(self):
        '''
        Creates a new image view with default values. The default
        values are:

        * black bg : ``False``
        * fitting : ``True``
        * image tool : an `ImageToolDragger` object
        * interpolation mode : ``gtk.gdk.INTERP_BILINEAR``
        * offset : (0, 0)
        * pixbuf : ``None``
        * show cursor : ``True``
        * show frame : ``True``
        * transp : `TRANSP_GRID`
        '''
    
    def sig_pixbuf_changed(cls):
        '''
        The pixbuf-changed signal is emitted when the pixbuf the image
        view shows is changed. Listening to this signal is useful if
        you, for example, have a label that displays the width and
        height of the pixbuf in the view.

        .. python::

           def pixbuf_cb(view, label):
               new_pb = view.get_pixbuf()
               if not new_pb:
                   # Emtpy label if no pixbuf
                   label.set_text('')
                   return
               label.set_text('%d, %s' % (new_pb.get_width(), new_pb.get_height()))
           ...
           label = gtk.Label('')
           view.connect('pixbuf-changed', pixbuf_cb, label)
        '''

    def sig_zoom_changed(cls):
        '''
        The zoom-changed signal is emitted when the zoom factor of the
        view changes. Listening to this signal is useful if, for
        example, you have a label that displays the zoom factor of the
        view. Use `get_zoom()` to retrieve the value. For example:

        .. python::

           def zoom_cb(view, label):
               zoom = view.get_zoom()
               label.set_text("%d%%" % int(zoom * 100))
           ...
           label = gtk.Label('100%')
           view.connect('zoom-changed', zoom_cb, label)
        '''
    
    def zoom_in(self):
        '''
        Zoom in the view one step. Calling this method causes the
        widget to immidiately repaint itself.
        '''

    def zoom_out(self):
        '''
        Zoom in the view one step. The widget is immidiately
        repainted.
        '''

    def get_viewport(self):
        '''
        Returns a ``gtk.gdk.Rectangle`` with the current viewport. If
        pixbuf is ``None``, there is no viewport and ``None`` is
        returned.

        The current viewport is defined as the rectangle, in zoomspace
        coordinates as the area of the loaded pixbuf the image viewer
        is currently showing.

        :return: a ``gtk.gdk.Rectangle`` with the current viewport or
            ``None``
        '''

    def get_draw_rect(self):
        '''
        Returns the rectangle in the widget where the pixbuf is
        painted.
 
        For example, if the widgets allocated size is 100, 100 and the
        pixbufs size is 50, 50 and the zoom factor is 1.0, then the
        pixbuf will be drawn centered on the widget. The rectangle is
        then (25,25)-[50,50]. If the widget is unallocated or the
        pixbuf is ``None`` then ``None`` is returned instead.
 
        This method is useful when converting from widget to image or
        zoom space coordinates.

        :return: the rectangle in the widget where the pixbuf is drawn
            or ``None``.
        '''

    def get_check_colors(self):
        '''
        Returns a tuple with the two colors used to draw transparent
        parts of images with an alpha channel. Note that if the
        transparency setting of the view is `TRANSP_BACKGROUND`
        or `TRANSP_COLOR`, then both colors will be equal.

            >>> view.get_check_colors()
            (6710886, 10066329)

        :return: a tuple containing two color values used to draw
            transparent parts.
        '''

    def set_offset(self, x, y, invalidate = False):
        '''
        Sets the offset of where in the image the image viewer should
        begin displaying image data.

        The offset is clamped so that it will never cause the image
        viewer to display pixels outside the pixbuf. Setting this
        attribute causes the widget to repaint itself if it is
        realized.

        If `invalidate` is `True`, the views entire area will be
        invalidated instead of redrawn immidiately. The view is then
        queued for redraw, which means that additional operations can
        be performed on it before it is redrawn.

        The difference can sometimes be important like when you are
        overlaying data and get flicker or artifacts when setting the
        offset. If that happens, setting `invalidate` to ``True``
        could fix the problem. See the source code for
        GtkImageToolSelector_ for an example.

        .. _GtkImageToolSelector: http://trac.bjourne.webfactional.com/browser/gtkimageview/src/gtkimagetoolselector.c

        Normally, `invalidate` should always be `False` because it is
        much faster to repaint immidately than invalidating.

        :param x: x-component of the offset in zoom space coordinates
        :param y: y-component of the offset in zoom space coordinates
        :param invalidate: whether to invalidate the view or redraw
            immediately
        '''

    def set_transp(self, transp, transp_color = 0x000000):
        '''
        Sets how the view should draw transparent parts of images with
        an alpha channel. If `transp` is `TRANSP_COLOR`, the
        specified `transp_color` will be used. Otherwise the
        `transp_color` argument is ignored. If it is
        `TRANSP_BACKGROUND`, the background color of the widget
        will be used. If it is `TRANSP_GRID`, then a grid with
        light and dark gray boxes will be drawn on the transparent
        parts.

        Calling this method causes the widget to immidiately repaint.
        It also causes the `sig_pixbuf_changed` signal to be emitted.
        This is done so that other widgets (such as `ImageNav`) will
        have a chance to render a view of the pixbuf with the new
        transparency settings.

        The default values are:

        * transp : `TRANSP_GRID`
        * transp_color : ``0x000000``

        :param transp: transparency type to use when drawing
            transparent images
        :param transp_color: color to use when drawing transparent
            images.
        '''

    def get_pixbuf(self):
        '''
        Returns the pixbuf this view shows.

        :return: the pixbuf this view shows
        '''

    def set_pixbuf(self, pixbuf, reset_fit = True):
        '''
        Sets the pixbuf to display, or ``None`` to not display any
        pixbuf.

        Normally, `reset_fit` should be ``True`` which enables
        fitting. Which means that, initially, the whole pixbuf will be
        shown.

        Sometimes, fitting should not be reset. For example, if
        ImageView is showing an animation, it would be bad to reset
        fitting for each new frame. The parameter should then be
        ``False`` which leaves the fit mode of the view untouched.

        This method must also be used to signal to the view that the
        contents of the pixbuf it display has been changed. This is
        the right way to force the redraw:

        .. python::

           view.set_pixbuf(view.get_pixbuf(), False)

        If `reset_fit` is ``True``, the `sig_zoom_changed` signal is
        emitted, otherwise not. The `sig_pixbuf_changed` signal is
        also emitted.

        The default pixbuf is ``None``.

        :param pixbuf: the pixbuf to display
        :param reset_fit: whether to set fitting or not
        '''

    def get_zoom(self):
        '''
        Get the current zoom factor of the view.

        :return: the current zoom factor
        '''

    def set_zoom(self, zoom):
        '''
        Sets the zoom of the view.

        Fitting is always disabled after this method has run. The
        `sig_zoom_changed` signal is unconditionally emitted.

        :param zoom: the new zoom factor
        '''

    def get_fitting(self):
        '''
        Returns the fitting setting of the view.

        :return: ``True`` if the view is fitting the image, ``False``
            otherwise
        '''

    def set_fitting(self, fitting):
        '''
        Sets whether to fit or not. If ``True``, then the view will
        adapt the zoom so that the whole pixbuf is visible.

        Setting the fitting causes the widget to immidiately repaint
        itself.

        Fitting is by default ``True``.

        :param fitting: whether to fit the image or not
        '''

    def get_black_bg(self):
        '''
        Returns whether the view renders the widget on a black
        background or not.

        :return: ``True`` if a black background is used, otherwise
            ``False``.
        '''

    def set_black_bg(self, black_bg):
        '''
        If ``True``, the view uses a black background. If ``False``,
        the view uses the default (normally gray) background.

        The default value is ``False``.

        :param black_bg: whether to use a black background or not
        '''

    def get_tool(self):
        '''
        Returns views tool which is an instance of a `IImageTool`.

        :return: the currently bound tool
        '''

    def set_tool(self, tool):
        '''
        Set the image tool to use. If the new tool is the same as the
        current tool, then nothing will be done. Otherwise
        `IImageTool.pixbuf_changed()` is called so that the tool has a
        chance to generate initial data for the pixbuf.

        :param tool: the image tool to use (must not be ``None``)
        '''

    def get_interpolation(self):
        '''
        Returns the current interpolation mode of the view.

        :return: the interpolation
        '''

    def set_interpolation(self, interp):
        '''
        Sets the interpolation mode of how the view.
        ``gtk.gdk.INTERP_HYPER`` is the slowest, but produces the best
        results. ``gtk.gdk.INTERP_NEAREST`` is the fastest, but
        provides bad rendering quality. ``gtk.gdk.INTERP_BILINEAR`` is
        a good compromise.

        Setting the interpolation mode causes the widget to immidiately
        repaint itself.

        The default interpolation mode is ``gtk.gdk.INTERP_BILINEAR``.

        :param interp: the interpolation to use. A
            ``gtk.gdk.InterpType`` object.
        '''

    def get_show_cursor(self):
        '''
        Returns whether to show the mouse cursor when the mouse is
        over the widget or not.
        
        :return: ``True`` if the cursor is shown, otherwise ``False``.
        '''

    def set_show_cursor(self, show_cursor):
        '''
        Sets whether to show the mouse cursor when the mouse is over
        the widget or not. Hiding the cursor is useful when the widget
        is fullscreened.

        The default value is ``True``.

        :param show_cursor: whether to show the cursor or not
        '''

    def get_show_frame(self):
        '''
        Returns whether a one pixel frame is drawn around the pixbuf
        or not.

        :return: ``True`` if a frame is drawn around the pixbuf,
            otherwise ``False``
        '''

    def set_show_frame(self, show_frame):
        '''
        Sets whether to draw a frame around the image or not. When
        ``True``, a one pixel wide frame is shown around the image.
        Setting this attribute causes the widget to immidiately
        repaint itself.

        The default value is ``True``.

        :param show_frame: whether to show a frame around the pixbuf
            or not
        '''
        

class AnimView(ImageView):
    '''
    AnimView subclasses ImageView. It has the same look and feel as
    its parent but is also capable of displaying GIF animations.

    Keybindings
    ===========
    AnimView uses a few more keybindings in addition to those used by
    ImageView:

    +---------+----------------------+-------------------------------------+
    |Keys     |Corresponding function|Description                          |
    +=========+======================+=====================================+
    |``GDK_p``|`set_is_playing()`    |Stops or resumes the animation       |
    +---------+----------------------+-------------------------------------+
    |``GDK_j``|`step()`              |Steps the animation one frame forward|
    +---------+----------------------+-------------------------------------+

    :group Read-write properties: get_anim, set_anim,
        get_is_playing, set_is_playing
    :group Signals: sig_anim_changed
    :see: `ImageView`
    :see: The file ``./tests/ex-anim.c`` program for an example of how
        this widget is used.
    '''

    def __init__(self):
        '''
        Creates a new AnimView with default values. The default values
        are:

        * anim : ``None``
        * is playing : ``False``
        '''

    def get_anim(self):
        '''
        Returns the current animation of the view.

        :return: the current animation
        '''

    def set_anim(self, anim):
        '''
        Sets the pixbuf animation to play, or ``None`` to not play any
        animation.

        The effect of this method is analoguous to
        `ImageView.set_pixbuf()`. Fitting is set to ``True`` so that
        the whole area of the animation fits in the view. Three
        signals are emitted, first the `ImageView` will emit
        `ImageView.sig_zoom_changed` and then
        `ImageView.sig_pixbuf_changed`.

        The default pixbuf animation is ``None``.

        :param anim: a pixbuf animation to play
        '''

    def get_is_playing(self):
        '''
        Returns whether the animation is playing or not. If there is
        no current animation, this method will always returns
        ``False``

        :return: ``True`` if an animation is playing, ``False``
            otherwise.
        '''

    def set_is_playing(self, playing):
        '''
        Sets whether the animation should play or not. If there is no
        current animation this method does not have any effect.

        :param playing: ``True`` to play the animation, ``False``
            otherwise.
        '''

    def step(self):
        '''
        Steps the animation one frame forward. If the animation is
        playing it will be stopped. Will it wrap around if the
        animation is at its last frame?
        '''

def zooms_get_zoom_in(zoom):
    '''
    Returns the zoom factor that is one step larger than the supplied
    zoom factor.

    :param zoom: a zoom factor
    :return: a zoom factor that is one step larger than the supplied
        one
    '''

def zooms_get_zoom_out(zoom):
    '''
    Returns the zoom factor that is one step smaller than the supplied
    zoom factor.

    :param zoom: a zoom factor
    :return: a zoom factor that is one step smaller than the supplied
        one
    '''

def zooms_get_min_zoom():
    '''
    Returns the minimum allowed zoom factor.

    :return: the minimal zoom factor
    '''

def zooms_get_max_zoom():
    '''
    Returns the maximum allowed zoom factor.

    :return: the maximal zoom factor
    '''

def zooms_clamp_zoom(zoom):
    '''
    Returns the zoom factor clamped to the minumum and maximum allowed
    value.

    :param zoom: a zoom factor
    :return: the zoom factor clamped to the interval [min, max]
    '''
