# coding: utf-8


"""
    PyLucid models
    ~~~~~~~~~~~~~~
    
    :copyleft: 2009-2011 by the PyLucid team, see AUTHORS for more details.
    :license: GNU GPL v3 or above, see LICENSE for more details.
"""


from django.conf import settings
from django.contrib import messages
from django.contrib.auth.models import User
from django.contrib.sites.managers import CurrentSiteManager
from django.contrib.sites.models import Site
from django.core.exceptions import ObjectDoesNotExist
from django.db import models
from django.template.loader import render_to_string
from django.utils.safestring import mark_safe
from django.utils.translation import ugettext_lazy as _

# http://code.google.com/p/django-tools/
from django_tools.middlewares import ThreadLocal
from django_tools.utils.messages import failsafe_message

from pylucid_project.utils import form_utils



TAG_INPUT_HELP_URL = \
"http://google.com/search?q=cache:django-tagging.googlecode.com/files/tagging-0.2-overview.html#tag-input"


class BaseModel(models.Model):
    def get_absolute_url(self):
        raise NotImplementedError
    get_absolute_url.short_description = _('absolute url')

    def get_site(self):
        raise NotImplementedError
    get_site.short_description = _('on site')

    def get_absolute_uri(self):
        """ returned the complete absolute URI (with the domain/host part) """
        request = ThreadLocal.get_current_request()
        is_secure = request.is_secure()
        if is_secure:
            protocol = "https://"
        else:
            protocol = "http://"
        site = self.get_site()
        domain = site.domain

        if "://" in domain:
            domain2 = domain.split("://", 1)[-1]
            msg = (
                "Wrong site domain %r: protocol should not inserted there!"
                " (Please change it to: %r)"
            ) % (domain, domain2)
            messages.error(request, msg)
            domain = domain2

        absolute_url = self.get_absolute_url()
        return protocol + domain + absolute_url
    get_absolute_uri.short_description = _('absolute uri')

    class Meta:
        abstract = True



class BaseModelManager(models.Manager):
    def easy_create(self, cleaned_form_data, extra={}):
        """
        Creating a new model instance with cleaned form data witch can hold more data than for
        this model needed.
        """
        keys = self.model._meta.get_all_field_names()
        model_kwargs = form_utils.make_kwargs(cleaned_form_data, keys)
        model_kwargs.update(extra)
        model_instance = self.model(**model_kwargs)
        model_instance.save()
        return model_instance

    def get_by_prefered_language(self, request, queryset, show_lang_errors=False):
        """
        return a item from queryset in this way:
         - try to get in current language
         - if not exist: try to get in system default language
         - if not exist: use the first found item
        """
        item = None
        tried_languages = []
        languages = request.PYLUCID.languages # languages are in client prefered order
        for language in languages:
            try:
                item = queryset.get(language=language)
            except ObjectDoesNotExist:
                tried_languages.append(language)
            else:
                break

        if show_lang_errors:
            current_language = request.PYLUCID.current_language
            if item.language != current_language:
                try:
                    item2 = queryset.get(language=current_language)
                except ObjectDoesNotExist:
                    pass
                else:
                    msg = render_to_string("pylucid/includes/language_info_link.html",
                        {
                            "item": item2,
                            "language": current_language.description,
                        }
                    )
                    messages.info(request, msg)


        return item, tried_languages


#------------------------------------------------------------------------------


class SiteM2M(models.Model):
    """ Base model with sites M2M and CurrentSiteManager. """
    objects = models.Manager()
    sites = models.ManyToManyField(Site, default=[settings.SITE_ID])
    on_site = CurrentSiteManager('sites')

    def site_info(self):
        """ for admin.ModelAdmin list_display """
        sites = self.sites.all()
        return ", ".join([site.name for site in sites])
    site_info.short_description = _('Exists on site')
    site_info.allow_tags = False

    class Meta:
        abstract = True


class AutoSiteM2M(SiteM2M):
    """ Base model with sites M2M and CurrentSiteManager. """
    def save(self, *args, **kwargs):
        """
        Automatic current site, if not exist.
        
        I don't know why default=[settings.SITE_ID] is not enough, see also:
            http://www.python-forum.de/viewtopic.php?t=21022 (de)
        """
        if self.pk == None:
            # instance needs to have a primary key value before a many-to-many relationship can be used. 
            super(AutoSiteM2M, self).save(*args, **kwargs)
            if "force_insert" in kwargs:
                # we can't pass force insert to the real save method, because we
                # have save it yet.
                del kwargs["force_insert"]

        if self.sites.count() == 0:
            if settings.DEBUG:
                failsafe_message("Automatic add site id '%s' to %r" % (settings.SITE_ID, self))
            self.sites.add(settings.SITE_ID)

        super(AutoSiteM2M, self).save(*args, **kwargs)

    class Meta:
        abstract = True

#------------------------------------------------------------------------------


class UpdateInfoBaseModel(models.Model):
    """
    Base model with update info attributes, used by many models.
    The createby and lastupdateby ForeignKey would be automaticly updated. This needs the 
    request object as the first argument in the save method.
    """
    objects = models.Manager()

    createtime = models.DateTimeField(auto_now_add=True, help_text="Create time")
    lastupdatetime = models.DateTimeField(auto_now=True, help_text="Time of the last change.")

    createby = models.ForeignKey(User, editable=False, related_name="%(class)s_createby",
        null=True, blank=True, # <- If the model used outsite a real request (e.g. unittest, db shell)
        help_text="User how create this entry.")
    lastupdateby = models.ForeignKey(User, editable=False, related_name="%(class)s_lastupdateby",
        null=True, blank=True, # <- If the model used outsite a real request (e.g. unittest, db shell)
        help_text="User as last edit this entry.")

    def save(self, *args, **kwargs):
        """
        Automatic update createby and lastupdateby attributes with the request object witch must be
        the first argument.
        """
        current_user = ThreadLocal.get_current_user()

        if current_user and isinstance(current_user, User):
            if self.pk == None or kwargs.get("force_insert", False): # New model entry
                self.createby = current_user
            self.lastupdateby = current_user

        return super(UpdateInfoBaseModel, self).save(*args, **kwargs)

    class Meta:
        abstract = True

#------------------------------------------------------------------------------

class PermissionsBase(object):
    """
    Shared model methods around view/edit permissions.
    Used in PageTree and PageMeta
    """
    def validate_permit_group(self, attribute, exclude, message_dict):
        """
        Prevents that a unprotected page created below a protected page.
        validate self.permitViewGroup and self.permitEditGroup
        """
        if attribute in exclude:
            return

        parent_page = self.recusive_attribute(attribute)
        if parent_page is None:
            # So parent page back to root has set a permission group
            return

        # we are below a protected page -> Check if permission group is the same.
        parent_page_group = getattr(parent_page, attribute)
        own_group = getattr(self, attribute)

        if parent_page_group == own_group:
            # permission is the same -> ok
            return

        # Add validation error message
        msg = _(
            "Error: Parent page <strong>%(parent_page)s</strong> used <strong>%(parent_page_group)s</strong>!"
            " You must used <strong>%(parent_page_group)s</strong> for this page, too."
        ) % {
            "parent_page": parent_page.get_absolute_url(),
            "parent_page_group": parent_page_group,
        }
        message_dict[attribute] = (mark_safe(msg),)

    def check_sub_page_permissions(self, attributes, exclude, message_dict, queryset):
        """
        Warn user if PageTree permissions mismatch with sub pages.
        
        self.check_sub_page_permissions(
            ("permitViewGroup", "permitEditGroup"),
            exclude, message_dict, queryset
        )
        
        """
        request = ThreadLocal.get_current_request()
        if request is None:
            # Check only if we are in a request
            return

        attributes2 = []
        for attribute in attributes:
            if attribute not in exclude and attribute not in message_dict:
                # ...and don't check if ValidationError exist for this field
                attributes2.append(attribute)

        if not attributes2:
            return

        sub_pages = queryset.only(*attributes2)

        for attribute in attributes2:
            own_permission = getattr(self, attribute)
            for sub_page in sub_pages:
                sub_page_permission = getattr(sub_page, attribute)
                if sub_page_permission != own_permission:
                    msg = _(
                        "Permission mismatch:"
                        " current %(attribute)s is set to '%(own_permission)s'"
                        " and sub page '%(slug)s' used '%(sub_permission)s'."
                        " This may be ok."
                    ) % {
                        "slug": sub_page.get_absolute_url(),
                        "attribute": attribute,
                        "own_permission": own_permission,
                        "sub_permission": sub_page_permission,
                    }
                    messages.warning(request, msg)
