#ifndef MTGP64_FAST_H
#define MTGP64_FAST_H
/**
 * @file mtgp64-fast.h
 *
 * @brief Mersenne Twister for Graphic Processors (mtgp64), which
 * generates 64-bit unsigned integers and double precision floating
 * point numbers based on IEEE 754 format.
 *
 * @author Mutsuo Saito (Hiroshima University)
 * @author Makoto Matsumoto (Hiroshima University)
 *
 * Copyright (C) 2009 Mutsuo Saito, Makoto Matsumoto and
 * Hiroshima University. All rights reserved.
 *
 * The new BSD License is applied to this software, see LICENSE.txt
 */
#include <string.h>
#include <assert.h>
#include <stdio.h>
#include <stdint.h>

#if defined(__cplusplus)
extern "C" {
#endif
/**
 * \struct MTGP64_PARAMS_FAST_T
 * MTGP64 parameters.
 * Some element is redundant to keep structure simple.
 *
 * \b pos is a pick up position which is selected to have good
 * performance on graphic processors.  3 < \b pos < Q, where Q is a
 * maximum number such that the size of status array - Q is a power of
 * 2.  For example, when \b mexp is 44497, size of 64-bit status array
 * is 696, and Q is 184, then \b pos is between 4 and 183. This means
 * 512 parallel calculations is allowed when \b mexp is 44497.
 *
 * \b poly_sha1 is SHA1 digest of the characteristic polynomial of
 * state transition function. SHA1 is calculated based on printing
 * form of the polynomial. This is important when we use parameters
 * generated by the dynamic creator which
 *
 * \b mask This is a mask to make the dimension of state space have
 * just Mersenne Prime. This is redundant.
 */
typedef struct MTGP64_PARAMS_FAST_T {
    int mexp;			/**< Mersenne exponent. This is redundant. */
    int pos;			/**< pick up position. */
    int sh1;			/**< shift value 1. 0 < sh1 < 32. */
    int sh2;			/**< shift value 2. 0 < sh2 < 32. */
    uint64_t tbl[16];		/**< a small matrix. */
    uint64_t tmp_tbl[16];	/**< a small matrix for tempering. */
    uint64_t dbl_tmp_tbl[16];	/**< a small matrix for tempering and
				 converting to double. */
    uint64_t mask;		/**< This is a mask for state space */
    unsigned char poly_sha1[21]; /**< SHA1 digest */
} mtgp64_params_fast_t;

/**
 * \struct MTGP64_STATUS_FAST_T
 * MTGP64 internal state array.
 * In this structure, we use \b larger_size and \b larger_mask to
 * avoid slow operation of remainder (%).
 *
 * \b size is the minimum needed size to represent linear space of
 * two element filed as 64-bit array.
 *
 * \b large_size is a minimum power of 2 which is larger than \b size.
 * \b array has this size.
 *
 * \b large_mask is a bit mask to update \b idx in a fast way.
 * \b idx is updated by `logical and' \b & with large_mask.
 */
typedef struct MTGP64_STATUS_FAST_T {
    int idx;			/**< index */
    int size;			/**< minimum needed size */
    int large_size;		/**< real size of array */
    int large_mask;		/**< bit mask to update \b idx */
    uint64_t array[];		/**< internal state array */
} mtgp64_status_fast_t;

/**
 * \struct MTGP64_STATUS_FAST_T
 * MTGP64 interface structure
 */
typedef struct MTGP64_FAST_T {
    mtgp64_params_fast_t params; /**< parameters */
    mtgp64_status_fast_t *status; /**< internal state */
} mtgp64_fast_t;

/**
 * parameter constants tables for MEXP=23209.
 * \b size - \b pos > 256.
 */
extern mtgp64_params_fast_t mtgp64_params_fast_23209[128];
/**
 * parameter constants tables for MEXP=44497.
 * \b size - \b pos > 512.
 */
extern mtgp64_params_fast_t mtgp64_params_fast_44497[128];
/**
 * parameter constants tables for MEXP=110503.
 * \b size - \b pos > 1024.
 */
extern mtgp64_params_fast_t mtgp64_params_fast_110503[128];

int mtgp64_init(mtgp64_fast_t *mtgp64,
		const mtgp64_params_fast_t *para, uint64_t seed);
void mtgp64_init_state(uint64_t array[],
		      const mtgp64_params_fast_t *para, uint64_t seed);
int mtgp64_init_by_array(mtgp64_fast_t *mtgp64,
			 const mtgp64_params_fast_t *para,
			uint64_t *array, int length);
int mtgp64_init_by_str(mtgp64_fast_t *mtgp64,
		       const mtgp64_params_fast_t *para,
		       char *str);
void mtgp64_free(mtgp64_fast_t *mtgp64);
void mtgp64_print_idstring(const mtgp64_fast_t *mtgp64, FILE *fp);

inline static void mtgp64_do_recursion(uint64_t *r, uint64_t x1,
				       uint64_t x2, uint64_t y,
				       int sh1, int sh2,
				       uint64_t mask, uint64_t tbl[16]);
inline static void mtgp64_next_state(mtgp64_fast_t *mtgp64);
inline static uint64_t mtgp64_genrand_uint64(mtgp64_fast_t *mtgp64);
inline static double mtgp64_genrand_close1_open2(mtgp64_fast_t *mtgp64);
inline static double mtgp64_genrand_close_open(mtgp64_fast_t *mtgp64);
inline static double mtgp64_genrand_open_close(mtgp64_fast_t *mtgp64);
inline static double mtgp64_genrand_open_open(mtgp64_fast_t *mtgp64);

/*
 * PRIVATE INLINE FUNCTIONS
 */
/**
 * This is a recursion formula of the generator.
 * MTGP64 is a 64-bit generator, but using 32-bit operations to fit to
 * graphic processors.

 * @param[out] r output
 * @param[in] x1 the farthest part of state array.
 * @param[in] x2 the second farthest part of state array.
 * @param[in] y a part of state array.
 * @param[in] sh1 the shift parameter 1.
 * @param[in] sh2 the shift parameter 2.
 * @param[in] mask the bit mask parameter.
 * @param[in] tbl the matrix parameter.
 */
inline static void mtgp64_do_recursion(uint64_t *r, uint64_t x1,
				       uint64_t x2, uint64_t y,
				       int sh1, int sh2,
				       uint64_t mask, uint64_t tbl[16]) {
    uint64_t x;
    uint32_t xh;
    uint32_t xl;
    uint32_t yh;
    uint32_t yl;

    x = (x1 & mask) ^ x2;
    xh = (uint32_t)(x >> 32);
    xl = (uint32_t)(x & 0xffffffffU);
    yh = (uint32_t)(y >> 32);
    yl = (uint32_t)(y & 0xffffffffU);
    xh ^= xh << sh1;
    xl ^= xl << sh1;
    yh = xl ^ (yh >> sh2);
    yl = xh ^ (yl >> sh2);
    *r = ((uint64_t)yh << 32) | yl;
    *r ^= tbl[yl & 0x0f];
}

/**
 * The state transition function.
 * @param[in,out] mtgp64 the all in one structure
 */
inline static void mtgp64_next_state(mtgp64_fast_t *mtgp64) {
    uint64_t *array = mtgp64->status->array;
    int idx;
    int pos = mtgp64->params.pos;
    int large_size = mtgp64->status->large_size;
    uint64_t large_mask = mtgp64->status->large_mask;
    int size = mtgp64->status->size;

    mtgp64->status->idx = (mtgp64->status->idx + 1) & large_mask;
    idx = mtgp64->status->idx;
    mtgp64_do_recursion(&(array[idx]),
			array[(idx - size + large_size) & large_mask],
			array[(idx - size + large_size + 1) & large_mask],
			array[(idx + pos - size + large_size) & large_mask],
			mtgp64->params.sh1,
			mtgp64->params.sh2,
			mtgp64->params.mask,
			mtgp64->params.tbl);
}

/**
 * The tempering function.
 * @param[in] tmp_tbl the pre-computed tempering table.
 * @param[in] r the value to be tempered.
 * @param[in] t the tempering helper value.
 * @return the tempered value.
 */
inline static uint64_t mtgp64_temper(const uint64_t tmp_tbl[16],
				     uint64_t r, uint64_t t) {
    t ^= t >> 16;
    t ^= t >> 8;
    r ^= tmp_tbl[t & 0x0f];
    return r;
}

/**
 * The tempering and converting function.
 * @param[in] dbl_tmp_tbl the pre-computed tempering table.
 * @param[in] r the value to be tempered.
 * @param[in] t the tempering helper value.
 * @return the tempered value.
 */
inline static double mtgp64_temper_double(const uint64_t dbl_tmp_tbl[16],
					  uint64_t r, uint64_t t) {
    union {
	uint64_t u;
	double d;
    } x;
    t ^= t >> 16;
    t ^= t >> 8;
    r = r >> 12;
    x.u = r ^ dbl_tmp_tbl[t & 0x0f];
    return x.d;
}

/**
 * The tempering and converting function for generating floating point
 * number \b f (0 < \b f < 1).
 * @param[in] dbl_tmp_tbl the pre-computed tempering table.
 * @param[in] r the value to be tempered.
 * @param[in] t the tempering helper value.
 * @return the tempered value.
 */
inline static double mtgp64_temper_double_open(const uint64_t dbl_tmp_tbl[16],
					       uint64_t r, uint64_t t) {
    union {
	uint64_t u;
	double d;
    } x;
    t ^= t >> 16;
    t ^= t >> 8;
    r = (r >> 12) | 1;
    x.u = r ^ dbl_tmp_tbl[t & 0x0f];
    return x.d;
}

/*
 * PUBLIC INLINE FUNCTIONS
 */
/**
 * This function generates and returns 64-bit unsigned integer.
 * mtgp64_init(), mtgp64_init_by_array() or mtgp64_init_by_str() must
 * be called before this function.
 *
 * @param[in,out] mtgp64 MTGP all in one structure.
 * @return 64-bit unsigned integer.
 */
inline static uint64_t mtgp64_genrand_uint64(mtgp64_fast_t *mtgp64) {
    int idx;
    uint64_t *tmp_tbl = mtgp64->params.tmp_tbl;
    uint64_t *array = mtgp64->status->array;
    int pos = mtgp64->params.pos;
    int large_size = mtgp64->status->large_size;
    int size = mtgp64->status->size;
    uint64_t large_mask = mtgp64->status->large_mask;

    mtgp64_next_state(mtgp64);
    idx = mtgp64->status->idx;
    return mtgp64_temper(tmp_tbl,
			 array[idx],
			 array[(idx + pos - 1 - size + large_size)
			       & large_mask]);
}

/**
 * This function generates and returns double precision pseudorandom
 * number which distributes uniformly in the range [1, 2).
 * mtgp64_init(), mtgp64_init_by_array() or mtgp64_init_by_str() must
 * be called before this function.
 *
 * @param[in,out] mtgp64 MTGP all in one structure.
 * @return double precision floating point pseudorandom number
 */
inline static double mtgp64_genrand_close1_open2(mtgp64_fast_t *mtgp64) {
    int idx;
    uint64_t *dbl_tmp_tbl = mtgp64->params.dbl_tmp_tbl;
    uint64_t *array = mtgp64->status->array;
    int pos = mtgp64->params.pos;
    int large_size = mtgp64->status->large_size;
    int size = mtgp64->status->size;
    uint64_t large_mask = mtgp64->status->large_mask;

    mtgp64_next_state(mtgp64);
    idx = mtgp64->status->idx;
    return mtgp64_temper_double(dbl_tmp_tbl,
				array[idx],
				array[(idx + pos - 1 - size + large_size)
				      & large_mask]);
}

/**
 * This function generates and returns double precision pseudorandom
 * number which distributes uniformly in the range [0, 1).
 * mtgp64_init(), mtgp64_init_by_array() or mtgp64_init_by_str() must
 * be called before this function.
 *
 * @param[in,out] mtgp64 MTGP all in one structure.
 * @return double precision floating point pseudorandom number
 */
inline static double mtgp64_genrand_close_open(mtgp64_fast_t *mtgp64) {
    return mtgp64_genrand_close1_open2(mtgp64) - 1.0;
}

/**
 * This function generates and returns double precision pseudorandom
 * number which distributes uniformly in the range (0, 1].
 * mtgp64_init(), mtgp64_init_by_array() or mtgp64_init_by_str() must
 * be called before this function.
 *
 * @param[in,out] mtgp64 MTGP all in one structure.
 * @return double precision floating point pseudorandom number
 */
inline static double mtgp64_genrand_open_close(mtgp64_fast_t *mtgp64) {
    return 2.0 - mtgp64_genrand_close1_open2(mtgp64);
}

/**
 * This function generates and returns double precision pseudorandom
 * number which distributes uniformly in the range (0, 1).
 * mtgp64_init(), mtgp64_init_by_array() or mtgp64_init_by_str() must
 * be called before this function.
 *
 * @param[in,out] mtgp64 MTGP all in one structure.
 * @return double precision floating point pseudorandom number
 */
inline static double mtgp64_genrand_open_open(mtgp64_fast_t *mtgp64) {
    double r;
    int idx;
    uint64_t *dbl_tmp_tbl = mtgp64->params.dbl_tmp_tbl;
    uint64_t *array = mtgp64->status->array;
    int pos = mtgp64->params.pos;
    int large_size = mtgp64->status->large_size;
    int size = mtgp64->status->size;
    uint64_t large_mask = mtgp64->status->large_mask;
    mtgp64_next_state(mtgp64);
    idx = mtgp64->status->idx;
    r = mtgp64_temper_double_open(dbl_tmp_tbl,
				  array[idx],
				  array[(idx + pos - 1 - size + large_size)
					& large_mask]);
    return r - 1.0;
}

#if defined(__cplusplus)
}
#endif

#endif
