import os.path
import os
from pandas import DataFrame
from pandas.io.parsers import read_csv
import numpy as np
import datetime
import gzip
import sys
import pickle


__version__ = '0.1.0'


class Error(Exception):
    pass


def _find_nomad_dir():
    """
    Will look for a NOMAD catalog directory in the following order and use the first found:
        as specified by environment variable:  $NOMAD_DIR
        ~/nomad
        ~/nomad_gz

    Note that this routine does no integrity checking beyond simply checking for the existence
    of the directory.
    """
    paths_to_search = [os.environ.get('NOMAD_DIR')]
    paths_to_search.append('~/nomad/')
    paths_to_search.append('~/nomad_gz/')
    for cur_path_to_test in paths_to_search:
        if cur_path_to_test is not None:
            if os.path.isdir(os.path.expanduser(cur_path_to_test)):
                return os.path.expanduser(cur_path_to_test)
    print "nomad.py:  No nomad installation found. "
    print "           User needs to specifiy location with, e.g.:"
    print "           nomad.set_nomad_path('~/my_nomad_dir')"
    return None


_nomad_dir = _find_nomad_dir()
_nomad_record_length_bytes = 22 * 4


"""
    Following listing of number of stars in each USNO-B1.0 was generated by running the following code in a full
    install of the catalog.  (Note:  only the accelerator files are needed.)
    This is used for decoding the integer entry in the NOMAD catalog into a USNO B1.0 star number.
    TODO: document code to generate the _usnob1_stars_per_file
    Here's a quick first version....needs work:
        def play_with_USNOB_numbers():
            filelist = glob.glob('*/*.acc')
            nstars_per_file = np.zeros([len(filelist)], dtype=np.int)
            for i, curfilename in enumerate(filelist):
                print curfilename
                lastline = open(curfilename, 'r').readlines()[-1]
                nstars_per_file[i] = int(lastline.split()[1]) + int(lastline.split()[2]) - 1
            f = open('acc_summary.txt', 'w')
            for i, curfilename in enumerate(filelist):
                f.write(curfilename[5:9] + ' ' + str(nstars_per_file[i]) + '\n')
            f.close()
"""
_usnob1_stars_per_file = np.array([322, 1169, 1823, 2612, 3371, 4139, 4808, 5765, 6444, 7322, 8016, 8909, 9670, 10314,
                                   11263, 11835, 13190, 14973, 16231, 17700, 20669, 23184, 26816, 28304, 31435, 33428,
                                   35381, 37796, 38808, 39156, 41500, 44794, 46042, 47331, 48725, 50009, 51695, 52946,
                                   52240, 52085, 53698, 54894, 56096, 56971, 57452, 57496, 57018, 58063, 58169, 59731,
                                   59652, 61062, 61576, 61485, 61466, 63310, 63306, 63681, 64476, 65877, 65452, 66213,
                                   67216, 67051, 68046, 68058, 69627, 73052, 75798, 78582, 82872, 87571, 92682, 97756,
                                   100864, 104973, 105860, 109878, 109098, 109109, 108937, 106702, 106182, 105003,
                                   105254, 102218, 100767, 100516, 99471, 100766, 100316, 101832, 102376, 102868, 104234,
                                   104787, 108139, 109205, 112527, 111954, 113577, 115244, 116881, 117955, 118217,
                                   118434, 119929, 121465, 122813, 123836, 124975, 126550, 128144, 128939, 130428,
                                   132562, 135897, 141742, 146595, 153319, 159736, 169726, 176177, 183895, 191318,
                                   193234, 193022, 194765, 194256, 196578, 193996, 191457, 190619, 188074, 186312,
                                   183860, 185818, 187596, 190087, 193349, 197948, 200828, 205082, 210720, 214383,
                                   219350, 222625, 226665, 231444, 235156, 238118, 241692, 246924, 250497, 255584,
                                   258416, 262831, 266058, 270435, 276428, 284383, 288846, 297821, 303760, 313491,
                                   317863, 326925, 338311, 351065, 364218, 447046, 474491, 506817, 520876, 522432,
                                   531027, 539203, 542403, 552881, 546832, 540962, 514613, 493017, 473456, 462544,
                                   455625, 455781, 461743, 466257, 465629, 475433, 480679, 500968, 508664, 519432,
                                   520393, 528083, 524984, 529948, 529990, 532562, 533738, 529047, 536051, 535179,
                                   542508, 544411, 544365, 541708, 545235, 561027, 562366, 577766, 575458, 577985,
                                   571268, 618881, 659641, 709672, 740766, 791860, 835359, 863263, 885092, 884682,
                                   899939, 901092, 904390, 897394, 874051, 800560, 770452, 749253, 735291, 734112,
                                   724409, 733649, 741701, 753091, 758987, 763966, 777820, 774044, 789055, 786689,
                                   798779, 795057, 789391, 775359, 778750, 798657, 793448, 806121, 784114, 777804,
                                   753972, 761240, 750261, 754235, 754080, 748366, 763460, 751827, 741396, 717554,
                                   703738, 734661, 791084, 843736, 861463, 889380, 915776, 946470, 973619, 993332,
                                   1015788, 1011352, 1011954, 997293, 951337, 896364, 864787, 857368, 838802, 847162,
                                   831316, 831901, 820314, 832769, 853858, 851011, 870906, 865702, 893468, 882305,
                                   880668, 842939, 837743, 832686, 828752, 821287, 801296, 812505, 811583, 824625,
                                   818959, 831240, 825865, 811980, 794296, 780852, 783875, 774490, 788995, 786836,
                                   797661, 863870, 966507, 1046615, 1072969, 1102005, 1130148, 1165979, 1163214, 1153892,
                                   1129347, 1102423, 1101112, 1087203, 1046206, 974479, 922956, 898909, 871262, 858917,
                                   847837, 853590, 853808, 853815, 846772, 841541, 834116, 829740, 834581, 830479,
                                   846752, 846831, 856890, 853485, 853606, 863551, 856903, 862409, 856026, 859597,
                                   861065, 861778, 860714, 873144, 867688, 859780, 860158, 862803, 866821, 856985,
                                   865025, 890419, 971260, 1023891, 1058944, 1076095, 1110507, 1140340, 1145636, 1137893,
                                   1115017, 1124505, 1115969, 1107457, 1074291, 1013663, 953747, 908479, 888285, 869566,
                                   859697, 850990, 838367, 827445, 829539, 823663, 821849, 811016, 821243, 810051,
                                   811644, 802577, 804441, 799059, 801840, 809837, 810168, 817468, 806812, 809270,
                                   806444, 805345, 793847, 794984, 786950, 785519, 794386, 793537, 800047, 788800,
                                   784113, 798537, 837428, 881388, 902530, 932286, 954198, 1003754, 1015007, 1018182,
                                   1023316, 1017370, 998837, 979964, 957416, 913907, 890150, 863365, 831875, 811066,
                                   792736, 795556, 790541, 799553, 798399, 796011, 807937, 801041, 804041, 796383,
                                   814232, 816772, 823262, 812829, 808416, 812822, 812356, 818952, 809627, 816703,
                                   801619, 797817, 798938, 784110, 777981, 765208, 762904, 752207, 758313, 756745,
                                   760019, 781504, 827233, 881073, 908340, 928394, 947256, 977446, 980754, 981545,
                                   996029, 990340, 987472, 975320, 955815, 925982, 890605, 853333, 824823, 817005,
                                   796934, 802122, 800158, 798902, 813649, 813514, 822576, 808642, 823352, 822189,
                                   831006, 818684, 821089, 828229, 828925, 851158, 841893, 852417, 833952, 843748,
                                   832727, 828306, 829921, 814599, 820888, 810625, 821353, 805791, 818666, 815668,
                                   817855, 836504, 866459, 923186, 971176, 1025919, 1050997, 1061632, 1076337, 1079128,
                                   1084733, 1073519, 1063928, 1038123, 1012116, 987805, 945640, 909852, 878057, 865007,
                                   839846, 833170, 830072, 820076, 835108, 838074, 862547, 851215, 865667, 850245,
                                   847904, 848330, 848791, 865642, 866930, 886415, 876883, 896017, 886032, 902264,
                                   900391, 898502, 906123, 898675, 911671, 901295, 907274, 912495, 936105, 941411,
                                   955211, 983534, 1025590, 1114628, 1172038, 1215131, 1244975, 1253647, 1263933,
                                   1250837, 1261017, 1236270, 1235517, 1167324, 1142490, 1095666, 1040985, 981374,
                                   942190, 930801, 906678, 912256, 918172, 919109, 923125, 913147, 931106, 931321,
                                   947449, 933094, 950038, 953099, 962441, 961168, 960823, 973313, 963788, 979653,
                                   969694, 980824, 969303, 970319, 967812, 964190, 970973, 955431, 965934, 952192,
                                   955970, 948235, 952093, 968590, 994247, 1062480, 1128788, 1171130, 1189988, 1208295,
                                   1209280, 1207908, 1197858, 1192690, 1192276, 1182737, 1173208, 1156118, 1110535,
                                   1064735, 1027051, 1015429, 989776, 998816, 984063, 964857, 944631, 933943, 938201,
                                   927898, 928696, 919642, 920576, 915230, 912776, 914202, 901268, 919043, 901830,
                                   900752, 885705, 888764, 879451, 872961, 880140, 869512, 896357, 881205, 905829,
                                   901079, 920986, 924005, 931153, 954238, 971425, 1025145, 1049405, 1072551, 1070955,
                                   1082510, 1090958, 1077112, 1091427, 1085273, 1104951, 1094398, 1079142, 1048757,
                                   1013855, 972329, 946217, 931976, 897879, 902637, 894856, 907610, 923233, 914852,
                                   915237, 899664, 900111, 872653, 880313, 873300, 879019, 869390, 849960, 852684,
                                   841266, 844413, 836404, 848976, 834699, 838707, 845364, 842967, 864622, 852018,
                                   862912, 848788, 861695, 859158, 869598, 907354, 956874, 1032335, 1095744, 1165693,
                                   1192294, 1213210, 1191558, 1186715, 1182222, 1175192, 1180628, 1167702, 1131600,
                                   1076789, 1023082, 961229, 913796, 879026, 854001, 842509, 843926, 843978, 840987,
                                   842199, 856655, 861666, 874152, 866637, 860745, 843822, 833936, 833694, 831613,
                                   832108, 813658, 800480, 789885, 777600, 780385, 778480, 787198, 775632, 774415,
                                   748643, 743469, 733440, 732730, 729742, 718681, 740278, 763054, 797479, 839794,
                                   854173, 872460, 872603, 877553, 862310, 873977, 866733, 873257, 879989, 852710,
                                   825179, 786206, 761117, 723453, 691978, 673300, 668379, 674323, 663918, 671450,
                                   665742, 670897, 670786, 672742, 671784, 670530, 670698, 663672, 672810, 664340,
                                   675951, 682507, 699103, 697539, 692890, 695494, 674927, 682403, 676852, 690950,
                                   700196, 704829, 709933, 703007, 698655, 694245, 723108, 740001, 766278, 790879,
                                   804176, 826479, 821534, 832035, 819058, 833255, 834309, 843351, 855745, 834477,
                                   815814, 783309, 755027, 712579, 687390, 673552, 666754, 676873, 664021, 674878,
                                   657130, 667065, 656510, 663071, 656034, 651656, 652763, 639297, 648935, 630923,
                                   639273, 615646, 616161, 613875, 606619, 603874, 597252, 598004, 589597, 592912,
                                   581004, 594559, 593476, 605153, 597423, 589421, 604840, 631893, 664305, 707212,
                                   747060, 792071, 820250, 815844, 807824, 822160, 849806, 876601, 888123, 852386,
                                   799267, 763133, 718566, 678505, 633194, 621620, 623982, 627788, 625720, 632535,
                                   635559, 643170, 640126, 629342, 630428, 631616, 632445, 630816, 634848, 627222,
                                   618275, 611568, 603359, 596777, 586464, 582391, 577854, 576103, 569710, 569350,
                                   571789, 571260, 574800, 576414, 579059, 577430, 593965, 619864, 645204, 669052,
                                   702009, 742807, 764986, 780018, 781123, 776539, 783938, 788930, 807308, 790220,
                                   765563, 741838, 717930, 695730, 664275, 643882, 642098, 642365, 629820, 624176,
                                   607997, 603489, 594044, 588724, 592261, 596149, 593362, 594039, 598475, 595575,
                                   601206, 600351, 599035, 596706, 592342, 591874, 591837, 598711, 595284, 597248,
                                   596380, 601362, 603373, 603747, 606689, 610520, 619766, 630999, 651163, 667385,
                                   698108, 719813, 741611, 749112, 758757, 755403, 758786, 767261, 768735, 757872,
                                   742972, 743307, 731274, 704412, 667731, 640137, 635453, 626702, 624245, 616278,
                                   623782, 619458, 628116, 622453, 628185, 635745, 638405, 639270, 632835, 630937,
                                   618282, 631044, 628834, 636031, 635388, 642654, 639339, 637234, 638553, 638800,
                                   647245, 644644, 648647, 650628, 662107, 665017, 673958, 695559, 704024, 728346,
                                   753921, 772425, 780281, 776160, 775949, 761855, 773536, 783706, 791463, 775111,
                                   747451, 730428, 705635, 671844, 641180, 633579, 632745, 632557, 629828, 632395,
                                   645726, 659216, 670185, 668101, 664902, 645316, 648679, 652791, 643182, 641084,
                                   636908, 649242, 659335, 659641, 654499, 658395, 667528, 671868, 657477, 646906,
                                   629719, 627748, 630478, 632224, 628109, 616424, 620070, 646785, 659809, 671220,
                                   698920, 717289, 725426, 726634, 726764, 725043, 741966, 754867, 761604, 748728,
                                   725916, 718643, 706393, 670605, 635108, 621064, 622262, 622388, 619354, 612384,
                                   611398, 604836, 607535, 598642, 600029, 602414, 608602, 611197, 605759, 604278,
                                   602127, 613763, 620022, 625641, 617278, 607739, 604246, 603566, 608764, 601454,
                                   602523, 603622, 607285, 607095, 602270, 607675, 616170, 633463, 628318, 628051,
                                   643327, 664600, 673305, 674060, 663089, 665555, 665586, 688754, 694943, 688934,
                                   671743, 660616, 633827, 599672, 577467, 567021, 572920, 570797, 571898, 565752,
                                   563759, 571296, 566548, 571037, 575457, 580872, 579815, 587218, 587837, 584257,
                                   581551, 577260, 580059, 580159, 576568, 565827, 563079, 559791, 558592, 567125,
                                   577348, 593586, 596419, 604727, 598518, 604098, 607430, 619733, 624515, 643479,
                                   667625, 690647, 713805, 712252, 730527, 724866, 749270, 767527, 767875, 742339,
                                   721014, 708088, 687187, 655547, 620770, 607893, 605617, 609174, 605391, 604347,
                                   601973, 598739, 601223, 591592, 594341, 594975, 601704, 604162, 607234, 610673,
                                   607087, 607124, 609425, 610198, 610285, 609620, 609060, 602002, 601336, 595735,
                                   597116, 588190, 588199, 595117, 597379, 595403, 602631, 616215, 625329, 633866,
                                   658347, 691286, 713340, 714644, 714635, 704288, 714249, 721372, 729730, 709481,
                                   688510, 684853, 670205, 644315, 616706, 596111, 587141, 585490, 565188, 551870,
                                   546328, 540621, 541346, 534166, 534550, 534626, 535416, 530613, 532251, 530168,
                                   526970, 530051, 529156, 535877, 537258, 534850, 533806, 539451, 548648, 546408,
                                   551355, 548508, 553019, 555004, 553330, 553480, 558540, 574683, 579628, 594387,
                                   614615, 643171, 657319, 655080, 650883, 646620, 647011, 651907, 659411, 650505,
                                   641111, 634694, 619528, 583934, 547719, 529163, 531819, 531812, 535515, 532914,
                                   527093, 529761, 531871, 538065, 532539, 539309, 545362, 543276, 531019, 515888,
                                   510226, 506344, 507127, 508090, 509898, 511681, 510472, 510010, 510473, 505811,
                                   501066, 503202, 502406, 504765, 505126, 508238, 517499, 533539, 543109, 555173,
                                   572238, 590839, 599050, 608497, 608025, 611051, 616761, 631594, 639659, 626523,
                                   616043, 610075, 589992, 548617, 515310, 504203, 506188, 505448, 504102, 508348,
                                   513592, 531112, 533835, 535224, 527241, 523074, 521076, 523680, 526732, 526107,
                                   529441, 526697, 533742, 534503, 540544, 542393, 543958, 545404, 532496, 527881,
                                   513914, 518901, 519567, 524723, 523325, 520438, 530161, 539585, 550303, 560301,
                                   587550, 610333, 623556, 624288, 621272, 618064, 622105, 640220, 638409, 628512,
                                   616676, 607554, 591709, 562659, 525581, 512612, 510778, 511069, 509121, 512681,
                                   513979, 515597, 511884, 511478, 514213, 514903, 513072, 510277, 513277, 510760,
                                   516690, 525518, 531812, 530483, 521095, 512181, 513305, 514874, 516989, 512115,
                                   508831, 509245, 510147, 520588, 513707, 516155, 531626, 542884, 543776, 539162,
                                   549710, 566509, 580444, 579941, 577312, 577250, 587003, 595975, 597743, 591007,
                                   580555, 580288, 559924, 524787, 487371, 475833, 475612, 476569, 477348, 475570,
                                   473628, 477141, 477030, 481182, 478507, 479390, 477442, 474191, 473339, 473009,
                                   473713, 472977, 470844, 472214, 472523, 472061, 472030, 471410, 471507, 472858,
                                   476143, 486121, 489225, 486468, 482263, 478699, 487815, 497133, 509344, 512434,
                                   522717, 544826, 562905, 562495, 553240, 555856, 564198, 564917, 556492, 541208,
                                   525671, 514605, 495378, 461561, 424153, 402854, 403134, 402111, 404147, 400673,
                                   401065, 403070, 396698, 398623, 396657, 397181, 394883, 395207, 394946, 392446,
                                   389395, 387282, 390170, 388304, 389743, 386621, 382795, 381773, 383691, 382188,
                                   376917, 379686, 378402, 379012, 373939, 376086, 380985, 388606, 390211, 393621,
                                   414970, 443761, 463310, 465440, 452816, 454306, 461810, 467816, 470141, 459365,
                                   451148, 442374, 434480, 402889, 364779, 353583, 358213, 355153, 341431, 334162,
                                   328670, 328709, 324541, 321846, 317608, 315979, 312642, 309174, 307054, 300600,
                                   299452, 295918, 295151, 289072, 282992, 279918, 275416, 272764, 268366, 268855,
                                   264701, 261930, 260938, 260418, 257476, 256779, 265033, 271019, 276184, 276235,
                                   286734, 300901, 314773, 309571, 309265, 310233, 317127, 326175, 319614, 318596,
                                   316229, 311031, 302046, 273612, 248909, 236143, 233707, 230021, 223559, 222571,
                                   221911, 224098, 221258, 221052, 221941, 223110, 222369, 220551, 213810, 207055,
                                   184366, 182059, 182656, 181529, 180631, 179082, 179881, 177987, 176422, 177562,
                                   175653, 173335, 173246, 171247, 171860, 172172, 174144, 177698, 178925, 180430,
                                   183724, 188353, 190485, 186900, 185941, 187083, 187591, 188037, 185679, 182466,
                                   173851, 165604, 159467, 150686, 142028, 137444, 138053, 139047, 138806, 141959,
                                   139537, 141096, 140410, 140371, 137283, 134779, 132918, 130988, 129999, 126713,
                                   126221, 122385, 123613, 122767, 123305, 123290, 121944, 131538, 126693, 123228,
                                   119917, 118425, 116271, 114670, 117688, 117271, 120443, 120351, 124310, 128089,
                                   134592, 138347, 142115, 140779, 140815, 144505, 146078, 147347, 145520, 137282,
                                   129534, 122080, 115007, 106154, 97886, 94421, 94549, 94918, 92353, 93494, 92458,
                                   91658, 90482, 90480, 90752, 91398, 90260, 91073, 90618, 89949, 84698, 84904, 84485,
                                   83301, 79901, 79647, 77832, 76557, 74923, 75808, 75384, 73672, 74206, 72904, 71879,
                                   70993, 72901, 71275, 71980, 71121, 71191, 72573, 72081, 74285, 72105, 70771, 70913,
                                   70382, 67968, 62205, 58451, 56489, 54524, 49800, 46462, 44237, 44029, 44049, 43651,
                                   45007, 44672, 45482, 47038, 47356, 45593, 46442, 46318, 45103, 45698, 43867, 45419,
                                   45368, 46018, 44501, 44212, 44648, 43434, 43482, 43496, 43368, 42490, 41392, 41773,
                                   42190, 41488, 39075, 38786, 39213, 39191, 37233, 34853, 33064, 30154, 28448, 27289,
                                   25810, 23157, 22341, 21056, 18164, 15700, 14128, 13210, 11470, 9955, 9308, 8862, 8477,
                                   8220, 8115, 7398, 6974, 6501, 5930, 5527, 4303, 2971, 2024, 1172, 322])
_usnob1_stars_per_file_cumsum = _usnob1_stars_per_file.cumsum()


"""
Needed to determine maximum absolute RA and DEC proper motions contained in the catalog in order to then be able
to calculate the maximum over-search distance needed for any given field (as a function of epoch).
Run as a one-time piece of code:
    import nomad
    max_RA_pm = 0
    max_DEC_pm = 0
    for curfilenum in range(1800):
        print curfilenum, max_RA_pm, max_DEC_pm
        df = nomad._convert_raw_byte_data_to_dataframe(open(nomad._nomad_dir + '/' +
                                                            ('%04i' % curfilenum)[0:3] + '/m' +
                                                            ('%04i' % curfilenum) + '.cat', 'rb').read(-1))
        max_RA_pm = max(max_RA_pm, max(abs(df['proper motion of RA*COS(dec) in integer 0.0001 arcsec/year'])))
        max_DEC_pm = max(max_DEC_pm, max(abs(df['proper motion of SPD in integer 0.0001 arcsec/year'])))
    print "_max_pm_RAcosDEC_arcsecPerYear = " + str(max_RA_pm * 0.0001)
    print "_max_pm_DEC_arcsecPerYear = " + str(max_DEC_pm * 0.0001)
"""
_max_pm_RAcosDEC_arcsecPerYear = 9.894
_max_pm_DEC_arcsecPerYear = 10.3269


def _determine_usnob1_id_from_usnob1_integer(usnob1_integer):
    if usnob1_integer == 0:
        return ''
    else:
        file_number = (_usnob1_stars_per_file_cumsum > usnob1_integer).argmax()
        star_number = usnob1_integer - _usnob1_stars_per_file_cumsum[file_number - 1]
        return ('%04i' % file_number) + '-' + ('%07i' % star_number)


def _process_epoch(epoch):
    """
    Input - datetime.datetime or datetime.date or integer year or float year

    Output - decimal year
    """
    if type(epoch) == datetime.datetime:
        return 2000.0 + (epoch - datetime.datetime(2000, 1, 1, 0, 0)).total_seconds() / (365.25 * 24. * 3600.)
    elif type(epoch) == datetime.date:
        return 2000.0 + (datetime.datetime(epoch.year, epoch.month, epoch.day, 0, 0) -
                         datetime.datetime(2000, 1, 1, 0, 0)).total_seconds() / (365.25 * 24. * 3600.)
    else:
        try:
            return float(epoch)
        except ValueError:
            raise Error('Unable to convert input epoch to floating point.')


def _read_accelerator_file(file_number):
    nomad_filenum_str = '%04i' % file_number
    if os.path.isfile(_nomad_dir + nomad_filenum_str[0:3] + '/m' + nomad_filenum_str + '.acc'):
        df = read_csv(open(_nomad_dir + nomad_filenum_str[0:3] + '/m' + nomad_filenum_str + '.acc', 'r'),
                      sep='\s+', header=None, names=['ra_band', 'start_record', 'num_records'],
                      index_col='ra_band')
    elif os.path.isfile(_nomad_dir + nomad_filenum_str[0:3] + '/m' + nomad_filenum_str + '.acc.gz'):
        df = read_csv(gzip.open(_nomad_dir + nomad_filenum_str[0:3] + '/m' + nomad_filenum_str + '.acc.gz', 'r'),
                      sep='\s+', header=None, names=['ra_band', 'start_record', 'num_records'],
                      index_col='ra_band')
    else:
        raise Error("Could not find accelerator file number " + str(file_number))
    return df


def _determine_record_numbers_to_retrieve(min_ra_swatch, max_ra_swatch, dec_filenum):
    acc = _read_accelerator_file(dec_filenum)
    if min_ra_swatch <= max_ra_swatch:
        return [(acc.ix[min_ra_swatch]['start_record'],
                 acc.ix[max_ra_swatch]['start_record'] + acc.ix[max_ra_swatch]['num_records'] - 1)]
    else:
        return [(1, acc.ix[max_ra_swatch]['start_record'] + acc.ix[max_ra_swatch]['num_records'] - 1),
                (acc.ix[min_ra_swatch]['start_record'],
                 acc.ix[23.75]['start_record'] + acc.ix[23.75]['num_records'] - 1)]


def _convert_raw_byte_data_to_dataframe(raw_byte_data, nomad_ids=None):
    """
    Input is a byte string of one or more raw NOMAD catalog entries.
    if available, nomad_star_ids will be used to set the index of the returned DataFrame.
        (otherwise index is just integers)

    Output is a pandas.DataFrame
    """
    # Each record contains 22 integers (4 byte).  The schema is:
    schema = """   ( 1)   RA at 2000.0 in integer 0.001 arcsec
       ( 2)   SPD at 2000.0 in integer 0.001 arcsec
       ( 3)   std. dev. of RA*COS(dec) in integer 0.001 arcsec at central epoch
       ( 4)   std. dev. of SPD in integer 0.001 arcsec at central epoch
       ( 5)   proper motion of RA*COS(dec) in integer 0.0001 arcsec/year
       ( 6)   proper motion of SPD in integer 0.0001 arcsec/year
       ( 7)   std. dev. of (5) in integer 0.0001 arcsec/year
       ( 8)   std. dev. of (6) in integer 0.0001 arcsec/year
       ( 9)   central epoch of RA in integer 0.001 year
       (10)   central epoch of SPD in integer 0.001 year
       (11)   B magnitude in integer 0.001 mag
       (12)   V magnitude in integer 0.001 mag
       (13)   R magnitude in integer 0.001 mag
       (14)   J magnitude in integer 0.001 mag
       (15)   H magnitude in integer 0.001 mag
       (16)   K magnitude in integer 0.001 mag
       (17)   USNO-B1.0 ID integer
       (18)   2MASS ID integer
       (19)   YB6 ID integer
       (20)   UCAC-2 ID integer
       (21)   Tycho2 ID integer
       (22)   flags integer"""
    dtype = np.dtype([(a[a.find(')') + 1:].strip(), '<i4') for a in schema.splitlines()])
    df = DataFrame(np.fromstring(raw_byte_data, dtype=dtype))
    if nomad_ids is not None:
        if type(nomad_ids) == str:
            df.index = [nomad_ids]
        else:
            df.index = nomad_ids
    df['RAJ2000_epoch2000'] = df['RA at 2000.0 in integer 0.001 arcsec'] * 0.001 / 3600.
    df['DEJ2000_epoch2000'] = -90. + df['SPD at 2000.0 in integer 0.001 arcsec'] * 0.001 / 3600.
    df['Bmag'] = df['B magnitude in integer 0.001 mag'] * 0.001
    df['Vmag'] = df['V magnitude in integer 0.001 mag'] * 0.001
    df['Rmag'] = df['R magnitude in integer 0.001 mag'] * 0.001
    df['Jmag'] = df['J magnitude in integer 0.001 mag'] * 0.001
    df['Hmag'] = df['H magnitude in integer 0.001 mag'] * 0.001
    df['Kmag'] = df['K magnitude in integer 0.001 mag'] * 0.001
    for cur_band in ['Bmag', 'Vmag', 'Rmag', 'Jmag', 'Hmag', 'Kmag']:
        df[cur_band][df[cur_band] == 30.0] = np.NaN
    df['USNO-B1.0'] = [_determine_usnob1_id_from_usnob1_integer(a) for a in df['USNO-B1.0 ID integer']]
    # df['2MASS'] = df['2MASS ID integer']  # TODO: someday add conversion to 2MASS ID
    # df['YB6'] = df['YB6 ID integer']  # TODO: someday add conversion to YB6 ID
    # df['UCAC-2'] = df['UCAC-2 ID integer']  # TODO: someday add conversion to UCAC-2 ID
    # df['flags'] = df['flags integer']  # TODO:  decode flags integer into something useful
    columns_to_drop = list(dtype.names)
    # We want to save some of the other original columns, though they will be deleted later after they
    # have been used in _apply_proper_motion
    for column_name in ['std. dev. of RA*COS(dec) in integer 0.001 arcsec at central epoch',
                        'std. dev. of SPD in integer 0.001 arcsec at central epoch',
                        'proper motion of RA*COS(dec) in integer 0.0001 arcsec/year',
                        'proper motion of SPD in integer 0.0001 arcsec/year',
                        'std. dev. of (5) in integer 0.0001 arcsec/year',
                        'std. dev. of (6) in integer 0.0001 arcsec/year',
                        'central epoch of RA in integer 0.001 year',
                        'central epoch of SPD in integer 0.001 year']:
        columns_to_drop.pop(columns_to_drop.index(column_name))
    return df.drop(columns_to_drop, axis=1)


def _apply_proper_motion(df, epoch=2000.0):
    """
    Apply proper motion for input epoch.
    Also calculate coordinate uncertainty at the epoch.
    Drop the original columns.

    Note: We encourage the user to NOT later re-calculate proper motions for a different epoch.  The better
    way is to reload the search for the new epoch.  The risk is the small edge case of a star that at epoch 1 is
    not in the requested field but is in the field at epoch 2.
    """
    epoch = _process_epoch(epoch)
    years_since_2000 = epoch - 2000.0
    cosDec = np.cos(np.radians(df['DEJ2000_epoch2000']))
    pm_RA = df['proper motion of RA*COS(dec) in integer 0.0001 arcsec/year'] * (0.0001 / 3600.) / cosDec
    df['RAJ2000'] = (df['RAJ2000_epoch2000'] + pm_RA * years_since_2000) % 360.0
    pm_DEC = df['proper motion of SPD in integer 0.0001 arcsec/year'] * (0.0001 / 3600.)
    df['DEJ2000'] = df['DEJ2000_epoch2000'] + pm_DEC * years_since_2000
    df['epoch'] = epoch
    years_since_central_epoch_RA = epoch - df['central epoch of RA in integer 0.001 year'] * 0.001
    base_err_RA = df['std. dev. of RA*COS(dec) in integer 0.001 arcsec at central epoch'] * (0.001 / 3600.) / cosDec
    pm_err_RA = (years_since_central_epoch_RA *
                 df['std. dev. of (5) in integer 0.0001 arcsec/year'] * (0.0001 / 3600.) / cosDec)
    df['errRAJ2000'] = np.sqrt((base_err_RA) ** 2 + (pm_err_RA) ** 2)
    years_since_central_epoch_DEC = epoch - df['central epoch of SPD in integer 0.001 year'] * 0.001
    base_err_DEC = df['std. dev. of SPD in integer 0.001 arcsec at central epoch'] * (0.001 / 3600.)
    pm_err_DEC = (years_since_central_epoch_DEC *
                  df['std. dev. of (6) in integer 0.0001 arcsec/year'] * (0.0001 / 3600.))
    df['errDEJ2000'] = np.sqrt((base_err_DEC) ** 2 + (pm_err_DEC) ** 2)
    columns_to_drop = ['RAJ2000_epoch2000', 'DEJ2000_epoch2000',
                       'std. dev. of RA*COS(dec) in integer 0.001 arcsec at central epoch',
                       'std. dev. of SPD in integer 0.001 arcsec at central epoch',
                       'proper motion of RA*COS(dec) in integer 0.0001 arcsec/year',
                       'proper motion of SPD in integer 0.0001 arcsec/year',
                       'std. dev. of (5) in integer 0.0001 arcsec/year',
                       'std. dev. of (6) in integer 0.0001 arcsec/year',
                       'central epoch of RA in integer 0.001 year',
                       'central epoch of SPD in integer 0.001 year']
    return df.drop(columns_to_drop, axis=1)


def fetch_star_by_nomad_id(nomad_ids, epoch=None):
    """
    nomad_ids - can be either a single NOMAD identifier, e.g.:
                    '0999-0192017'
                or an iterable object (e.g. list, etc) of NOMAD identifiers, e.g.:
                    ['0999-0192017', '0999-0192019']
    """
    if type(nomad_ids) == str:
        nomad_ids = [nomad_ids]
    try:
        file_numbers = np.array([np.int(a.split('-')[0]) for a in nomad_ids], dtype=np.int)
        star_numbers = np.array([np.int(a.split('-')[1]) for a in nomad_ids], dtype=np.int)
    except:
        raise Error('Unable to parse input into NOMAD file numbers and star numbers')
    nstars = len(file_numbers)
    raw_byte_data = [None] * nstars
    for i in range(nstars):
        if raw_byte_data[i] is None:
            nomad_filenum_str = '%04i' % file_numbers[i]
            if os.path.isfile(_nomad_dir + nomad_filenum_str[0:3] + '/m' + nomad_filenum_str + '.cat'):
                f = open(_nomad_dir + nomad_filenum_str[0:3] + '/m' + nomad_filenum_str + '.cat', 'rb')
            elif os.path.isfile(_nomad_dir + nomad_filenum_str[0:3] + '/m' + nomad_filenum_str + '.cat.gz'):
                f = gzip.open(_nomad_dir + nomad_filenum_str[0:3] + '/m' + nomad_filenum_str + '.cat.gz', 'rb')
            else:
                raise Error("Could not find nomad file number " + nomad_filenum_str)
            for j in np.where(file_numbers == file_numbers[i])[0]:
                f.seek(_nomad_record_length_bytes * (star_numbers[j] - 1))
                raw_byte_data[j] = f.read(_nomad_record_length_bytes)
            f.close()
    df = _convert_raw_byte_data_to_dataframe(''.join(raw_byte_data), nomad_ids=nomad_ids)
    if epoch is None:
        return _apply_proper_motion(df, epoch=2000.0)
    else:
        return _apply_proper_motion(df, epoch=epoch)


def fetch_nomad_box(ra_range, dec_range, epoch=2000.0):
    """
    ra_range - [>=low, <high] RA in degrees
               can wrap around 360, e.g. [359.5, 0.5]
    dec_range - [>=low, <high] DEC in degrees
                order of dec_range is irrelevant as search area is >=min(dec_range) to <max(dec_range)
    epoch - default of 2000.0, can also be a datetime.date or datetime.datetime

    Note that search is over a larger box than requested, then proper motions are applied, then
    requested ra_range/dec_range limits are applied.  Size of over-search is dictated by
    the largest proper motion in the NOMAD catalog.
    """
    # DEC swatches are 0.1 degree
    #     stars within a DEC swatch are >=minDec and <(minDec + 0.1deg)
    # RA swatches (within accelerator files) are 0.25 degree
    #     stars within a RA range are >=minRA and <(minRA + 0.25deg)
    epoch = _process_epoch(epoch)
    years_since_2000 = epoch - 2000.0
    dec_oversearch = np.abs((years_since_2000 * _max_pm_DEC_arcsecPerYear) / 3600.)
    min_dec = max(-90.0, min(dec_range) - dec_oversearch)
    max_dec = min(90.0, max(dec_range) + dec_oversearch)
    min_dec_filenum = int((min_dec + 90) * 10.)
    max_dec_filenum = min(1799, int((max_dec + 90) * 10.))
    # TODO:  fix that the next two lines break when max_dec == 90 or min_dec == -90 because of divide-by-zero
    min_cosDec = min(np.cos(np.radians(min_dec)), np.cos(np.radians(max_dec)))
    ra_oversearch = np.abs((years_since_2000 * _max_pm_RAcosDEC_arcsecPerYear) / (3600. * min_cosDec))
    min_ra = ((ra_range[0] - ra_oversearch) % 360.) / 15.
    max_ra = ((ra_range[1] + ra_oversearch) % 360.) / 15.
    min_ra_swatch = int(min_ra / 0.25) * 0.25
    max_ra_swatch = int(max_ra / 0.25) * 0.25
    raw_byte_data = ''
    nomad_ids = []
    for cur_dec_filenum in np.arange(min_dec_filenum, max_dec_filenum + 1):
        records_to_retrieve = _determine_record_numbers_to_retrieve(min_ra_swatch, max_ra_swatch, cur_dec_filenum)
        nomad_filenum_str = '%04i' % cur_dec_filenum
        if os.path.isfile(_nomad_dir + nomad_filenum_str[0:3] + '/m' + nomad_filenum_str + '.cat'):
            f = open(_nomad_dir + nomad_filenum_str[0:3] + '/m' + nomad_filenum_str + '.cat', 'rb')
        elif os.path.isfile(_nomad_dir + nomad_filenum_str[0:3] + '/m' + nomad_filenum_str + '.cat.gz'):
            f = gzip.open(_nomad_dir + nomad_filenum_str[0:3] + '/m' + nomad_filenum_str + '.cat.gz', 'rb')
        else:
            raise Error("Could not find nomad file number " + nomad_filenum_str)
        for cur_rec in records_to_retrieve:
            f.seek((cur_rec[0] - 1) * _nomad_record_length_bytes)
            raw_byte_data += f.read((cur_rec[1] - cur_rec[0] + 1) * _nomad_record_length_bytes)
            nomad_ids.extend([nomad_filenum_str + '-' + ('%07i' % a) for a in range(cur_rec[0], cur_rec[1] + 1)])
    stars = _apply_proper_motion(_convert_raw_byte_data_to_dataframe(raw_byte_data, nomad_ids=nomad_ids), epoch=epoch)
    stars = stars[(stars['DEJ2000'] >= min(dec_range)) & (stars['DEJ2000'] < max(dec_range))]
    if min_ra <= max_ra:
        stars = stars[(stars['RAJ2000'] >= ra_range[0]) & (stars['RAJ2000'] < ra_range[1])]
    else:
        stars = stars[(stars['RAJ2000'] < ra_range[1]) | (stars['RAJ2000'] >= ra_range[0])]
    return stars


if __name__ == '__main__':
    if len(sys.argv) == 1:
        pass
    else:
        if sys.argv[1] == 'help':
            pass  # TODO: write instructions on options
        elif sys.argv[1] == 'radec_range':
            try:
                ra_range = [float(sys.argv[2]), float(sys.argv[3])]
                dec_range = [float(sys.argv[4]), float(sys.argv[5])]
                epoch = float(sys.argv[6])
            except:
                raise Error("Expected 5 numbers after radec_range:  \n\t" +
                            "RA_low_deg  RA_high_deg  DEC_low_deg  DEC_high_deg  Epoch")
            stars = fetch_nomad_box(ra_range, dec_range, epoch)
            if 'pickle' in sys.argv:
                pickle.dump(stars, sys.stdout)
            else:
                sys.stdout.write(stars.to_string() + '\n')
        else:
            raise Error("unrecognized input on command line: \n\t" + ' '.join(sys.argv))
