from pdfobjects.pdfobject import _PDFObject
import os


class _Session(object):

    """ Manages the storing of the
        text output for the PDF objects
        as they are generated by the pdfobject
        classes. Has a centralized count of the
        objects, and some helper functions (putStream,
        compression, etc.)

        One and only one session object should be
        instantiated per PDF file. The object should be
        passed by-reference to objects that will need
        to generate output (either PDF objects or to streams).

    """

    def __init__(self, parent):
        self.parent = parent

        # Central buffer for storing PDF code not in pages.
        self.buffer = ''

        # Offest is used to calculate the binary lengths in the cross-reference
        # section
        self.offset = 0

        self.objects = []

        # Creates the placeholders in the object list to maintain proper count.
        self._create_placeholder_objects()
        self.compression = False
        self.drawn_color = None
        self.color = None
        self.project_dir = os.path.dirname(__file__)

        if '/modules' in self.project_dir:  # pypdflite is a submodule
            self.project_dir, other = self.project_dir.split('/modules')

    # Compression
    def _set_compression(self, value):
        """ May be used to compress PDF files. Code is more readable
            for testing and inspection if not compressed. Requires a
            boolean.

        """
        if isinstance(value, bool):
            self.compression = value
        else:
            raise Exception(
                TypeError, "%s is not a valid option for compression" % value)

    # Objects
    def _create_placeholder_objects(self):
        """ PDF objects #1 through #3 are typically saved for the
            Zeroth, Catalog, and Pages Objects. This program will save
            the numbers, but outputs the individual Page and Content objects
            first. The actual Catalog and Pages objects are calculated after.

        """
        self.objects.append("Zeroth")
        self.objects.append("Catalog")
        self.objects.append("Pages")

    def _add_object(self, flag=None):
        """ The flag is a simple integer to force the placement
            of the object into position in the object array.
            Used for overwriting the placeholder objects.

        """
        self.offset += len(self.buffer)
        if flag is None:
            objnum = len(self.objects)
            obj = _PDFObject(objnum, self.offset)
            self.objects.append(obj)
        else:
            objnum = flag
            obj = _PDFObject(objnum, self.offset)
            self.objects[flag] = obj
        self._out(str(objnum) + ' 0 obj')
        return obj

    def _save_object_number(self):
        self.saved = len(self.objects)

    def _get_saved_number(self):
        return self.saved

    # Inputs
    def _out(self, stream, page=None):
        """ Stores the pdf code in a buffer. If it is page related,
            provide the page object.

        """
        if page is not None:
            page.buffer += str(stream) + "\n"
        else:
            self.buffer += str(stream) + "\n"

    def _put_stream(self, stream):
        """ Creates a PDF text stream sandwich.

        """
        self._out('stream')
        self._out(stream)
        self._out('endstream')

    def _add_page(self, text):
        """ Helper function for PDFText, to have the document
            add a page, and retry adding a large block of
            text that would otherwise have been to long for the
            page.

        """
        self.parent.document.add_page()
        self.parent.document.add_text(text)

    # Strings
    @staticmethod
    def _UTF8toUTF16(utf8, setbom=True):
        result = ''
        if setbom:
            result = '\xFE\xFF'
        if not isinstance(utf8, unicode):
            utf8 = utf8.decode('UTF-8')
        result += utf8.encode('UTF-16BE')
        return result

    # Colors
    def _save_color(self, color):
        if color.color_type == 'd':
            self.drawn_color = color
        else:
            self.color = color

    def _compare_color(self, newcolor):
        if newcolor._is_equal(self.drawn_color):
            return True
        elif newcolor._is_equal(self.color):
            return True
        else:
            return False

    def _reset_colors(self):
        self.drawn_color = None
        self.color = None
