# coding: UTF-8
#
# Copyright © 2012, Elizabeth J. Myers, et al. All rights reserved.
# License terms can be found in the LICENSE file at the top level of the source
# tree.

from __future__ import unicode_literals

import bisect
import os, platform
import socket

from PyProto.eventloop.Events import (EVENT_IN, EVENT_OUT, EVENT_EXCEPT,
                                      FDEvent, SocketEvent)
from PyProto.eventloop import Exceptions
from PyProto.utils import Clock

# Import eventloops
try: from PyProto.eventloop.engines import epoll
except: from PyProto.eventloop.engines import nullengine as epoll
try: from PyProto.eventloop.engines import poll
except: from PyProto.eventloop.engines import nullengine as poll
try: from PyProto.eventloop.engines import devpoll
except: from PyProto.eventloop.engines import nullengine as devpoll
try: from PyProto.eventloop.engines import _select as select
except: from PyProto.eventloop.engines import nullengine as select

class UID(object): 
    def __init__(self): 
        self.uidcount = 0 
        self.uidholes = [] 
 
    def next(self): 
        if not self.uidholes: 
            self.uidcount += 1 
            return self.uidcount 
        else: 
            return self.uidholes.pop() 
 
    def free(self, uid): 
        if uid == self.uidcount: 
            self.uidcount -= 1 
            return 
        self.uidholes.append(uid)

class EventLoop:
    """Base eventloop class.

    This implements the eventloop of PyProto, selecting a given module for
    interfacing with the operating system's FD notification system.

    It also implements timers, e.g. events that fire at a given time.

    Errors in this class raise EventLoopException or another appropriate
    exception.
    """

    def __init__(self, evlooptype=None):
        """Initalise the eventloop.

        evlooptype should be a string with the FD notification system you want
        to use. Supported types are select (all platforms), epoll (Linux), and
        poll (windows, BSD, etc.)
        """
        # Give this a shot...
        if hasattr(evlooptype, 'EventEngine'):
            try: self.engine = evlooptype.EventEngine()
            except: self.engine = self.select_evloop(evlooptype)
        else:
            self.engine = self.select_evloop()

        # FD:Event map
        self.fdmap = dict()

        self.timerlist = list()
        self.uid = UID()

        self.evloop_deadline = None

    @staticmethod
    def select_evloop(evloop=None):
        if evloop == '' or evloop is None:
            evloop = EventLoop.best_evloop()

        if evloop == 'select':
            return select.SelectEventEngine()
        elif evloop == 'poll':
            return poll.PollEventEngine()
        elif evloop == 'devpoll':
            return devpoll.DevPollEventEngine()
        elif evloop == 'epoll':
            return epoll.EpollEventEngine()
        else:
            raise EventLoopException("Unsupported eventloop type")

    @staticmethod
    def best_evloop():
        system = platform.system()
        if system.startswith('Linux'):
            return 'epoll'
        elif system.startswith('SunOS') or system.startswith('Solaris'):
            return 'devpoll'
        # TODO: kqueue backend, but the kqueue python module sux
        elif system.endswith('BSD'):
            return 'poll'
        else:
            return 'select'

    def set_event(self, event, evtype, removemask=False):
        """Set an FDEvent/SocketEvent to be used for evtype. When the event
        happens the appropriate callback will be called in FDEvent/SocketEvent.

        Set removemask to remove the mask from the event.
        """
        if hasattr(event, 'readfd') and hasattr(event, 'writefd'):
            readfd = event.readfd
            writefd = event.writefd
            if readfd == writefd:
                exceptfd = readfd
            else:
                exceptfd = None
        elif hasattr(event.sock, 'fileno'):
            readfd = writefd = exceptfd = event.sock.fileno()
        else:
            raise ValueError("Unsupported event type")

        if evtype & EVENT_IN:
            if readfd is None:
                raise EventLoopException("Cannot set an event with EVENT_IN with no readfd!")
            self.set_fd(readfd, EVENT_IN, event, removemask)

        if evtype & EVENT_OUT:
            if writefd is None:
                raise EventLoopException("Cannot set an event with EVENT_OUT and no writefd!")
            self.set_fd(writefd, EVENT_OUT, event, removemask)

        if evtype & EVENT_EXCEPT:
            if exceptfd is None:
                raise EventLoopException("Ambiguous file descriptor for EVENT_EXCEPT, use EventLoop.set_fd instead")
            self.set_fd(exceptfd, EVENT_EXCEPT, event, removemask)

    def remove_event(self, event):
        """Remove event by FDEvent."""
        if isinstance(event, FDEvent):
            self.remove_fd(event.readfd)
            self.remove_fd(event.writefd)
        elif hasattr(event, 'fileno'):
            self.remove_fd(event.sock.fileno())
        else:
            raise ValueError("Unsupported event type")

    def set_fd(self, fd, evtype, event=None, removemask=False):
        """Set event by fd, specifying the fdevent callback.

        If an fd has already been added to the event pool, event may be
        omitted. Otherwise event is mandatory. removemask specifies the
        given eventmask in evtype should be removed
        """
        if hasattr(fd, 'fileno'):
            fd = fd.fileno()

        if fd not in self.fdmap:
            if event is None:
                raise EventLoopException("Trying to add an empty event!")
            self.fdmap[fd] = [event, evtype]
            event.eventmask = evtype
            self.engine.register_fd(fd, self.fdmap[fd][1])
        else:
            if removemask:
                self.fdmap[fd][1] &= ~evtype
            else:
                self.fdmap[fd][1] |= evtype
            self.fdmap[fd][0].eventmask = self.fdmap[fd][1]
            self.engine.modify_fd(fd, self.fdmap[fd][1])

    def remove_fd(self, fd):
        """Remove an fd from the set"""
        if hasattr(fd, 'fileno'):
            fd = fd.fileno()

        if fd not in self.fdmap:
            return

        del self.fdmap[fd]
        self.engine.unregister_fd(fd)

    def set_timer(self, when, timerev):
        """Create a timer to go off every when seconds

        timerev is the timer callback.
        Set recur to true for a recurring event, false for oneshot
        """
        timerev.last_ran = None
        uid = self.uid.next()
        bisect.insort_right(self.timerlist, (when, uid, timerev))
        return uid

    def remove_timer(self, when=None, uid=None, timerev=None):
        """Remove a timer matching the given criteria"""
        if (when, uid, timerev) != (None, None, None):
            self.timerlist.remove((when, uid, timerev))
            self.uid.free(uid)
        elif (when, uid, timerev) == (None, None, None):
            raise ValueError("No specifiers for removal")

        for item_when, item_uid, item_timerev in list(self.timerlist):
            if when is not None and when != item_when:
                continue

            if uid is not None and uid != item_uid:
                continue

            if timerev is not None and timerev != item_timerev:
                continue

            self.uid.free(item.uid)
            self.timerlist.remove(item_when, item_uid, item_timerev)

    def process_timer(self):
        """Process each timer and return the time needed to sleep
        until the next event, and the next event(s) to run.

        Returns a tuple of (sleeptime, [events, ...])

        Do not call this directly.
        """
        if len(self.timerlist) == 0:
            return (None, [])

        curtime = round(Clock.monotonic())
        next_time = None
        next_events = []
        late = False
        for when, uid, timerev in list(self.timerlist):
            # If not run, pretend we just ran it for timing purposes
            if timerev.last_ran is None:
                timerev.last_ran = curtime
                continue

            evinterval = curtime - timerev.last_ran # seconds since last ran
            adj = when - evinterval # Seconds until next firing
            if adj >= 0:
                # Adjust returned sleep value if we must...
                if not late:
                    if next_time is None or adj < next_time:
                        next_time = adj
                        next_events = [(when, uid, timerev)]
                        continue
                    elif adj == next_time:
                        next_events.append((when, uid, timerev))

            # Late events
            else:
                if not late:
                    late = True
                    next_events = list()
                    next_time = 0
                next_events.append((when, uid, timerev))

        return (next_time, next_events)

    def run_once(self):
        """Run the eventloop once, firing timers also."""
        timeout, timerevents = self.process_timer()
        start_time = Clock.monotonic()
        while timeout is None or timeout > 0:
            fdevents = self.engine.run_once(timeout)
            for fd, event in fdevents:
                if event & EVENT_IN != 0:
                    if self.fdmap[fd][0].read_callback() == False:
                        self.set_fd(fd, EVENT_IN, removemask=True)
                if event & EVENT_OUT != 0:
                    if self.fdmap[fd][0].write_callback() == False:
                        self.set_fd(fd, EVENT_OUT, removemask=True)
                if event & EVENT_EXCEPT != 0:
                    if self.fdmap[fd][0].except_callback() == False:
                        self.set_fd(fd, EVENT_EXCEPT, removemask=True)

            if timeout is None:
                return # Nothing left to do

            timeout -= (Clock.monotonic() - start_time)

        for when, uid, timerev in timerevents:
            timerev.last_ran = Clock.monotonic()
            if timerev.run_timer() == False:
                self.timerlist.remove((when, uid, timerev))

    def run_forever(self):
        """Run the eventloop forever."""
        while True:
            self.run_once()

    def run_until(self, ticks):
        """Run for a given number of iterations"""
        for x in range(1, ticks):
            self.run_once()

