# coding: UTF-8
#
# Copyright © 2012, Elizabeth J. Myers, et al. All rights reserved.
# License terms can be found in the LICENSE file at the top level of the source
# tree.

from __future__ import print_function, unicode_literals

from PyProto.eventloop import EventLoop
from PyProto.splitter import LineSplitter

import copy

# Non-exhaustive TODO:
# - MIME handling
# - Chunked encoding
# - GZip/BZip encoding
# - Basic cookie support (maybe?)
# - Pipelining (maybe?)

class HTTPClient(LineSplitter.LineSplitter):
    default_headers = {
            'User-Agent' : 'PyProto',
            'X-Animals' : 'Cats rule.',
    }

    def __init__(self, event, **kwargs):
        self.client_headers = kwargs.get('client_headers', self.default_headers)
        self.server_headers = {}

        self.connected = False
        self.got_headers = False
        self.read_data = False
        self.pending = False

        self.remote_vers = None
        self.reset_request()

        super(HTTPClient, self).__init__(event, **kwargs)

    def reset_request(self):
        self.reply_req = None
        self.request = b''
        self.request_response = None
        self.headers_buf = list() # FIXME: save certain headers

    def connect(self, hostport=tuple()):
        super(HTTPClient, self).connect(hostport)

        if self.host.find(':') != -1:
            host = ''.join(['[', self.host, ']'])
        else:
            host = self.host
        self.client_headers['Host'] = host

    def headers_to_str(self, headers, format_wire=True):
        l = list()
        for k, v in headers.items():
            l += [k, ': ', v, '\r\n']

        if format_wire: l.append('\r\n')
        return ''.join(l)

    def str_to_headers(self, string):
        headers = {}

        for v in string.split('\r\n'):
            split = v.partition(':')
            headers[split[0]] = split[2][1:]

        return headers

    def send_request(self, request, res, body=None, headers={}):
        if not self.connected:
            vers = 'HTTP/1.1'
            self.connected = True # version exchange
        else:
            vers = ''

        send = [request, ' ', res, ' ', vers, '\r\n']

        sendheaders = copy.copy(self.client_headers)
        sendheaders.update(headers)

        if isinstance(body, str):
            if 'Content-Encoding' not in sendheaders:
                body = body.encode("iso-8859-1")
            else:
                body = body.encode(sendheaders['Content-Encoding'])

        if body is not None:
            sendheaders['Content-Length'] = str(len(body))
            body += '\r\n'
        else:
            body = ''

        send += [self.headers_to_str(sendheaders), body]

        self.write_line(''.join(send), raw=True)

    def read_callback(self):
        if self.read_data:
            # Bypass the line splitter :P
            super(LineSplitter.LineSplitter, self).read_callback()
            # We've reached Content-Length
            if len(self.read_buffer) == int(self.server_headers['Content-Length']):
                self.read_data = False
                self.request = self.read_buffer
                try:
                    self.read_request(self.server_headers, self.request_response, self.request)
                finally:
                    self.reset_request()
                    self.read_buffer = b''
        else:
            super(HTTPClient, self).read_callback()

    def read_request(self, headers, response, body):
        if 'Content-Encoding' in headers:
            encoding = headers['Content-Encoding']
        else:
            encoding = 'iso-8859-1' 

        print("Headers", headers)
        print("Response", response[0], '({})'.format(response[1]))
        print("Body", body.decode(encoding))

    def read_line(self, line):
        # Not blank
        if line != '':
            if not self.remote_vers:
                vers, sep, line = line.partition(' ')
                self.remote_vers = vers.partition('/')[2]
            if not self.request_response:
                req = line.partition(' ')
                self.request_response = (req[0], req[2])
                return
            if not self.got_headers:
                self.headers_buf.append(line)
        else:
            if not self.got_headers:
                self.got_headers = True
                self.server_headers = self.str_to_headers('\r\n'.join(self.headers_buf))
                if 'Content-Length' in self.server_headers: self.read_data = True # body follows

