# coding: UTF-8
#
# Copyright © 2012, Elizabeth J. Myers, et al. All rights reserved.
# License terms can be found in the LICENSE file at the top level of the source
# tree.

from __future__ import print_function, unicode_literals

from PyProto.eventloop import EventLoop
from PyProto.splitter import LineSplitter
import PyProto.protocol.IRC.numeric as numeric

from collections import Callable # Bleh Python 2.7 > ver < 3.2 do not have callable()
import os, socket

class IRCClient(LineSplitter.LineSplitter):
    def __init__(self, event, **kwargs):
        self.nick = kwargs.get('nick', 'PyProto')
        self.altnick = kwargs.get('nick', 'PyProto')
        self.user = kwargs.get('user', 'PyProto')
        self.irc_hostname = kwargs.get('irc_hostname', '*')
        self.irc_servername = kwargs.get('irc_servername', '*') # mode bitmask on IRCnet
        self.gecos = kwargs.get('gecos', 'PyProto based IRC client/bot')
        self.commands = kwargs.get('commands', dict())

        self.sent_handshake = False

        super(IRCClient, self).__init__(event, **kwargs)

    def parse_hostmask(self, host):
        first, sep, sec = host.partition('@')

        # If we got nothing, then check if it's a server or a nick
        if not sec:
            # More likely to find one faster going backwards...
            if first.rfind('.') == -1:
                # Nick
                return (first, None, None)
            else:
                # Server
                return (None, None, first)

        # look for !
        nick, sep, user = first.partition('!')
        if user == '':
            user = None

        host = sec
        return (nick, user, host)

    def parse(self, line):
        if line[0] == ':':
            recv, sep, line = line[1:].partition(' ')
            sender = self.parse_hostmask(recv)
        else:
            sender = (None, None, None)

        # Command/numeric
        command, sep, line = line.partition(' ')

        if line == '':
            return (sender, command, None)

        # Split out the final param
        params, sep, lastparam = line.partition(':')

        # Split up after stripping leading/trailing spaces + removing blank params
        params = params.split()
        if lastparam != '': params.append(lastparam)

        params = tuple(params)

        return (sender, command, params)

    def __dispatch_method(self, name, sender, command, params):
        attr = getattr(self, name, None)
        if isinstance(attr, Callable):
            attr(sender, command, params)
        else:
            return False

    def read_line(self, line):
        sender, command, params = self.parse(line)
        command = command.upper()

        if command in numeric.numeric_to_name:
            commandname = numeric.numeric_to_name[command]
        else:
            commandname = command

        self.__dispatch_method('command_' + command, sender, command, params)
        
        # This is not recommended, but not likely to cause issues.
        if command != commandname:
            self.__dispatch_method('command_' + commandname, sender, command, params)

        # Obtain command from dispatcher
        if command in self.commands:
            for self, cmd in self.commands[command]:
                if isinstance(cmd, Callable):
                    cmd(self, sender, command, params)
                else:
                    # Try this?
                    self.__dispatch_method('command_' + cmd, sender, command, params)

        # Create line for pretty printing
        if params[-1].find(' ') != -1:
            lastparam = ':{}'.format(params[-1])
        else:
            lastparam = params[-1]

        if sender[0] is None and sender[2] is not None:
            printfrom = sender[2]
        elif sender == (None, None, None):
            printfrom = '<SERVER>'
        else:
            printfrom = "{}!{}@{}".format(*sender)

        printline = "[{}] <{}> {} {}".format(commandname, printfrom, ' '.join(params[:-1]), lastparam)
        return super(IRCClient, self).read_line(printline)

    def command_PING(self, sender, command, params):
        if params:
            out = ' '.join(params)
        else:
            out = str()

        self.write_line("PONG :{}".format(out))

    def command_NOTICE(self, sender, command, params):
        if self.sent_handshake:
            return

        self.write_line("USER {} {} {} :{}".format(self.user, self.irc_hostname, self.irc_servername, self.gecos))
        self.write_line("NICK {}".format(self.nick))

        self.sent_handshake = True

