# coding: UTF-8
#
# Copyright © 2012, Elizabeth J. Myers, et al. All rights reserved.
# License terms can be found in the LICENSE file at the top level of the source
# tree.

from __future__ import unicode_literals

import os
from collections import deque

from PyProto.eventloop import EventLoop, Events
from PyProto.splitter import Exceptions
from PyProto.utils import Errors

class LineSplitter(Events.SocketEvent):
    """SocketEvent derivative for protocols delimited by line ending"""
    def __init__(self, event, line_end='\r\n', read_count=512, encoding='UTF-8', **kwargs):
        super(LineSplitter, self).__init__(event, **kwargs)

        self.line_end = line_end
        self.line_buf = deque()
        self.read_count = read_count
        self.encoding = encoding

    def read_callback(self):
        try: super(LineSplitter, self).read_callback()
        finally:
            lines = self.read_buffer.decode(self.encoding).split(self.line_end)
            self.read_buffer = lines[-1].encode(self.encoding)

            for line in lines[:-1]:
                self.read_line(line)

        return True

    def write_callback(self):
        while True:
            try:
                self.write_buffer += self.line_buf.popleft()
            except IndexError:
                return False # Drained

            res = super(LineSplitter, self).write_callback()

            if res:
                # Put back on line buffer.
                self.line_buf.append(self.write_buffer)
                return True

    def read_line(self, line):
        return line

    def write_line(self, line, raw=False):
        if not raw and not line.endswith(self.line_end):
            line += self.line_end

        self.line_buf.append(line.encode(self.encoding))
        self.event.set_event(self, EventLoop.EVENT_OUT)
        return line

