from collections import Sequence
import errno
import socket

from PyProto.eventloop import EventLoop, Exceptions
from PyProto.utils import Errors

class BaseEvent(object):
    def __init__(self, event):
        """ Initalise the BaseEvent """
        if event is None:
            raise Exceptions.EventException("Invalid event type")
        self.event = event

class FDEvent(BaseEvent):
    """Events that can happen on a file descriptor"""
    def __init__(self, event, fd):
        """Initalise the FDEvent

        fd should be a sequence pair of the format
        readfd = [0], writefd = [1], OR a socket, or
        a file number. If it is not a pair, read and write
        fd will be congruent.

        set event to your EventLoop for internal uses.
        """

        super(FDEvent, self).__init__(event)

        if isinstance(fd, Sequence):
            self.readfd, self.writefd = fd[0], fd[1]
        elif hasattr(fd, 'fileno'):
            self.readfd = self.writefd = fd.fileno()
        else:
            self.readfd = self.writefd = fd

        if self.readfd:
            self.event.set_event(self, EventLoop.EVENT_IN)
        if self.writefd:
            self.event.set_event(self, EventLoop.EVENT_OUT)

    def read_callback(self):
        """This is called when a read event fires in the eventloop.

        Override this in child classes to alter behaviour.
        """
        self.read_buffer = os.read(self.readfd, 512)
        return True

    def write_callback(self):
        """This is called when a write event fires in the eventloop.

        Override this in child classes to alter behaviour.
        """
        if not hasattr(self, 'write_buffer'):
            raise Exceptions.EventException("No write buffer and method not overridden in event")

        if hasattr(self.write_buffer, 'encode'):
            buf = self.write_buffer.encode('UTF-8')
        else:
            buf = self.write_buffer

        os.write(self.writefd, buf)
        return False

    def except_callback(self):
        """This does nothing yet."""
        pass

"""Like FDEvent, but for sockets"""
class SocketEvent(FDEvent):
    def __init__(self, event, **kwargs):
        self.ipv6 = kwargs.get('ipv6', True)
        sock = kwargs.get('fd', None)
        self.host = kwargs.get('host', None)
        self.port = kwargs.get('port', None)
        self.bindhost = kwargs.get('bindhost', None)
        self.bindport = kwargs.get('bindport', None)
        proto = kwargs.get('proto', socket.SOCK_STREAM)

        if self.ipv6:
            family = socket.AF_INET6
        else:
            family = socket.AF_INET

        if not sock:
            self.sock = socket.socket(family, proto)
            self.sock.setblocking(False)
        else:
            self.sock = None

        super(SocketEvent, self).__init__(event, self.sock)

    def connect(self, hostport=tuple()):
        if not isinstance(self.sock, socket.socket):
            raise Exceptions.EventException("Cannot connect using a non-socket")

        if self.readfd != self.writefd:
            raise Exceptions.EventException("Ambiguous file descriptor to connect with")

        if len(hostport) >= 2:
            host = hostport[0]
            port = hostport[1]
        else:
            host = self.host
            port = self.port

        try:
            self.sock.connect((host, port))
        except (IOError, OSError) as e:
            if e.errno not in Errors.ignore_errno:
                raise e

        # When we're connected, this will fire. =)
        self.event.set_event(self, EventLoop.EVENT_OUT)

        return True

class TimerEvent:
    """Event for timers"""
    def run_timer(self):
        """this is called when a write event fires in the eventloop.

        Override this in child classes to alter behaviour
        """
        pass

