/* file: $RCSfile: nutations.c,v $
** rcsid: $Id: nutations.c 261 2007-10-19 19:07:02Z laidler $
** Copyright Jeffrey W Percival
** *******************************************************************
** Space Astronomy Laboratory
** University of Wisconsin
** 1150 University Avenue
** Madison, WI 53706 USA
** *******************************************************************
** Do not use this software without attribution.
** Do not remove or alter any of the lines above.
** *******************************************************************
*/

/*
** *******************************************************************
** $RCSfile: nutations.c,v $
** compute the nutations in longitude and obliquity
**
** this formulation is from Kaplan, USNO circular 163, 1981.
** See also Woolard APAE v. 15. and Exp. Supp, 1992, p. 111
**
** we also include the corrections from Exp. Supp. 1992, p. 116
** and the planetary terms by Vondrak, p.118-119
**
** return the nutations in radians
** *******************************************************************
*/

#include "astro.h"

#undef PLAMETARY_TERMS

struct s_iau {
	int lm;
	int ls;
	int F;
	int D;
	int omega;
	int S0;
	double S1;
	int C0;
	double C1;
};

static struct s_iau iau[] = {
	{ 0, 0, 0, 0, 1,-171996,-174.2,92025, 8.9 },
	{ 0, 0, 0, 0, 2,   2062,   0.2, -895, 0.5 },
	{-2, 0, 2, 0, 1,     46,   0.0,  -24, 0.0 },
	{ 2, 0,-2, 0, 0,     11,   0.0,    0, 0.0 },
	{-2, 0, 2, 0, 2,     -3,   0.0,    1, 0.0 },

	{ 1,-1, 0,-1, 0,     -3,   0.0,    0, 0.0 },
	{ 0,-2, 2,-2, 1,     -2,   0.0,    1, 0.0 },
	{ 2, 0,-2, 0, 1,      1,   0.0,    0, 0.0 },
	{ 0, 0, 2,-2, 2, -13187,  -1.6, 5736,-3.1 },
	{ 0, 1, 0, 0, 0,   1426,  -3.4,   54,-0.1 },

	{ 0, 1, 2,-2, 2,   -517,   1.2,  224,-0.6 },
	{ 0,-1, 2,-2, 2,    217,  -0.5,  -95, 0.3 }, 
	{ 0, 0, 2,-2, 1,    129,   0.1,  -70, 0.0 }, 
	{ 2, 0, 0,-2, 0,     48,   0.0,    1, 0.0 }, 
	{ 0, 0, 2,-2, 0,    -22,   0.0,    0, 0.0 }, 

	{ 0, 2, 0, 0, 0,     17,  -0.1,    0, 0.0 }, 
	{ 0, 1, 0, 0, 1,    -15,   0.0,    9, 0.0 }, 
	{ 0, 2, 2,-2, 2,    -16,   0.1,    7, 0.0 }, 
	{ 0,-1, 0, 0, 1,    -12,   0.0,    6, 0.0 }, 
	{-2, 0, 0, 2, 1,     -6,   0.0,    3, 0.0 }, 

	{ 0,-1, 2,-2, 1,     -5,   0.0,    3, 0.0 }, 
	{ 2, 0, 0,-2, 1,      4,   0.0,   -2, 0.0 }, 
	{ 0, 1, 2,-2, 1,      4,   0.0,   -2, 0.0 }, 
	{ 1, 0, 0,-1, 0,     -4,   0.0,    0, 0.0 }, 
	{ 2, 1, 0,-2, 0,      1,   0.0,    0, 0.0 }, 

	{ 0, 0,-2, 2, 1,      1,   0.0,    0, 0.0 }, 
	{ 0, 1,-2, 2, 0,     -1,   0.0,    0, 0.0 }, 
	{ 0, 1, 0, 0, 2,      1,   0.0,    0, 0.0 }, 
	{-1, 0, 0, 1, 1,      1,   0.0,    0, 0.0 }, 
	{ 0, 1, 2,-2, 0,     -1,   0.0,    0, 0.0 }, 

	{ 0, 0, 2, 0, 2,  -2274,  -0.2,  977,-0.5 }, 
	{ 1, 0, 0, 0, 0,    712,   0.1,   -7, 0.0 }, 
	{ 0, 0, 2, 0, 1,   -386,  -0.4,  200, 0.0 }, 
	{ 1, 0, 2, 0, 2,   -301,   0.0,  129,-0.1 }, 
	{ 1, 0, 0,-2, 0,   -158,   0.0,   -1, 0.0 }, 

	{-1, 0, 2, 0, 2,    123,   0.0,  -53, 0.0 }, 
	{ 0, 0, 0, 2, 0,     63,   0.0,   -2, 0.0 }, 
	{ 1, 0, 0, 0, 1,     63,   0.1,  -33, 0.0 }, 
	{-1, 0, 0, 0, 1,    -58,  -0.1,   32, 0.0 }, 
	{-1, 0, 2, 2, 2,    -59,   0.0,   26, 0.0 }, 

	{ 1, 0, 2, 0, 1,    -51,   0.0,   27, 0.0 }, 
	{ 0, 0, 2, 2, 2,    -38,   0.0,   16, 0.0 }, 
	{ 2, 0, 0, 0, 0,     29,   0.0,   -1, 0.0 }, 
	{ 1, 0, 2,-2, 2,     29,   0.0,  -12, 0.0 }, 
	{ 2, 0, 2, 0, 2,    -31,   0.0,   13, 0.0 }, 

	{ 0, 0, 2, 0, 0,     26,   0.0,   -1, 0.0 }, 
	{-1, 0, 2, 0, 1,     21,   0.0,  -10, 0.0 }, 
	{-1, 0, 0, 2, 1,     16,   0.0,   -8, 0.0 }, 
	{ 1, 0, 0,-2, 1,    -13,   0.0,    7, 0.0 }, 
	{-1, 0, 2, 2, 1,    -10,   0.0,    5, 0.0 }, 

	{ 1, 1, 0,-2, 0,     -7,   0.0,    0, 0.0 }, 
	{ 0, 1, 2, 0, 2,      7,   0.0,   -3, 0.0 }, 
	{ 0,-1, 2, 0, 2,     -7,   0.0,    3, 0.0 }, 
	{ 1, 0, 2, 2, 2,     -8,   0.0,    3, 0.0 }, 
	{ 1, 0, 0, 2, 0,      6,   0.0,    0, 0.0 }, 

	{ 2, 0, 2,-2, 2,      6,   0.0,   -3, 0.0 }, 
	{ 0, 0, 0, 2, 1,     -6,   0.0,    3, 0.0 }, 
	{ 0, 0, 2, 2, 1,     -7,   0.0,    3, 0.0 }, 
	{ 1, 0, 2,-2, 1,      6,   0.0,   -3, 0.0 }, 
	{ 0, 0, 0,-2, 1,     -5,   0.0,    3, 0.0 }, 

	{ 1,-1, 0, 0, 0,      5,   0.0,    0, 0.0 }, 
	{ 2, 0, 2, 0, 1,     -5,   0.0,    3, 0.0 }, 
	{ 0, 1, 0,-2, 0,     -4,   0.0,    0, 0.0 }, 
	{ 1, 0,-2, 0, 0,      4,   0.0,    0, 0.0 }, 
	{ 0, 0, 0, 1, 0,     -4,   0.0,    0, 0.0 }, 

	{ 1, 1, 0, 0, 0,     -3,   0.0,    0, 0.0 }, 
	{ 1, 0, 2, 0, 0,      3,   0.0,    0, 0.0 }, 
	{ 1,-1, 2, 0, 2,     -3,   0.0,    1, 0.0 }, 
	{-1,-1, 2, 2, 2,     -3,   0.0,    1, 0.0 }, 
	{-2, 0, 0, 0, 1,     -2,   0.0,    1, 0.0 }, 

	{ 3, 0, 2, 0, 2,     -3,   0.0,    1, 0.0 }, 
	{ 0,-1, 2, 2, 2,     -3,   0.0,    1, 0.0 }, 
	{ 1, 1, 2, 0, 2,      2,   0.0,   -1, 0.0 }, 
	{-1, 0, 2,-2, 1,     -2,   0.0,    1, 0.0 }, 
	{ 2, 0, 0, 0, 1,      2,   0.0,   -1, 0.0 }, 

	{ 1, 0, 0, 0, 2,     -2,   0.0,    1, 0.0 }, 
	{ 3, 0, 0, 0, 0,      2,   0.0,    0, 0.0 }, 
	{ 0, 0, 2, 1, 2,      2,   0.0,   -1, 0.0 }, 
	{-1, 0, 0, 0, 2,      1,   0.0,   -1, 0.0 }, 
	{ 1, 0, 0,-4, 0,     -1,   0.0,    0, 0.0 }, 

	{-2, 0, 2, 2, 2,      1,   0.0,   -1, 0.0 }, 
	{-1, 0, 2, 4, 2,     -2,   0.0,    1, 0.0 }, 
	{ 2, 0, 0,-4, 0,     -1,   0.0,    0, 0.0 }, 
	{ 1, 1, 2,-2, 2,      1,   0.0,   -1, 0.0 }, 
	{ 1, 0, 2, 2, 1,     -1,   0.0,    1, 0.0 }, 

	{-2, 0, 2, 4, 2,     -1,   0.0,    1, 0.0 }, 
	{-1, 0, 4, 0, 2,      1,   0.0,    0, 0.0 }, 
	{ 1,-1, 0,-2, 0,      1,   0.0,    0, 0.0 }, 
	{ 2, 0, 2,-2, 1,      1,   0.0,   -1, 0.0 }, 
	{ 2, 0, 2, 2, 2,     -1,   0.0,    0, 0.0 }, 

	{ 1, 0, 0, 2, 1,     -1,   0.0,    0, 0.0 }, 
	{ 0, 0, 4,-2, 2,      1,   0.0,    0, 0.0 }, 
	{ 3, 0, 2,-2, 2,      1,   0.0,    0, 0.0 }, 
	{ 1, 0, 2,-2, 0,     -1,   0.0,    0, 0.0 }, 
	{ 0, 1, 2, 0, 1,      1,   0.0,    0, 0.0 }, 

	{-1,-1, 0, 2, 1,      1,   0.0,    0, 0.0 }, 
	{ 0, 0,-2, 0, 1,     -1,   0.0,    0, 0.0 }, 
	{ 0, 0, 2,-1, 2,     -1,   0.0,    0, 0.0 }, 
	{ 0, 1, 0, 2, 0,     -1,   0.0,    0, 0.0 }, 
	{ 1, 0,-2,-2, 0,     -1,   0.0,    0, 0.0 }, 

	{ 0,-1, 2, 0, 1,     -1,   0.0,    0, 0.0 }, 
	{ 1, 1, 0,-2, 1,     -1,   0.0,    0, 0.0 }, 
	{ 1, 0,-2, 2, 0,     -1,   0.0,    0, 0.0 }, 
	{ 2, 0, 0, 2, 0,      1,   0.0,    0, 0.0 }, 
	{ 0, 0, 2, 4, 2,     -1,   0.0,    0, 0.0 }, 

	{ 0, 1, 0, 1, 0,      1,   0.0,    0, 0.0 }
};

#define N_IAU (106)

struct s_bih {
	int lm;
	int ls;
	int F;
	int D;
	int omega;
	int LS;
	int LC;
	int OC;
	int OS;
};

struct s_bih bih[] = {
	{ 0, 0, 0,  0, 1,   -725,  417, 213, 224 },
	{ 0, 1, 0,  0, 0,    523,   61, 208, -24 },
	{ 0, 0, 2, -2, 2,    102, -118, -41, -47 },
	{ 0, 0, 2,  0, 2,    -81,    0,  32,   0 }
};

#define N_BIH	(4)

struct s_plan {
	int lm;
	int F;
	int D;
	int omega;
	int Q;
	int V;
	int E;
	int M;
	int J;
	int S;
	int LS;
	int LC;
	int OC;
	int OS;
};

struct s_plan plan[] = {
	{  0,  0,  0,  0,  0,   0,   2,  0,  0,  0,  -1,  0,  0,  0 },
	{  1,  0, -2, -1,  3,   0,  -1,  0,  0,  0,   0, -1,  0,  0 },
	{  0,  0,  0,  0,  0,   1,  -2,  0,  0,  0,   0, -1,  0,  0 },
	{  0,  0,  0,  0,  0,   1,  -1,  0,  0,  0,  16,  0,  0,  0 },
	{  0,  0,  0,  0,  0,   1,   1,  0,  0,  0,  -4,  0,  0,  1 },

	{  0,  0,  0,  0,  0,   1,  -3,  0,  0,  0,  -1,  0,  0,  0 },
	{  0,  0,  0,  0,  0,   2,  -4,  0,  0,  0,   4,  0,  0,  3 },
	{  0,  0,  0,  0,  0,   2,  -3,  0,  0,  0,   0,  7,  0,  0 },
	{  0,  0,  0,  0,  0,   2,  -5,  0,  0,  0,   0, -1,  1,  0 },
	{  0,  0,  0,  0,  0,   2,  -2,  0,  0,  0,  -6,  1,  0,  0 },

	{  0,  0,  0,  0,  0,   2,  -1,  0,  0,  0,   0, -2, -1,  0 },
	{  0,  0,  0,  0,  0,   2,   0,  0,  0,  0,   4,  0,  0, -2 },
	{  0,  0,  0,  0,  0,   3,  -7,  0,  0,  0,   1,  0,  0,  0 },
	{  0,  0,  0,  0,  0,   3,  -6,  0,  0,  0,   0, -1,  0,  0 },
	{  0,  0,  0,  0,  0,   3,  -5,  0,  0,  0,  20, -4,  1,  7 },

	{  0,  0,  0,  1,  0,  -3,   5,  0,  0,  0,   0,  1,  0,  0 },
	{  0,  0,  0,  0,  0,   3,  -4,  0,  0,  0,   0,  4,  0,  0 },
	{  0,  0,  0,  0,  0,   3,  -3,  0,  0,  0,   2,  0,  0,  0 },
	{  0,  2, -2,  1,  0,  -3,   3,  0,  0,  0,   1,  0,  0, -1 },
	{  2,  0, -2,  0,  0,  -3,   3,  0,  0,  0,   1,  0,  0,  0 },

	{  0,  0,  0,  0,  0,   3,  -2,  0,  0,  0,   0, -1, -1,  0 },
	{  0,  0,  0,  0,  0,   4,  -7,  0,  0,  0,   0,  1,  0,  0 },
	{  0,  0,  0,  0,  0,   4,  -6,  0,  0,  0,  -5,  1, -1, -2 },
	{  0,  0,  0,  0,  0,   4,  -4,  0,  0,  0,   1,  0,  0,  0 },
	{  0,  0,  0,  0,  0,   5,  -8,  0,  0,  0,   0, -3,  2,  0 },

	{  0,  0,  0,  0,  0,   5,  -7,  0,  0,  0,   2,  0,  0,  1 },
	{  0,  2, -2,  1,  0,  -5,   6,  0,  0,  0,   0, -4, -2,  0 },
	{  0,  2, -2,  0,  0,  -5,   6,  0,  0,  0,   0,  2,  0,  0 },
	{  0,  0,  0,  0,  0,   5,  -5,  0,  0,  0,  -1,  0,  0,  0 },
	{  0,  0,  0,  0,  0,   6,  -9,  0,  0,  0,   0, -1,  0,  0 },

	{  0,  0,  0,  0,  0,   6,  -8,  0,  0,  0,  -1,  0,  0, -1 },
	{  2,  0, -2,  0,  0,  -6,   8,  0,  0,  0,  -1,  0,  0,  0 },
	{  0,  0,  0,  0,  0,   7,  -9,  0,  0,  0,   1,  0,  0,  0 },
	{  0,  0,  0,  0,  0,   8, -10,  0,  0,  0,  -1,  0,  0,  0 },
	{  0,  0,  0,  0,  0,   8, -15,  0,  0,  0,  -1, -1,  0,  0 },

	{  0,  0,  0,  0,  0,   8, -13,  0,  0,  0,   3, -4,  0,  0 },
	{  0,  0,  0,  1,  0,   8, -13,  0,  0,  0,   1,  1,  0,  0 },
	{  0,  0,  0,  1,  0,  -8,  13,  0,  0,  0,  -1,  1,  0,  0 },
	{  0,  0,  0,  0,  0,   8, -11,  0,  0,  0,  -1, -1,  0,  0 },
	{  1,  0,  0, -1,  0, -10,   3,  0,  0,  0,   0, -1,  0,  0 },

	{  1,  0,  0,  1,  0, -10,   3,  0,  0,  0,   0, -1,  0,  0 },
	{  1,  0,  0, -1,  0, -18,  16,  0,  0,  0,   1,  0,  0,  0 },
	{  1,  0,  0,  1,  0, -18,  16,  0,  0,  0,   1,  0,  0,  0 },
	{  1, -2,  0, -2,  0, -18,  16,  0,  0,  0,   1, -1,  0,  1 },
	{  1,  2,  0,  2,  0, -18,  16,  0,  0,  0,   1, -1,  0, -1 },

	{  1,  0, -2,  1,  0,  20, -21,  0,  0,  0,   0,  1,  1,  0 },
	{  0,  0,  0,  0,  0,   0,   1, -1,  0,  0,  -1,  0,  0,  0 },
	{  0,  0,  0,  0,  0,   0,   1,  2,  0,  0,  -1, -1,  0,  0 },
	{  0,  0,  0,  0,  0,   0,   1, -2,  0,  0,  -5,  3,  0,  0 },
	{  0,  0,  0,  1,  0,   0,  -1,  2,  0,  0,  -6, -6, -3,  3 },

	{  0,  0,  0,  0,  0,   0,   0,  2,  0,  0,  -1,  0,  0,  0 },
	{  0,  0,  0,  0,  0,   0,   2, -2,  0,  0,  -4,  0,  0,  0 },
	{  0,  0,  0,  0,  0,   0,   3, -2,  0,  0,   1, -1,  0,  0 },
	{  0,  0,  0,  0,  0,   0,   4, -2,  0,  0,   1,  0,  0, -1 },
	{  0,  0,  0,  0,  0,   0,   2, -3,  0,  0,  -1,  0,  0,  0 },

	{  0,  0,  0,  0,  0,   0,   2, -4,  0,  0,   0,  1,  0,  0 },
	{  0,  0,  0,  0,  0,   0,   3, -4,  0,  0,  -1,  1,  0,  0 },
	{  0,  0,  0,  0,  0,   0,   2, -8,  3,  0,   1, -4,  2,  0 },
	{  0,  0,  0,  1,  0,   0,  -4,  8, -3,  0,   1,  3,  2, -1 },
	{  0,  0,  0,  1,  0,   0,   4, -8,  3,  0,  -1,  3,  2,  1 },

	{  0,  0,  0,  0,  0,   0,   6, -8,  3,  0,   1, -4, -2, -1 },
	{  0,  0,  0,  0,  0,   0,   1,  0, -3,  0,   1,  0,  0,  0 },
	{  0,  0,  0,  0,  0,   0,   2,  0, -3,  0,   1,  0,  0,  0 },
	{  2,  0, -2, -1,  0,   0,  -2,  0,  3,  0,   1,  0,  0,  1 },
	{  2,  0, -2,  1,  0,   0,  -2,  0,  3,  0,   2,  0,  0, -1 },

	{  2,  0, -2,  0,  0,   0,  -2,  0,  3,  0,   4,  0,  0,  0 },
	{  0,  0,  0,  0,  0,   0,   1,  0, -2,  0,  -2,  3,  0,  0 },
	{  0,  0,  0,  0,  0,   0,   2,  0, -2,  0,   4,  0,  0,  0 },
	{  2,  0, -2, -1,  0,   0,  -2,  0,  2,  0,  -5,  0,  0, -3 },
	{  2,  0, -2,  1,  0,   0,  -2,  0,  2,  0,   1,  0,  0,  0 },

	{  0,  2, -2,  1,  0,   0,  -2,  0,  2,  0,   1,  0,  0,  0 },
	{  2,  0, -2,  0,  0,   0,  -2,  0,  2,  0,   4,  0,  0,  0 },
	{  0,  0,  0,  0,  0,   0,   3,  0, -2,  0,   1, -1,  0,  0 },
	{  0,  0,  0,  0,  0,   0,   4,  0, -2,  0,  -2,  0,  0,  1 },
	{  0,  0,  0,  0,  0,   0,   1,  0, -1,  0, -16,  0,  0,  0 },

	{  0,  0,  0,  0,  0,   0,   2,  0, -1,  0,  -2,  0,  0,  1 },
	{  0,  0,  0,  0,  0,   0,   3,  0, -1,  0,   5,  1,  0, -2 },
	{  0,  0,  0,  0,  0,   0,   0,  0,  1,  0,  -1, -1, -1,  0 },
	{  0,  0,  0,  1,  0,   0,   0,  0,  1,  0,   0, -1,  0,  0 },
	{  0,  0,  0,  0,  0,   0,   1,  0,  1,  0,  -1,  0,  0,  1 },

	{  0,  0,  0,  0,  0,   0,   2,  0,  1,  0,   2,  0,  0, -1 },
	{  0,  0,  0,  0,  0,   0,   0,  0,  2,  0, -12,  2,  1,  4 },
	{  0,  0,  0,  0,  0,   0,   1,  0,  2,  0,   0, -1,  0,  0 },
	{  0,  0,  0,  0,  0,   0,   0,  0,  3,  0,  -1,  0,  0,  0 },
	{  0,  0,  0,  0,  0,   0,   1,  0,  0, -1,  -1,  0,  0,  0 },
};

#define N_PLAN	(85)

void
nutations(double tdt, double *delta_phi, double *delta_eps)
{
    double D;
    double F;
    double T;		/* elapsed julian centuries */
    double deps;	/* nutation in obliquity */
    double dphi;	/* nutation in longitude */
    double lm;
    double ls;
    double omega;
    int i;
#ifdef PLANETARY_TERMS
    double V;		/* venus */
    double S;		/* saturn */
    double Q;		/* mercury */
    double M;		/* moon */
    double J;		/* jupiter */
    double E;		/* earth */
#endif

    T = (tdt - J2000) / 36525;

    /* in arcseconds */
    omega = 450160.280 + (T * (-482890.539 + (T * (7.455 + (T * 0.008)))));
    /* convert to radians */
    omega = as2r(omega);
    /* add in whole revs */
    omega -= 5.0 * 2 * M_PI * T;

    /* in arcseconds */
    lm = 485866.733 + (T * (715922.633 + (T * (31.310 + (T * 0.064)))));
    /* convert to radians */
    lm = as2r(lm);
    /* add in whole revs */
    lm += 1325.0 * 2 * M_PI * T;

    /* in arcseconds */
    ls = 1287099.804 + (T * (1292581.244 + (T * (-0.577 + (T * -0.012)))));
    /* convert to radians */
    ls = as2r(ls);
    /* add in whole revs */
    ls += 99.0 * 2 * M_PI * T;

    /* in arcseconds */
    F = 335778.877 + (T * (295263.137 + (T * (-13.257 + (T * 0.011)))));
    /* convert to radians */
    F = as2r(F);
    /* add in whole revs */
    F += 1342.0 * 2 * M_PI * T;

    /* in arcseconds */
    D = 1072261.307 + (T * (1105601.328 + (T * (-6.891 + (T * 0.019)))));
    /* convert to radians */
    D = as2r(D);
    /* add in whole revs */
    D += 1236.0 * 2 * M_PI * T;

    /**************************/
    /* initialize the outputs */
    /**************************/
    *delta_phi = 0.0;
    *delta_eps = 0.0;

    /*********************/
    /* do the IAU theory */
    /*********************/
    dphi = 0.0;
    deps = 0.0;
    for (i = 0; i < N_IAU; i++) {
	double angle;

	angle = 0.0;
	angle += iau[i].lm * lm;
	angle += iau[i].ls * ls;
	angle += iau[i].F * F;
	angle += iau[i].D * D;
	angle += iau[i].omega * omega;

	dphi += (iau[i].S0 + (T * iau[i].S1)) * sin(angle);

	deps += (iau[i].C0 + (T * iau[i].C1)) * cos(angle);
    }
    /* convert to arcseconds */
    dphi *= 1e-4;
    deps *= 1e-4;

    /* convert to radians */
    dphi = as2r(dphi);
    deps = as2r(deps);

    /* add this contribution */
    *delta_phi += dphi;
    *delta_eps += deps;

#ifdef IAU_CORRECTIONS
    /**************************/
    /* do the IAU corrections */
    /**************************/
    dphi = 0.0;
    deps = 0.0;
    for (i = 0; i < N_BIH; i++) {
	double angle;

	angle = 0.0;
	angle += bih[i].lm * lm;
	angle += bih[i].ls * ls;
	angle += bih[i].F * F;
	angle += bih[i].D * D;
	angle += bih[i].omega * omega;

	dphi += bih[i].LS * sin(angle) + bih[i].LC * cos(angle);
	deps += bih[i].OC * cos(angle) + bih[i].OS * sin(angle);
    }
    /* convert to arcseconds */
    dphi *= 1e-5;
    deps *= 1e-5;

    /* convert to radians */
    dphi = as2r(dphi);
    deps = as2r(deps);

    /* add this contribution */
    *delta_phi += dphi;
    *delta_eps += deps;
#endif

#ifdef PLANETARY_TERMS
    /**************************/
    /* do the planetary terms */
    /**************************/

    Q = d2r(252.3 + T * 149472.7);
    V = d2r(179.9 + T * 58517.8);
    E = d2r(98.4 + T * 35999.4);
    M = d2r(353.3 + T * 19140.3);
    J = d2r(32.3 + T * 3034.9);
    S = d2r(48.0 + T * 1222.1);

    dphi = 0.0;
    deps = 0.0;
    for (i = 0; i < N_PLAN; i++) {
	double angle;

	angle = 0.0;
	angle += plan[i].lm * lm;
	angle += plan[i].F * F;
	angle += plan[i].D * D;
	angle += plan[i].omega * omega;
	angle += plan[i].Q * Q;
	angle += plan[i].V * V;
	angle += plan[i].E * E;
	angle += plan[i].M * M;
	angle += plan[i].J * J;
	angle += plan[i].S * S;

	dphi = plan[i].LS * sin(angle) + plan[i].LC * cos(angle);
	deps = plan[i].OC * cos(angle) + plan[i].OS * sin(angle);
    }
    /* convert to arcseconds */
    dphi *= 1e-5;
    deps *= 1e-5;

    /* convert to radians */
    dphi = as2r(dphi);
    deps = as2r(deps);

    /* add this contribution */
    *delta_phi += dphi;
    *delta_eps += deps;
#endif

    return;
}
