################################################################################
##
## This file is part of PyTango, a python binding for Tango
## 
## http://www.tango-controls.org/static/PyTango/latest/doc/html/index.html
##
## Copyright 2011 CELLS / ALBA Synchrotron, Bellaterra, Spain
## 
## PyTango is free software: you can redistribute it and/or modify
## it under the terms of the GNU Lesser General Public License as published by
## the Free Software Foundation, either version 3 of the License, or
## (at your option) any later version.
## 
## PyTango is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU Lesser General Public License for more details.
## 
## You should have received a copy of the GNU Lesser General Public License
## along with PyTango.  If not, see <http://www.gnu.org/licenses/>.
##
################################################################################

"""
This is an internal PyTango module. It provides tango log classes that can
be used as decorators in any method of :class:`PyTango.DeviceImpl`.

To access these members use directly :mod:`PyTango` module and NOT PyTango.log4tango.

Example::

    import PyTango
    
    class MyDev(PyTango.Device_4Impl):
        
        PyTango.InfoIt()
        def read_Current(self, attr):
            attr.set_value(self._current)
"""

__all__ = [ "TangoStream", "LogIt", "DebugIt", "InfoIt", "WarnIt", 
            "ErrorIt", "FatalIt" ]
            
__docformat__ = "restructuredtext"

import functools

class TangoStream:
    
    def __init__(self, fn):
        self._fn = fn
        self._accum = ""

    def write(self, s):
        self._accum += s
        # while there is no new line, just accumulate the buffer
        try:
            if s[-1] == '\n' or s.index('\n') >= 0:
                self.flush()
        except ValueError:
            pass
        
    def flush(self):
        b = self._accum
        if b is None or len(self._accum) == 0:
            return
        #take the '\n' because the log adds it
        if b[-1] == '\n': b = b[:-1]
        self._fn(b)
        self._accum = ""


class LogIt(object):
    """A class designed to be a decorator of any method of a 
    :class:`PyTango.DeviceImpl` subclass. The idea is to log the entrance and 
    exit of any decorated method.

    Example::
    
        class MyDevice(PyTango.Device_4Impl):
            
            @PyTango.LogIt()
            def read_Current(self, attr):
                attr.set_value(self._current, 1)

    All log messages generated by this class have DEBUG level. If you whish
    to have different log level messages, you should implement subclasses that
    log to those levels. See, for example, :class:`PyTango.InfoIt`.

    The constructor receives three optional arguments:
        * show_args - shows method arguments in log message (defaults to False)
        * show_kwargs - shows keyword method arguments in log message (defaults to False)
        * show_ret - shows return value in log message (defaults to False)
    """
    
    def __init__(self, show_args=False, show_kwargs=False, show_ret=False):
        """Initializes de LogIt object.
            
            :param show_args: (bool) show arguments in log message (default is False)
            :param show_kwargs: (bool) show keyword arguments in log message (default is False)
            :param show_ret: (bool) show return in log message (default is False)
        """
        self._show_args = show_args
        self._show_kwargs = show_kwargs
        self._show_ret = show_ret

    def __compact(self, v, maxlen=25):
        v = repr(v)
        if len(v) > maxlen:
            v = v[:maxlen-6] + " [...]"
        return v

    def __compact_dict(self, k, v, maxlen=None):
        if maxlen is None:
            return "%s=%s" % (k, __compact(v))
        return "%s=%s" % (k, __compact(v, maxlen=maxlen))

    def is_enabled(self, d):
        return d.get_logger().is_debug_enabled()

    def get_log_func(self, d):
        return d.debug_stream

    def __call__(self, f):
        @functools.wraps(f)
        def log_stream(*args, **kwargs):
            d = args[0]
            if not self.is_enabled(d):
                return f(*args, **kwargs)
            in_msg = "-> %s(" % f.func_name
            if self._show_args:
                in_msg += ", ".join(map(self.__compact, args[1:]))
            if self._show_kwargs:
                kwargs_str = ( self.__compact_dict(k,v) for k,v in kwargs.items() )
                in_msg += ", ".join(kwargs_str)
            in_msg += ")"
            self.get_log_func(d)(in_msg)
            ret = f(*args, **kwargs)
            out_msg = ""
            if self._show_ret:
                out_msg += self.__compact(ret) + " "
            out_msg += "<- %s()" % f.func_name
            self.get_log_func(d)(out_msg)
            return ret
        return log_stream


class DebugIt(LogIt):
    """A class designed to be a decorator of any method of a 
    :class:`PyTango.DeviceImpl` subclass. The idea is to log the entrance and 
    exit of any decorated method as DEBUG level records.

    Example::
    
        class MyDevice(PyTango.Device_4Impl):
            
            @PyTango.DebugIt()
            def read_Current(self, attr):
                attr.set_value(self._current, 1)

    All log messages generated by this class have DEBUG level.

    The constructor receives three optional arguments:
        * show_args - shows method arguments in log message (defaults to False)
        * show_kwargs - shows keyword method arguments in log message (defaults to False)
        * show_ret - shows return value in log message (defaults to False)
    """

    def is_enabled(self, d):
        return d.get_logger().is_debug_enabled()

    def get_log_func(self, d):
        return d.debug_stream


class InfoIt(LogIt):
    """A class designed to be a decorator of any method of a 
    :class:`PyTango.DeviceImpl` subclass. The idea is to log the entrance and 
    exit of any decorated method as INFO level records.

    Example::
    
        class MyDevice(PyTango.Device_4Impl):
            
            @PyTango.InfoIt()
            def read_Current(self, attr):
                attr.set_value(self._current, 1)

    All log messages generated by this class have INFO level.
    
    The constructor receives three optional arguments:
        * show_args - shows method arguments in log message (defaults to False)
        * show_kwargs - shows keyword method arguments in log message (defaults to False)
        * show_ret - shows return value in log message (defaults to False)
    """

    def is_enabled(self, d):
        return d.get_logger().is_info_enabled()

    def get_log_func(self, d):
        return d.info_stream


class WarnIt(LogIt):
    """A class designed to be a decorator of any method of a 
    :class:`PyTango.DeviceImpl` subclass. The idea is to log the entrance and 
    exit of any decorated method as WARN level records.

    Example::
    
        class MyDevice(PyTango.Device_4Impl):
            
            @PyTango.WarnIt()
            def read_Current(self, attr):
                attr.set_value(self._current, 1)

    All log messages generated by this class have WARN level.

    The constructor receives three optional arguments:
        * show_args - shows method arguments in log message (defaults to False)
        * show_kwargs - shows keyword method arguments in log message (defaults to False)
        * show_ret - shows return value in log message (defaults to False)
    """
    
    def is_enabled(self, d):
        return d.get_logger().is_warn_enabled()

    def get_log_func(self, d):
        return d.warn_stream


class ErrorIt(LogIt):
    """A class designed to be a decorator of any method of a 
    :class:`PyTango.DeviceImpl` subclass. The idea is to log the entrance and 
    exit of any decorated method as ERROR level records.

    Example::
    
        class MyDevice(PyTango.Device_4Impl):
            
            @PyTango.ErrorIt()
            def read_Current(self, attr):
                attr.set_value(self._current, 1)

    All log messages generated by this class have ERROR level.

    The constructor receives three optional arguments:
        * show_args - shows method arguments in log message (defaults to False)
        * show_kwargs - shows keyword method arguments in log message (defaults to False)
        * show_ret - shows return value in log message (defaults to False)
    """
    
    def is_enabled(self, d):
        return d.get_logger().is_error_enabled()

    def get_log_func(self, d):
        return d.error_stream


class FatalIt(LogIt):
    """A class designed to be a decorator of any method of a 
    :class:`PyTango.DeviceImpl` subclass. The idea is to log the entrance and 
    exit of any decorated method as FATAL level records.

    Example::
    
        class MyDevice(PyTango.Device_4Impl):
            
            @PyTango.FatalIt()
            def read_Current(self, attr):
                attr.set_value(self._current, 1)

    All log messages generated by this class have FATAL level.

    The constructor receives three optional arguments:
        * show_args - shows method arguments in log message (defaults to False)
        * show_kwargs - shows keyword method arguments in log message (defaults to False)
        * show_ret - shows return value in log message (defaults to False)
    """
    
    def is_enabled(self, d):
        return d.get_logger().is_fatal_enabled()

    def get_log_func(self, d):
        return d.fatal_stream