#!/usr/bin/env python2
import requests
import logging

logger = logging.getLogger('PyToRest')


class RestHTTPError(Exception):

    HTTPERROR = {
        400: "Bad Request",
        401: "Unauthorized",
        402: "Payement Required",
        403: "Forbidden",
        404: "Not Found",
        405: "Method Not Allowed",
        406: "Not Acceptable",
        407: "Proxy Authentication Required",
        408: "Request Time-out",
        409: "Conflict",
        410: "Gone",
        411: "Length Required",
        412: "Precondition Failed",
        413: "Request Entity Too Large",
        414: "Request-URI Too Long",
        415: "Unsupported Media Type",
        416: "Requested range unsatisfiable",
        417: "Expectation failed",
        418: "I'm a teapot",
        500: "Internal Server Error",
        501: "Not Implemented",
        502: "Bad Gateway or Proxy Error",
        503: "Service Unavailable",
        504: "Gateway Time-out",
        505: "HTTP Version not supported"
    }

    def __init__(self, r):
        self.code = r.status_code
        try:
            json = r.json()
            self.msg = json['error']['message']
        except ValueError:
            self.msg = (self.HTTPERROR[self.code] if self.code in self.HTTPERROR.keys()
                        else "Unknown error.")

    def __str__(self):
        return "%s: %s" % (self.code, self.msg)


class RestRequest(object):

    def __init__(self, url, auth=None):
        self.url = url
        self.auth = auth

    def __call__(self, cls):

        if self.auth and (not isinstance(self.auth, tuple) or len(self.auth) != 2):
            raise TypeError
        setattr(cls, '__url__', self.url)
        setattr(cls, '__auth__', self.auth)

        if 'GET' in dir(cls):
            GET = getattr(cls, 'GET')
            def http_get(self, *args, **kwargs):
                logger.info("GET " + cls.__url__)
                r = requests.get(self.__url__, auth=self.__auth__)
                if not r.ok:
                    raise RestHTTPError(r)
                else:
                    GET(self, r, *args, **kwargs)
            setattr(cls, 'GET', http_get)

        if 'POST' in dir(cls):
            POST = getattr(cls, 'POST')
            def http_post(self, data, *args, **kwargs):
                logger.info("POST " + cls.__url__)
                r = requests.post(self.__url__, auth=self.__auth__, data=data)
                if not r.ok:
                    raise RestHTTPError(r)
                else:
                    POST(self, r, *args, **kwargs)
            setattr(cls, 'POST', http_post)

        if 'PUT' in dir(cls):
            PUT = getattr(cls, 'PUT')
            def http_put(self, data, *args, **kwargs):
                logger.info("PUT " + cls.__url__)
                r = requests.put(self.__url__, auth=self.__auth__, data=data)
                if not r.ok:
                    raise RestHTTPError(r)
                else:
                    PUT(self, r, *args, **kwargs)
            setattr(cls, 'PUT', http_put)

        if 'DELETE' in dir(cls):
            DELETE = getattr(cls, 'DELETE')
            def http_delete(self, *args, **kwargs):
                logger.info("DELETE " + cls.__url__)
                r = requests.delete(self.__url__, auth=self.__auth__)
                if not r.ok:
                    raise RestHTTPError(r)
                else:
                    DELETE(self, r, *args, **kwargs)
            setattr(cls, 'DELETE', http_delete)

        return cls
