
""" Generic module for any kind of Python UI, as distinct from the classes these derive from which contains 
stuff also applicable even without this """

import scriptengine, replayer, recorder, definitions, os, sys, logging, subprocess
from ordereddict import OrderedDict
from locale import getdefaultlocale

# We really need our ConfigParser to be ordered, copied the one from 2.6 into the repository
if sys.version_info[:2] >= (2, 6):
    from ConfigParser import ConfigParser
else: # pragma: no cover - not currently running older than 2.5 in regular tests
    from ConfigParser26 import ConfigParser

class WidgetAdapter:
    adapterClass = None
    @staticmethod
    def setAdapterClass(adapterCls):
        WidgetAdapter.adapterClass = adapterCls
        
    @classmethod
    def adapt(cls, widget):
        return cls.adapterClass(widget)
    
    def __init__(self, widget):
        self.widget = widget

    def __getattr__(self, name):
        return getattr(self.widget, name)

    def __hash__(self):
        return hash(self.widget)

    def __cmp__(self, other):
        return cmp(self.widget, other)

    def getTitle(self):
        try:
            return self.getWidgetTitle()
        except AttributeError:
            return ""

    def getChildren(self):
        return map(self.adapt, self.getChildWidgets())

    def getType(self):
        return self.widget.__class__.__name__

    def isInstanceOf(self, widgetClass):
        return isinstance(self.widget, widgetClass)

    def getUIMapIdentifier(self):
        widgetName = self.getName()
        if not self.isAutoGenerated(widgetName): 
            return "Name=" + widgetName

        title = self.getTitle()
        if title:
            return self.encodeToLocale("Title=" + title)
       
        label = self.getLabel()
        if label:
            return self.encodeToLocale("Label=" + label)

        tooltip = self.getTooltip()
        if tooltip:
            return "Tooltip=" + tooltip
        return "Type=" + self.getType()

    def getTooltip(self):
        return ""

    def encodeToLocale(self, unicodeText):
        if unicodeText:
            try:
                encoding = getdefaultlocale()[1] or "utf-8"
                return unicodeText.encode(encoding, 'replace')
            except ValueError:
                # Get this if locale is invalid for example
                # Return the text as-is and hope for the best
                return unicodeText
        else:
            return ""

    def findPossibleUIMapIdentifiers(self):
        return [ "Name=" + self.getName(), "Title=" + self.encodeToLocale(self.getTitle()), 
                 "Label=" + self.encodeToLocale(self.getLabel()), "Tooltip=" + self.getTooltip(),
                 "Type=" + self.getType() ]


class GuiEvent(definitions.UserEvent):
    def __init__(self, name, widget, *args):
        definitions.UserEvent.__init__(self, name)
        self.widget = widget
        self.programmaticChange = False
        self.changeMethod = self.getRealMethod(self.getChangeMethod())
        if self.changeMethod:
            allChangeMethods = [ self.changeMethod ] + self.getProgrammaticChangeMethods()
            for method in allChangeMethods:
                self.interceptMethod(method, ProgrammaticChangeIntercept)

    def getRealMethod(self, method):
        if isinstance(method, MethodIntercept):
            return method.method
        else:
            return method
        
    def interceptMethod(self, method, interceptClass):
        if isinstance(method, MethodIntercept):
            method.addEvent(self)
        else:
            setattr(self.getSelf(method), method.__name__, interceptClass(method, self))

    def getSelf(self, method):
        # seems to be different for built-in and bound methods
        try:
            return method.im_self
        except AttributeError:
            return method.__self__

    def getChangeMethod(self):
        pass

    def getProgrammaticChangeMethods(self):
        return []

    def shouldRecord(self, *args):
        return not self.programmaticChange

    def setProgrammaticChange(self, val, *args, **kwargs):
        self.programmaticChange = val

    @classmethod
    def getAssociatedSignatures(cls, widget):
        return set([ cls.getAssociatedSignal(widget) ])

    def widgetDisposed(self):
        return False

    def checkWidgetStatus(self):
        if self.widgetDisposed():
            raise definitions.UseCaseScriptError, "widget " + self.describeWidget() + \
                  " has already been disposed, cannot simulate event " + repr(self.name)

        if not self.widgetVisible():
            raise definitions.UseCaseScriptError, "widget " + self.describeWidget() + \
                  " is not visible at the moment, cannot simulate event " + repr(self.name)

        if not self.widgetSensitive():
            raise definitions.UseCaseScriptError, "widget " + self.describeWidget() + \
                  " is not sensitive to input at the moment, cannot simulate event " + repr(self.name)


class MethodIntercept:
    def __init__(self, method, event):
        self.method = method
        self.events = [ event ]
    def addEvent(self, event):
        self.events.append(event)

class ProgrammaticChangeIntercept(MethodIntercept):
    def __call__(self, *args, **kwds):
        # Allow for possibly nested programmatic changes, observation can have knock-on effects
        eventsToBlock = filter(lambda event: not event.programmaticChange, self.events)
        for event in eventsToBlock:
            event.setProgrammaticChange(True, *args, **kwds)
        retVal = apply(self.method, args, kwds)
        for event in eventsToBlock:
            event.setProgrammaticChange(False)
        return retVal



class ScriptEngine(scriptengine.ScriptEngine):
    defaultMapFile = os.path.join(scriptengine.ScriptEngine.usecaseHome, "ui_map.conf")
    def __init__(self, enableShortcuts=False, uiMapFiles=[ defaultMapFile ],
                 customEventTypes=[], universalLogging=True, binDir=""):
        self.uiMap = self.createUIMap(uiMapFiles)
        self.binDir = binDir
        self.addCustomEventTypes(customEventTypes)
        self.importCustomEventTypes()
        scriptengine.ScriptEngine.__init__(self, enableShortcuts, universalLogging=universalLogging)

    def createUIMap(self, uiMapFiles):
        return UIMap(self, uiMapFiles)

    def importCustomEventTypes(self):
        try:
            from customwidgetevents import customEventTypes
            self.addCustomEventTypes(customEventTypes)
        except ImportError:
            pass

    def addCustomEventTypes(self, customEventTypes):
        for customWidgetClass, customEventClasses in customEventTypes:
            for widgetClass, currEventClasses in self.eventTypes:
                if widgetClass is customWidgetClass:
                    # Insert at the start, to give first try to the custom events
                    currEventClasses[0:0] = customEventClasses
                    break
            self.eventTypes.insert(0, (customWidgetClass, customEventClasses))

    def findEventClassesFor(self, widget):
        eventClasses = []
        currClass = None
        for widgetClass, currEventClasses in self.eventTypes:
            if widget.isInstanceOf(widgetClass):
                if not currClass or issubclass(widgetClass, currClass):
                    eventClasses = currEventClasses
                    currClass = widgetClass
                elif not issubclass(currClass, widgetClass):
                    eventClasses = eventClasses + currEventClasses # make a copy
        return eventClasses

    def monitorSignal(self, eventName, signalName, widget, *args, **kw):
        if self.active():
            return self._monitorSignal(eventName, signalName, WidgetAdapter.adapt(widget), *args, **kw)

    def _monitorSignal(self, eventName, signalName, widget, argumentParseData=None):
        signalEvent = self._createSignalEvent(eventName, signalName, widget, argumentParseData)
        if signalEvent:
            self._addEventToScripts(signalEvent)
            return signalEvent

    def _addEventToScripts(self, event):
        if event.name and self.replayerActive():
            self.replayer.addEvent(event)
        if event.name and self.recorderActive():
            event.connectRecord(self.recorder.writeEvent)

    def getUsecaseNameChooserEnv(self):
        new_env = {}
        for var, value in os.environ.items():
            if var == "PATH":
                new_env[var] = value + os.pathsep + self.binDir
            elif not var.startswith("USECASE_RE"): # Don't transfer our record scripts!
                new_env[var] = value
        return new_env

    def getUsecaseNameChooserCmdArgs(self, recordScript, interface):
        mapFiles = self.uiMap.getMapFileNames()
        return [ "usecase_name_chooser", "-m", ",".join(mapFiles), 
                 "-r", recordScript, "-i", interface ]

    def hasAutoRecordings(self, fileName):
        # Don't start the name chooser process unnecessarily
        for line in open(fileName):
            if line.startswith("Auto."):
                return True
        return False

    def replaceAutoRecordingForUsecase(self, interface):
        recordScript = os.getenv("USECASE_RECORD_SCRIPT")
        if self.uiMap and recordScript and os.path.isfile(recordScript) and self.hasAutoRecordings(recordScript):
            sys.stdout.flush()
            cmdArgs = self.getUsecaseNameChooserCmdArgs(recordScript, interface)
            env = self.getUsecaseNameChooserEnv()
            if os.name == "posix":
                os.execvpe(cmdArgs[0], cmdArgs, env)
            else:
                subprocess.call(cmdArgs, env=env)

    def replaceAutoRecordingForShortcut(self, script):
        if self.uiMap and self.binDir:
            cmdArgs = self.getUsecaseNameChooserCmdArgs(script.scriptName, "gtk")
            subprocess.call(cmdArgs, env=self.getUsecaseNameChooserEnv())
    
    def getClassName(self, widgetClass, module):
        return module + "." + widgetClass.__name__

    def getFormatted(self, text, html, title):
        if html:
            return '<div class="Text_Header">' + title + "</div>\n" + \
                '<div class="Text_Normal">' + text + "</div>"
        else:
            return text

    def run(self, options, args):
        if options.supported:
            return self.describeSupportedWidgets()
        elif options.supported_html:
            return self.describeSupportedWidgets(html=True)
        else:
            try:
                return scriptengine.ScriptEngine.run(self, options, args)
            finally:
                if not options.disable_usecase_names:
                    self.replaceAutoRecordingForUsecase(options.interface)

    def describeSupportedWidgets(self, html=False):
        toolkit, module, actionWord, linkPrefix = self.getDescriptionInfo()
        intro = """The following lists the %s widget types and the associated %s on them which 
PyUseCase %s is currently capable of recording and replaying. Any type derived from the listed
types is also supported.
""" % (toolkit, actionWord, definitions.__version__)
        print self.getFormatted(intro, html, toolkit + " Widgets and " + actionWord + " supported for record/replay")
        classes = self.getRecordReplayInfo(module)
        classNames = sorted(classes.keys())
        if html:
            self.writeHtmlTable(classNames, classes, linkPrefix)
        else:
            self.writeAsciiTable(classNames, classes)

        logIntro = """
The following lists the %s widget types whose status and changes PyUseCase %s is 
currently capable of monitoring and logging. Any type derived from the listed types 
is also supported but will only have features of the listed type described.
""" % (toolkit, definitions.__version__)
        print self.getFormatted(logIntro, html, toolkit + " Widgets supported for automatic logging")
        classNames = [ self.getClassName(w, module) for w in self.getSupportedLogWidgets() ]
        classNames.sort()
        if html:
            self.writeHtmlList(classNames, module, linkPrefix)
        else:
            for className in classNames:
                print className
        return True

    def getRecordReplayInfo(self, module):
        classes = {}
        for widgetClass, currEventClasses in self.eventTypes:
            if len(currEventClasses):
                self.addSignals(classes, widgetClass, currEventClasses, module)
        return classes

    def writeAsciiTable(self, classNames, classes):
        for className in classNames:
            print className.ljust(self.getClassNameColumnSize()) + ":", " , ".join(classes[className])

    def getClassNameColumnSize(self):
        return 25 # seems to work, mostly

    def writeHtmlTable(self, classNames, classes, linkPrefix):
        print '<div class="Text_Normal"><table border=1 cellpadding=1 cellspacing=1>'
        for className in classNames:
            print '<tr><td>' + self.getLink(className, linkPrefix) + '</td><td><div class="Table_Text_Normal">' + \
                " , ".join(classes[className]) + "</div></td></tr>"
        print "</table></div>"

    def getLink(self, className, linkPrefix):
        docName = self.getDocName(className)
        return '<a class="Text_Link" href=' + linkPrefix + \
            docName + '.html>' + className + '</a>'

    def getDocName(self, className):
        return className.split(".")[-1].lower()

    def writeHtmlList(self, classNames, module, linkPrefix):
        print '<div class="Text_Normal">'
        for className in classNames:
            print '<li>' + self.getLink(className, linkPrefix)
        print '</div><div class="Text_Normal"><i>(Note that a textual version of this page can be auto-generated by running "pyusecase -s -i ' + module.lower() + '")</i></div>'

    @classmethod
    def getDisplayName(cls, signalName):
        return cls.signalDescs.get(signalName)

    @classmethod
    def getColumnDisplayName(cls, signalName):
        return cls.columnSignalDescs.get(signalName, signalName)


class WriteParserHandler:
    def __init__(self, fileName):
        self.fileName = fileName
        self.parser = ConfigParser(dict_type=OrderedDict)
        self.parser.optionxform = str # don't automatically lower-case everything
        self.parser.read([ self.fileName ])
        self.changed = False

    def write(self):
        if self.changed:
            dirName = os.path.dirname(self.fileName)
            if dirName and not os.path.isdir(dirName):
                os.makedirs(dirName)
            self.parser.write(open(self.fileName, "w"))
            self.changed = False

    def add_section(self, *args):
        self.changed = True
        self.parser.add_section(*args)

    def set(self, *args):
        self.changed = True
        self.parser.set(*args)

    def __getattr__(self, name):
        return getattr(self.parser, name)


class UIMapFileHandler:
    def __init__(self, uiMapFiles): 
        self.readFiles(uiMapFiles)

    def readFiles(self, uiMapFiles):
        # See top of file: uses the version from 2.6
        self.writeParsers = map(WriteParserHandler, uiMapFiles)
        if len(self.writeParsers) == 1:
            self.readParser = self.writeParsers[0]
        else:
            self.readParser = ConfigParser(dict_type=OrderedDict)
            self.readParser.optionxform = str # don't automatically lower-case everything
            self.readParser.read(uiMapFiles)

    def storeInfo(self, sectionName, signature, eventName):
        if not self.readParser.has_section(sectionName):
            self.writeParsers[-1].add_section(sectionName)
           
        signature = signature.replace("::", "-") # Can't store :: in ConfigParser unfortunately
        if not self.readParser.has_option(sectionName, signature):
            for writeParser in self.writeParsers:
                if writeParser.has_section(sectionName):
                    writeParser.set(sectionName, signature, eventName)
            
    def findWriteParser(self, section):
        for parser in self.writeParsers:
            if parser.has_section(section):
                return parser

    def updateSectionName(self, section, newName):
        writeParser = self.findWriteParser(section)
        if not writeParser.has_section(newName):
            writeParser.add_section(newName)
        for name, value in self.readParser.items(section):
            writeParser.set(newName, name, value)
        writeParser.remove_section(section)
        return newName

    def write(self, *args):
        for parserHandler in self.writeParsers:
            parserHandler.write()

    def __getattr__(self, name):
        return getattr(self.readParser, name)

    def findSectionAndOption(self, valueString):
        for section in self.readParser.sections():
            for optionName, value in self.readParser.items(section):
                if valueString.startswith(value):
                    return section, optionName
        return None, None

class UIMap:
    ignoreWidgetTypes = []
    def __init__(self, scriptEngine, uiMapFiles):
        self.fileHandler = UIMapFileHandler(uiMapFiles)
        self.scriptEngine = scriptEngine
        self.windows = []
        self.logger = logging.getLogger("gui map")

    def readFiles(self, uiMapFiles):
        self.fileHandler.readFiles(uiMapFiles)

    def findWidgetDetails(self, scriptCommand):
        return self.fileHandler.findSectionAndOption(scriptCommand)

    def getMapFileNames(self):
        return [ parser.fileName for parser in self.fileHandler.writeParsers ]

    def monitorAndStoreWindow(self, window):
        if window not in self.windows:
            self.windows.append(window)
            self.monitorWindow(WidgetAdapter.adapt(window))

    def monitorWindow(self, window):
        self.logger.debug("Monitoring new window with title " + repr(window.getTitle()))
        self.monitor(window)

    def monitor(self, widget, excludeWidgets=[]):
        if widget.widget not in excludeWidgets:
            self.monitorWidget(widget)
            self.monitorChildren(widget, excludeWidgets)
        
    def monitorChildren(self, widget, *args, **kw):
        for child in widget.getChildren():
            self.monitor(child, *args, **kw)

    def monitorWidget(self, widget):
        signaturesInstrumented, autoInstrumented = self.instrumentFromMapFile(widget)
        if self.scriptEngine.recorderActive():
            widgetType = widget.getType()
            for signature in self.findAutoInstrumentSignatures(widget, signaturesInstrumented):
                identifier = widget.getUIMapIdentifier().replace("'", "<APOSTROPHE>")
                autoEventName = "Auto." + widgetType + "." + signature + ".'" + identifier + "'"
                signalName, argumentParseData = self.parseSignature(signature)
                self.autoInstrument(autoEventName, signalName, widget, argumentParseData, widgetType)
        return autoInstrumented

    def tryImproveSectionName(self, widget, section):
        if not section.startswith("Name="):
            newName = widget.getUIMapIdentifier()
            if newName != section:
                return self.fileHandler.updateSectionName(section, newName)
        return section

    def instrumentFromMapFile(self, widget):
        widgetType = widget.getType()
        if widgetType in self.ignoreWidgetTypes:
            return set(), False
        signaturesInstrumented = set()
        autoInstrumented = False
        section = self.findSection(widget)
        if section:
            section = self.tryImproveSectionName(widget, section)
            self.logger.debug("Reading map file section " + repr(section) + " for widget of type " + widgetType)
            for signature, eventName in self.fileHandler.items(section):
                if self.tryAutoInstrument(eventName, signature, signaturesInstrumented, widget, widgetType):
                    autoInstrumented = True
        return signaturesInstrumented, autoInstrumented

    def tryAutoInstrument(self, eventName, signature, signaturesInstrumented, widget, widgetType):
        try:
            signalName, argumentParseData = self.parseSignature(signature)
            if self.autoInstrument(eventName, signalName, widget, argumentParseData, widgetType):
                signaturesInstrumented.add(signature)
                return True
        except definitions.UseCaseScriptError, e:
            sys.stderr.write("ERROR in UI map file: " + str(e) + "\n")
        return False

    def findAutoInstrumentSignatures(self, widget, preInstrumented):
        signatures = []
        for eventClass in self.scriptEngine.findEventClassesFor(widget):
            for signature in eventClass.getAssociatedSignatures(widget):
                if signature not in signatures and signature not in preInstrumented:
                    signatures.append(signature)
        return signatures

    def findSection(self, widget):
        sectionNames = widget.findPossibleUIMapIdentifiers()
        for sectionName in sectionNames:
            self.logger.debug("Looking up section name " + repr(sectionName))
            if self.fileHandler.has_section(sectionName):
                return sectionName

    def parseSignature(self, signature):
        parts = signature.split(".", 1)
        signalName = parts[0]
        if len(parts) > 1:
            return signalName, parts[1]
        else:
            return signalName, None

    def autoInstrument(self, eventName, signalName, widget, argumentParseData, *args):
        self.logger.debug("Monitor " + eventName + ", " + signalName + ", " + widget.getType() + ", " + str(argumentParseData))
        self.scriptEngine._monitorSignal(eventName, signalName, widget, argumentParseData)
        return True
        

# Use the GTK idle handlers instead of a separate thread for replay execution
class UseCaseReplayer(replayer.UseCaseReplayer):
    def __init__(self, uiMap, universalLogging, recorder):
        self.readingEnabled = False
        self.uiMap = uiMap
        self.idleHandler = None
        self.loggerActive = universalLogging
        self.recorder = recorder
        self.delay = float(os.getenv("USECASE_REPLAY_DELAY", 0.0))
        self.tryAddDescribeHandler()
        replayer.UseCaseReplayer.__init__(self)

    def isMonitoring(self):
        return self.loggerActive or (self.recorder.isActive() and self.uiMap)

    def tryAddDescribeHandler(self):
        if self.idleHandler is None and self.isMonitoring():
            self.idleHandler = self.makeDescribeHandler(self.handleNewWindows)
        else:
            self.idleHandler = None

    def makeDescribeHandler(self, method):
        return self.makeIdleHandler(method)

    def makeIdleReplayHandler(self, method):
        return self.makeIdleHandler(method)

    def handleNewWindows(self):
        for window in self.findWindowsForMonitoring():
            if self.uiMap and (self.isActive() or self.recorder.isActive()):
                self.uiMap.monitorAndStoreWindow(window)
            if self.loggerActive:
                self.describeNewWindow(window)
        return True

    def callReplayHandlerAgain(self):
        self.enableReplayHandler()

    def enableReading(self):
        self.readingEnabled = True
        self._disableIdleHandlers()
        self.enableReplayHandler()
    
    def _disableIdleHandlers(self):
        if self.idleHandler is not None:
            self.removeHandler(self.idleHandler)
            self.idleHandler = None
    
    def enableReplayHandler(self):
        self.idleHandler = self.makeReplayHandler(self.describeAndRun)

    def makeReplayHandler(self, method):
        if self.delay:
            milliseconds = int(self.delay * 1000)
            return self.makeTimeoutReplayHandler(method, milliseconds)
        else:
            return self.makeIdleReplayHandler(method)

    def describeAndRun(self, *args):
        self.handleNewWindows()
        if self.readingEnabled:
            self.readingEnabled = self.runNextCommand()
            if not self.readingEnabled:
                self.idleHandler = None
                self.tryAddDescribeHandler()
                if not self.idleHandler and self.uiMap: # pragma: no cover - cannot test with replayer disabled
                    # End of shortcut: reset for next time
                    self.logger.debug("Shortcut terminated: Resetting UI map ready for next shortcut")
                    self.uiMap.windows = [] 
                    self.events = {}
        if self.readingEnabled:
            return self.callReplayHandlerAgain()
        else:
            return False

    def getParseError(self, scriptCommand):
        widgetDescriptor, actionName = self.uiMap.findWidgetDetails(scriptCommand)
        if widgetDescriptor:
            return "Could not execute script command '" + scriptCommand + "'.\n" + \
                   "No widget found with descriptor '" + widgetDescriptor + "' to perform action '" + actionName + "' on."
        else:
            return replayer.UseCaseReplayer.getParseError(self, scriptCommand)

# Base class for tkinter and wx only right now, should be developed further and bring in gtk also
class Describer:
    def __init__(self):
        self.logger = logging.getLogger("gui log")
        self.windows = set()
        self.widgetsWithState = OrderedDict()

    def describe(self, window):
        if window in self.windows:
            return
        self.windows.add(window)
        title = self.getSpecificState(window)
        message = "-" * 10 + " " + self.getWindowString() + " '" + title + "' " + "-" * 10
        self.widgetsWithState[window] = title
        self.logger.info("\n" + message)
        self.logger.info(self.getWindowContentDescription(window))
        footerLength = min(len(message), 100) # Don't let footers become too huge, they become ugly...
        self.logger.info("-" * footerLength)

    def getWindowContentDescription(self, window):
        return self.getChildrenDescription(window)

    def getWindowString(self):
        return "Window"

    def findStateChanges(self, *args):
        defunctWidgets = []
        stateChanges = []
        for widget, oldState in self.widgetsWithState.items():
            if not self.shouldCheckForUpdates(widget, *args):
                continue
            try:
                state = self.getState(widget)
            except:
                # If the frame where it existed has been removed, for example...
                defunctWidgets.append(widget)
                continue

            if state != oldState:
                stateChanges.append((widget, oldState, state))
                self.widgetsWithState[widget] = state
            
        for widget in defunctWidgets:
            del self.widgetsWithState[widget]
        return stateChanges

    def shouldCheckForUpdates(self, *args):
        return True

    def describeStateChanges(self, stateChanges):
        for widget, oldState, state in stateChanges:
            changeDesc = self.getStateChangeDescription(widget, oldState, state).rstrip()
            if changeDesc:
                self.logger.info(changeDesc)

    def describeUpdates(self):
        stateChanges = self.findStateChanges()
        self.describeStateChanges(stateChanges)

    def getStateChangeDescription(self, widget, oldState, state):
        if isinstance(widget, self.getWindowClasses()):
            return "Changed title of " + self.getWindowString().lower() + " to '" + state + "'"
        else:
            return self.getUpdatePrefix(widget, oldState, state) + self.getDescription(widget)

    def getUpdatePrefix(self, widget, oldState, state):
        if isinstance(widget, self.getTextEntryClass()):
            return "Updated "
        else:
            return "\n"

    def addToDescription(self, desc, newText):
        if newText:
            if desc:
                desc += "\n"
            desc += newText.rstrip() + "\n"
        return desc

    def getDescription(self, widget, **kw):
        desc = ""
        desc = self.addToDescription(desc, self.getWidgetDescription(widget))
        desc = self.addToDescription(desc, self.getChildrenDescription(widget, **kw))
        return desc.rstrip()
    
    def getWidgetDescription(self, widget):
        for widgetClass in self.stateWidgets + self.statelessWidgets:
            if self.checkInstance(widget, widgetClass):
                methodName = "get" + widgetClass.__name__ + "Description"
                return getattr(self, methodName)(widget)
        
        return self.widgetTypeDescription(widget.__class__.__name__) # pragma: no cover - should be unreachable

    def widgetTypeDescription(self, typeName): # pragma: no cover - should be unreachable
        return "A widget of type '" + typeName + "'" 

    def checkInstance(self, widget, widgetClass):
        return isinstance(widget, widgetClass) # SWT has classloader trouble, override this

    def getState(self, widget):
        state = self.getSpecificState(widget)
        return state.strip()

    def getSpecificState(self, widget):
        for widgetClass in self.stateWidgets:
            if self.checkInstance(widget, widgetClass):
                methodName = "get" + widgetClass.__name__ + "State"
                return getattr(self, methodName)(widget)
        return ""
