# Copyright (c) 2010 Marco Giusti
# See LICENSE.txt for details.

from twisted.internet import gtk2reactor
gtk2reactor.install()
from twisted.internet import reactor
from twisted.python import usage, hook
from twisted.web import client
from zope.interface import implements
import gtk

from pycabulary import utils, interfaces


class Options(usage.Options):
    pass


class MainWindow(gtk.Window):

    def __init__(self):
        gtk.Window.__init__(self, gtk.WINDOW_TOPLEVEL)
        self.lang = 0
        self.set_default_size(500, 400)
        self.add_events(gtk.gdk.KEY_PRESS_MASK)
        self.connect('delete_event', lambda w, e, d=None: False)
        self.connect('destroy', self.actQuit)
        self.connect('key-release-event', self.keyReleased)
        self.createWidgets()

    def keyReleased(self, widget, event, data=None):
        if event.type == gtk.gdk.KEY_RELEASE and \
                event.state == gtk.gdk.CONTROL_MASK and \
                event.keyval == ord('q'):
            self.actQuit(widget, data)

    def createWidgets(self):
        h1 = gtk.HBox(False, 0)
        rbtn = gtk.RadioButton(None, 'To ITA')
        h1.pack_start(rbtn, False, False, 0)
        rbtn = gtk.RadioButton(rbtn, 'To ENG')
        h1.pack_start(rbtn, False, False, 0)
        rbtn.set_active(self.lang)

        v1 = gtk.VBox(False, 0)
        v1.pack_start(h1, False, False, 0)
        self.entry = gtk.Entry()
        self.entry.connect('activate', self.actTranslate)
        self.entry.connect('focus-out-event',
                           lambda w, *a: w.select_region(0, 0))
        self.entry.grab_focus()
        v1.pack_start(self.entry, True, True, 0)

        v2 = gtk.VBox(False, 0)
        clrBtn = gtk.Button('Clear')
        v2.pack_start(clrBtn, False, False, 0)
        clrBtn.connect('clicked', self.actClear)
        trnBtn = gtk.Button('Translate')
        v2.pack_start(trnBtn, False, False, 0)
        trnBtn.connect('clicked', self.actTranslate)

        h2 = gtk.HBox(False, 0)
        h2.pack_start(v1, True, True, 5)
        h2.pack_start(v2, False, False, 5)

        sw = gtk.ScrolledWindow()
        sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        self.store = gtk.ListStore(str, str)
        tw = gtk.TreeView(self.store)
        tw.set_property('can-focus', False)
        sw.add_with_viewport(tw)

        c1 = gtk.TreeViewColumn('Word', gtk.CellRendererText(), text=0)
        c1.set_resizable(True)
        c1.set_sort_column_id(0)
        tw.append_column(c1)

        c2 = gtk.TreeViewColumn('Translation', gtk.CellRendererText(),
                text=1)
        c2.set_resizable(True)
        c2.set_sort_column_id(1)
        tw.append_column(c2)

        v3 = gtk.VBox(False, 0)
        v3.pack_start(h2, False, False, 5)
        v3.pack_start(sw, True, True, 0)
        self.add(v3)

    def setLang(self, action, current, data=None):
        self.lang = current.get_current_value()

    def actClear(self, action, data=None):
        pass

    def actTranslate(self, action, data=None):
        pass

    def actQuit(self, widget, data=None):
        pass

    def getWord(self):
        """Return the word to translate.
        @rtype: C{str}
        """
        return self.entry.get_text()

    def getLang(self):
        """Return the language to translate to.
        @rtype: C{int}
        @return: 0: eng -> ita, 1: ita -> eng
        """
        return self.lang

    def insert(self, word, translation):
        """Insert a word in the list.
        @type word: C{str}
        @type translation: C{str}
        """
        self.store.append((word, translation))

    def clear(self):
        """Clear the words list."""
        self.store.clear()

    def getStore(self):
        """
        Return the object where the translations are stored

        @rtype: C{gtk.ListStore}
        @return: A C{list} like object
        """
        return self.store


class GtkApp:
    implements(interfaces.IApp)

    def run(self):
        hook.addPost(MainWindow, 'actClear', self.clear)
        hook.addPost(MainWindow, 'actTranslate', self.translate)
        hook.addPost(MainWindow, 'actQuit', self.quit)
        win = MainWindow()
        win.set_title('pycabulary')
        win.show_all()
        reactor.run()

    def clear(self, window, action):
        window.clear()

    def translate(self, window, action):
        word = window.getWord()
        lang = window.getLang()
        url = utils.getUrl(lang, word)
        d = client.getPage(url, timeout=30)
        d.addCallback(utils.translate, li=window.getStore())

    def quit(self, window, widget, action):
        reactor.stop()

def makeApp(config):
    return GtkApp()

