# Copyright (c) 2010 Marco Giusti
# See LICENSE.txt for details.

from Tkinter import *
from twisted.python import hook, usage
from twisted.internet import tksupport, reactor
from twisted.web import client
from zope.interface import implements
from pycabulary import utils, interfaces


NSWE = N+S+W+E


class Options(usage.Options):
    pass


class AppFrame(Frame):

    def __init__(self, master=None):
        Frame.__init__(self, master)
        self.rowconfigure(2, weight=1)
        self.columnconfigure(0, weight=1)
        self.createWidgets()

    def createWidgets(self):
        ef = Frame(self)
        ef.grid(row=0, column=0, rowspan=2, sticky=NSWE)
        ef.rowconfigure(1, weight=1)
        ef.columnconfigure(1, weight=1)
        self.lang = IntVar()
        for val, txt in enumerate(('To ITA', 'To ENG')):
            rbtn = Radiobutton(ef, text=txt, value=val, variable=self.lang)
            rbtn.grid(row=0, column=val, sticky=W, padx=5)
        self.lang.set(0)
        self.entry = Entry(ef)
        self.entry.grid(row=1, column=0, columnspan=2, sticky=W+E, padx=5)
        self.entry.bind("<KeyPress-Return>", lambda e: self.btnTranslate())

        cbtn = Button(self, text="Clear", command=self.btnClear)
        cbtn.grid(row=0, column=1)
        tbtn = Button(self, text="Translate", command=self.btnTranslate)
        tbtn.grid(row=1, column=1)

        f = Frame(self)
        f.grid(columnspan=2, sticky=N+S+W+E)
        f.rowconfigure(0, weight=1)
        f.columnconfigure(0, weight=1)
        sbar = Scrollbar(f, orient=VERTICAL)
        sbar.grid(row=0, column=1, sticky=N+S)
        self.listbox = Listbox(f, yscrollcommand=sbar.set)
        self.listbox.grid(row=0, column=0, sticky=N+S+W+E)
        sbar['command'] = self.listbox.yview

    def getWord(self):
        """Return the word to translate.
        @rtype: C{str}
        """
        return self.entry.get()

    def getLang(self):
        """Return the language to translate to.
        @rtype: C{int}
        @return: 0: eng -> ita, 1: ita -> eng
        """
        return self.lang.get()

    def insert(self, word, translation):
        """Insert a word in the list.
        @type word: C{str}
        @type translation: C{str}
        """
        self.listbox.insert(END, '%s: %s' % (word, translation))

    def clear(self):
        """Clear the words list."""
        self.listbox.delete(0, self.listbox.size())

    def getRoot(self):
        """Return the top level widget of this frame.
        @rtype: C{Tkinter.}
        """
        return self.winfo_toplevel()

    def btnClear(self):
        pass

    def btnTranslate(self):
        pass

    def eClose(self):
        pass


class TkApp:
    implements(interfaces.IApp)

    def run(self):
        root = Tk()
        root.title("pycabulary")
        root.protocol("WM_DELETE_WINDOW", reactor.stop)
        root.bind("<Control-KeyPress-q>", self.quit)
        hook.addPost(AppFrame, 'btnClear', self.clear)
        hook.addPost(AppFrame, 'btnTranslate', self.translate)
        AppFrame(root).grid(row=0, column=0, sticky=W+E+N+S)
        root.rowconfigure(0, weight=1)
        root.columnconfigure(0, weight=1)
        tksupport.install(root)
        reactor.run()

    def quit(self, a=None):
        reactor.stop()

    def clear(self, win):
        win.clear()

    def translate(self, win):
        utils.parser.reset()
        word = win.getWord()
        lang = win.getLang()
        url = utils.getUrl(lang, word)
        d = client.getPage(url, timeout=30)
        d.addCallback(utils.translate)
        d.addCallback(self.insert, win)

    def insert(self, words, win):
        for w, t in words:
            win.insert(w, t)

def makeApp(config):
    return TkApp()

