# Copyright (c) 2010 Marco Giusti
# See LICENSE.txt for details.

from __future__ import absolute_import

import types
import Tkinter as Tk

from twisted.python import usage, log
from twisted.internet import tksupport, reactor
from zope.interface import implements

from . import utils, interfaces, dicios, client


N, S, W, E = Tk.N, Tk.S, Tk.W, Tk.E


class Options(usage.Options):
    pass


class TkTranslatorWindow(Tk.Tk):

    def __init__(self, app, translator):
        Tk.Tk.__init__(self)
        self.translator = translator
        self.app = app
        self.map = utils.LanguageMap(translator)
        self.createWidgets()

    def createWidgets(self):
        self.title("pycabulary")
        self.protocol("WM_DELETE_WINDOW", self.onQuit)
        self.bind("<Control-KeyPress-q>", lambda a: self.onQuit())
        self.rowconfigure(0, weight=1)
        self.columnconfigure(0, weight=1)

        frame = Tk.Frame(self)
        frame.grid(row=0, column=0, sticky=W+E+N+S)
        frame.rowconfigure(2, weight=1)
        frame.columnconfigure(0, weight=1)

        self.langstr = Tk.StringVar()
        languages = self.map.getLabels()
        opt = Tk.OptionMenu(frame, self.langstr, *languages)
        opt.grid(sticky=W, padx=5)
        self.langstr.set(languages[0])
        self.wordEtr = Tk.Entry(self)
        self.wordEtr.grid(row=1, column=0, sticky=W+E, padx=5)
        self.wordEtr.bind("<KeyPress-Return>", lambda e: self.onTranslate())

        cbtn = Tk.Button(self, text="Clear", command=self.onClear)
        cbtn.grid(row=0, column=1)
        tbtn = Tk.Button(self, text="Translate", command=self.onTranslate)
        tbtn.grid(row=1, column=1)

        f = Tk.Frame(self)
        f.grid(columnspan=2, sticky=N+S+W+E)
        f.rowconfigure(0, weight=1)
        f.columnconfigure(0, weight=1)
        sbar = Tk.Scrollbar(f, orient=Tk.VERTICAL)
        sbar.grid(row=0, column=1, sticky=N+S)
        self.listbox = Tk.Listbox(f, yscrollcommand=sbar.set)
        self.listbox.grid(row=0, column=0, sticky=N+S+W+E)
        sbar['command'] = self.listbox.yview

    def onQuit(self):
        self.app.quit()

    def onClear(self):
        self.listbox.delete(0, self.listbox.size())

    def onTranslate(self):
        word = self.wordEtr.get()
        if type(word) is not types.UnicodeType:
            word = unicode(word)
        frm, to = self.getLanguages()
        d = self.translator.translate(word, frm, to)
        d.addCallback(self.insert)
        d.addErrback(log.err)

    def getLanguages(self):
        return self.map.getCodes(self.langstr.get())

    def insert(self, li):
        for i, row in enumerate(li):
            self.listbox.insert(i, '%s: %s' % row)


class TkApp(object):
    implements(interfaces.IApp)

    def __init__(self, translatorFactory=dicios.Dicios,
                 parserFactory=dicios.Parser, clientFactory=client.Client):
        self.translator = translatorFactory(clientFactory(), parserFactory)

    def quit(self):
        reactor.stop()

    def run(self):
        root = TkTranslatorWindow(self, self.translator)
        tksupport.install(root)
        reactor.run()


def makeApp(config):
    return TkApp()
