# Copyright (c) 2010 Marco Giusti
# See LICENSE.txt for details.

from __future__ import absolute_import

from twisted.python import usage, log
from zope.interface import implements
import urwid

from . import dicios, utils, interfaces, client


class Options(usage.Options):
    pass


class Body(urwid.Pile):

    def __init__(self):
        self.edit = urwid.Edit('Word: ')
        self.content = urwid.SimpleListWalker([])
        self.listbox = urwid.ListBox(self.content)
        l = [('fixed', 1, urwid.Filler(self.edit)), self.listbox]
        super(Body, self).__init__(l)

    def keypress(self, size, key):
        if key == 'ctrl l':
            self.content[:] = []
            return key
        elif key == 'ctrl w':
            txt = self.edit.get_edit_text()
            self.edit.set_edit_text(' '.join(txt.split()[:-1]))
        elif key == 'ctrl u':
            self.edit.set_edit_text('')
        elif key in ('up', 'down', 'page up', 'page down'):
            return self.listbox.keypress(size, key)
        else:
            return self.edit.keypress(size[:1], key)

    def getWord(self):
        return self.edit.get_edit_text()

    def append(self, widgets):
        self.content += widgets


class Main(urwid.Frame):

    def __init__(self, app, translator, drawCallback=None):
        self.translator = translator
        self.app = app
        body = Body()
        header = self.createHeader()
        footer = self.createFooter()
        self.drawCallback = drawCallback
        super(Main, self).__init__(body, header, footer)
        # self.lang = 0
        # self.map = utils.LanguageMap(translator)

    def createHeader(self):
        ver = urwid.Text('Pycabulary')
        header = urwid.AttrMap(ver, 'header-fg')
        return header

    def createFooter(self):
        pass

    def handleInput(self, inpt):
        if isinstance(inpt, tuple): # mouse
            return
        if inpt == 'esc':
            self.app.quit()
        elif inpt == 'enter':
            self.translate()
        else:
            return inpt

    def translate(self):
        word = self.body.getWord()
    #     lang = self.lang_cb.get_active_text()
    #     frm, to = self.map.getCodes(lang)
        frm, to = 'en', 'it'
        d = self.translator.translate(word, frm, to)
        d.addCallback(self.insert)
        if self.drawCallback:
            d.addCallback(self.drawCallback)
        d.addErrback(log.err)

    def insert(self, li):
        widgets = []
        for word, translation in li:
            w = urwid.Text(word.encode('utf8'))
            t = urwid.Text(translation.encode('utf8'))
            c = urwid.Columns([w, t], dividechars=2)
            widgets.append(c)
        self.body.append(widgets)


class UrwidApp:
    implements(interfaces.IApp)

    def __init__(self, translatorFactory=dicios.Dicios,
                 parserFactory=dicios.Parser, clientFactory=client.Client):
        self.translator = translatorFactory(clientFactory(), parserFactory)

    def run(self):
        # this use of namespaces really sucks
        w = Main(self, self.translator, lambda _: loop.draw_screen())
        el = urwid.TwistedEventLoop()
        loop = urwid.MainLoop(w, self.palette(), unhandled_input=w.handleInput,
                              event_loop=el)
        loop.run()

    def palette(self):
        return [
            ('header-fg', 'black', 'light gray', 'standout,underline')
        ]

    def quit(self):
        from twisted.internet import reactor
        reactor.stop()


def makeApp(config):
    return UrwidApp()
