# Copyright (c) 2010 Marco Giusti
# See LICENSE.txt for details.

from __future__ import absolute_import

from twisted.internet import gtk2reactor
gtk2reactor.install()
from twisted.internet import reactor
from twisted.python import usage, log
from zope.interface import implements
import gtk

from . import dicios, utils, interfaces, client


class Options(usage.Options):
    pass


class MainWindow(gtk.Window):

    def __init__(self, app, translator):
        gtk.Window.__init__(self, gtk.WINDOW_TOPLEVEL)
        self.lang = 0
        self.translator = translator
        self.app = app
        self.map = utils.LanguageMap(translator)
        self.createWidgets()

    def createWidgets(self):
        self.set_title('pycabulary')
        self.set_default_size(500, 400)
        self.add_events(gtk.gdk.KEY_PRESS_MASK)
        self.connect('delete_event', lambda w, e, d=None: False)
        self.connect('destroy', lambda *a: self.onQuit())
        self.connect('key-release-event', self.onKeyReleased)

        self.lang_cb = cb = gtk.combo_box_new_text()
        map(cb.append_text, self.map.getLabels())

        v1 = gtk.VBox(False, 0)
        v1.pack_start(self.lang_cb, False, False, 0)
        self.entry = gtk.Entry()
        self.entry.connect('activate', lambda *a: self.onTranslate())
        self.entry.connect('focus-out-event',
                           lambda w, *a: w.select_region(0, 0))
        self.entry.grab_focus()
        v1.pack_start(self.entry, True, True, 0)

        v2 = gtk.VBox(False, 0)
        clrBtn = gtk.Button('Clear')
        v2.pack_start(clrBtn, False, False, 0)
        clrBtn.connect('clicked', self.onClear)
        trnBtn = gtk.Button('Translate')
        v2.pack_start(trnBtn, False, False, 0)
        trnBtn.connect('clicked', lambda *a: self.onTranslate())

        h2 = gtk.HBox(False, 0)
        h2.pack_start(v1, True, True, 5)
        h2.pack_start(v2, False, False, 5)

        sw = gtk.ScrolledWindow()
        sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        self.store = gtk.ListStore(str, str)
        tw = gtk.TreeView(self.store)
        tw.set_property('can-focus', False)
        sw.add_with_viewport(tw)

        c1 = gtk.TreeViewColumn('Word', gtk.CellRendererText(), text=0)
        c1.set_resizable(True)
        c1.set_sort_column_id(0)
        tw.append_column(c1)

        c2 = gtk.TreeViewColumn('Translation', gtk.CellRendererText(),
                text=1)
        c2.set_resizable(True)
        c2.set_sort_column_id(1)
        tw.append_column(c2)

        v3 = gtk.VBox(False, 0)
        v3.pack_start(h2, False, False, 5)
        v3.pack_start(sw, True, True, 0)
        self.add(v3)
        self.show_all()

    def insert(self, li):
        map(self.store.append, li)

    def onKeyReleased(self, widget, event, data=None):
        if event.type == gtk.gdk.KEY_RELEASE and \
                event.state == gtk.gdk.CONTROL_MASK and \
                event.keyval == ord('q'):
            self.onQuit()

    def onClear(self, action, data=None):
        self.store.clear()

    def onTranslate(self):
        word = self.entry.get_text()
        lang = self.lang_cb.get_active_text()
        frm, to = self.map.getCodes(lang)
        d = self.translator.translate(word, frm, to)
        d.addCallback(self.insert)
        d.addErrback(log.err)

    def onQuit(self):
        self.app.quit()


class GtkApp:
    implements(interfaces.IApp)

    def __init__(self, translatorFactory=dicios.Dicios,
                 parserFactory=dicios.Parser, clientFactory=client.Client):
        self.translator = translatorFactory(clientFactory(), parserFactory)

    def run(self):
        MainWindow(self, self.translator)
        reactor.run()

    def quit(self):
        reactor.stop()


def makeApp(config):
    return GtkApp()
