"""
Utiltiy routines for working with states (wave functions or density matrices)
"""
import numpy as np
from numpy import pi
import re
from .io import read_complex


class State(object):
    """
    Class describing state (wave function or density matrix)

    Attributes
    ----------

    rho : ndarray(float64)
        time points at which the pulse values are defined
    psi : ndarray(float64), ndarray(complex128)
        array of real or complex pulse values
    """

    def __init__(self, from_dump=None):
        """
        Wave function or density matrix
        """
        self.psi = None
        self.rho = None
        if from_dump is not None:
            self.read_dump(from_dump)

    def distance(self, other):
        """
        Return the distance to another state
        """
        assert self.psi is not None, "self must be wave function"
        assert other.psi is not None, "other must be wave function"
        diff = self.psi - other.psi
        return np.sqrt(abs(np.vdot(diff, diff)))

    def read_dump(self, dump_file):
        """
        Read dump, as generated by the dump_ascii_state_t QDYN routine
        """
        int_line = re.compile(r'^\d+$')
        complex_line = re.compile(r'^[\dE.+-]+\s+[\dE.+-]+$')
        with open(dump_file) as in_fh:
            in_psi = False
            i = 0
            for line in in_fh:
                line = line.strip()
                if line.startswith("#"):
                    in_psi = False
                if line == "# psi":
                    in_psi = True
                    i = 0
                if in_psi:
                    if int_line.match(line):
                        self.psi = np.zeros(int(line), dtype=np.complex128)
                    elif complex_line.match(line):
                        self.psi[i] = read_complex(line)
                        i += 1




def random_density_matrix(N):
    """
    Return a random N x N density matrix

    >>> state = State()
    >>> state.rho = random_density_matrix(10)

    The resulting density matrix is normalized, positive semidefinite, and
    Hermitian

    >>> assert( abs(np.trace(state.rho) - 1.0) <= 1.0e-14 )
    >>> assert( np.min(np.linalg.eigvals(state.rho).real) > 0.0 )
    >>> assert( np.max(np.abs(state.rho.H - state.rho)) <= 1.0e-14 )
    """
    rho = np.matrix(np.zeros(shape=(N,N)), dtype=np.complex128)
    for i in xrange(N):
        for j in xrange(N):
            r   = np.random.rand()
            phi = np.random.rand()
            rho[i,j] = r * np.exp(2.0j*pi*phi)
    # make hermitian
    rho = 0.5 * (rho + rho.H)
    # make positive-semidifinite by squaring
    rho = rho * rho
    # normalize
    rho = rho / (rho.trace()[0,0])
    return rho


