#
#   Copyright 2014 Olivier Kozak
#
#   This file is part of QuickBean.
#
#   QuickBean is free software: you can redistribute it and/or modify it under the terms of the GNU Lesser General
#   Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option)
#   any later version.
#
#   QuickBean is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
#   warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
#   details.
#
#   You should have received a copy of the GNU Lesser General Public License along with QuickBean.  If not, see
#   <http://www.gnu.org/licenses/>.
#

import collections
import json


def exclude_hidden_properties():
    return lambda property_: not property_.startswith('_')


def exclude_properties(*properties):
    return lambda property_: property_ not in properties


def only_include_properties(*properties):
    return lambda property_: property_ in properties


class AutoInit(object):
    """A decorator used to enhance the given class with an auto-generated initializer.

    To use this decorator, you just have to place it in front of your class :

    >>> from quickbean import AutoInit
    >>>
    >>> @AutoInit('property_', 'other_property')
    ... class TestObject(object):
    ...     pass

    You will get an auto-generated initializer taking all the declared properties :

    >>> test_object = TestObject(
    ...     property_='value',
    ...     other_property='otherValue',
    ... )
    >>>
    >>> test_object.property_
    'value'
    >>> test_object.other_property
    'otherValue'

    """
    def __init__(self, *properties):
        self.properties = properties

    def __call__(self, bean_class):
        namespace = {}

        template = '\n'.join([
            'def __init__(self, %s):' % ', '.join(self.properties),
            '\n'.join(['\tself.%s = %s' % (property_, property_) for property_ in self.properties]),
        ])

        exec(template, namespace)

        bean_class.__init__ = namespace['__init__']

        return bean_class


class AutoInitFromJson(object):
    """A decorator used to enhance the given class with an auto-generated JSON decoder.

    To use this decorator, you just have to place it in front of your class :

    >>> from quickbean import AutoInitFromJson
    >>>
    >>> @AutoInitFromJson
    ... class TestObject(object):
    ...     def __init__(self, property_, other_property):
    ...         self.property_ = property_
    ...         self.other_property = other_property

    You will get an auto-generated JSON decoder :

    >>> test_object = TestObject.from_json_str('{"other_property": "otherValue", "property_": "value"}')
    >>>
    >>> str(test_object.property_)
    'value'
    >>> str(test_object.other_property)
    'otherValue'

    Many frameworks handle JSON with dictionaries, so you never have to parse JSON strings directly. In that case, you
    should use the 'from_json_dict' method instead :

    >>> test_object = TestObject.from_json_dict({'other_property': 'otherValue', 'property_': 'value'})
    >>>
    >>> str(test_object.property_)
    'value'
    >>> str(test_object.other_property)
    'otherValue'

    This decorator relies on the standard JSON decoder (https://docs.python.org/2/library/json.html). Values are then
    decoded according to this JSON decoder. But sometimes, it may be useful to customize how to decode some particular
    properties. This is done with types. Types are entities set to fields named as the corresponding properties suffixed
    with '_json_type' that decode the taken property value through the available 'from_json_str' method :

    >>> class CustomJsonType(object):
    ...     # noinspection PyMethodMayBeStatic
    ...     def from_json_str(self, value):
    ...         return '%sFromJson' % json.loads(value)
    >>>
    >>> @AutoInitFromJson
    ... class TestObject(object):
    ...     def __init__(self, property_, other_property):
    ...         self.property_ = property_
    ...         self.other_property = other_property
    ...
    ...     other_property_json_type = CustomJsonType()
    >>>
    >>> test_object = TestObject.from_json_str('{"other_property": "otherValue", "property_": "value"}')
    >>>
    >>> str(test_object.property_)
    'value'
    >>> str(test_object.other_property)
    'otherValueFromJson'

    If you prefer handle JSON with dictionaries, use the 'from_json_dict' method instead :

    >>> class CustomJsonType(object):
    ...     # noinspection PyMethodMayBeStatic
    ...     def from_json_dict(self, value):
    ...         return '%sFromJson' % value
    >>>
    >>> @AutoInitFromJson
    ... class TestObject(object):
    ...     def __init__(self, property_, other_property):
    ...         self.property_ = property_
    ...         self.other_property = other_property
    ...
    ...     other_property_json_type = CustomJsonType()
    >>>
    >>> test_object = TestObject.from_json_str('{"other_property": "otherValue", "property_": "value"}')
    >>>
    >>> str(test_object.property_)
    'value'
    >>> str(test_object.other_property)
    'otherValueFromJson'

    And if you have inner objects to map, you may simply use types like this :

    >>> from quickbean import AutoInitFromJson
    >>>
    >>> @AutoInitFromJson
    ... class InnerTestObject(object):
    ...     def __init__(self, property_, other_property):
    ...         self.property_ = property_
    ...         self.other_property = other_property
    >>>
    >>> @AutoInitFromJson
    ... class TestObject(object):
    ...     def __init__(self, inner):
    ...         self.inner = inner
    ...
    ...     inner_json_type = InnerTestObject
    >>>
    >>> test_object = TestObject.from_json_str('{"inner": {"other_property": "otherValue", "property_": "value"}}')
    >>>
    >>> str(test_object.inner.property_)
    'value'
    >>> str(test_object.inner.other_property)
    'otherValue'

    """
    def __new__(cls, bean_class):
        # noinspection PyShadowingNames
        def from_json_dict(cls, json_dict):
            for property_, value in dict(json_dict).items():
                if hasattr(cls, '%s_json_type' % property_):
                    type_ = getattr(cls, '%s_json_type' % property_)

                    if hasattr(type_, 'from_json_dict'):
                        json_dict[property_] = type_.from_json_dict(value)
                    if hasattr(type_, 'from_json_str'):
                        json_dict[property_] = type_.from_json_str(json.dumps(value))

            return cls(**json_dict)

        # noinspection PyShadowingNames
        def from_json_str(cls, json_str):
            return cls.from_json_dict(json.loads(json_str))

        bean_class.from_json_dict = classmethod(from_json_dict)
        bean_class.from_json_str = classmethod(from_json_str)

        return bean_class


class AutoBean(object):
    """A decorator used to enhance the given class with an auto-generated equality, representation and JSON encoder.

    To use this decorator, you just have to place it in front of your class :

    >>> from quickbean import AutoBean
    >>>
    >>> @AutoBean
    ... class TestObject(object):
    ...     def __init__(self, property_, other_property, _hidden_property):
    ...         self.property_ = property_
    ...         self.other_property = other_property
    ...         self._hidden_property = _hidden_property

    This is strictly equivalent as :

    >>> from quickbean import AutoEq, AutoRepr, AutoToJson
    >>>
    >>> @AutoEq
    ... @AutoRepr
    ... @AutoToJson
    ... class TestObject(object):
    ...     def __init__(self, property_, other_property, _hidden_property):
    ...         self.property_ = property_
    ...         self.other_property = other_property
    ...         self._hidden_property = _hidden_property

    """
    def __new__(cls, bean_class):
        return cls.with_properties_filter(exclude_hidden_properties())(bean_class)

    @classmethod
    def with_properties_filter(cls, properties_filter):
        def decorate_bean_class(bean_class):
            bean_class = AutoEq.with_properties_filter(properties_filter)(bean_class)
            bean_class = AutoRepr.with_properties_filter(properties_filter)(bean_class)
            bean_class = AutoToJson.with_properties_filter(properties_filter)(bean_class)

            return bean_class

        return decorate_bean_class


class AutoEq(object):
    """A decorator used to enhance the given class with an auto-generated equality.

    To use this decorator, you just have to place it in front of your class :

    >>> from quickbean import AutoEq
    >>>
    >>> @AutoEq
    ... class TestObject(object):
    ...     def __init__(self, property_, other_property, _hidden_property):
    ...         self.property_ = property_
    ...         self.other_property = other_property
    ...         self._hidden_property = _hidden_property

    You will get an auto-generated equality taking all the properties available from your class :

    >>> test_object = TestObject(
    ...     property_='value',
    ...     other_property='otherValue',
    ...     _hidden_property='hiddenValue',
    ... )
    >>>
    >>> test_object == TestObject(
    ...     property_='value',
    ...     other_property='otherValue',
    ...     _hidden_property='differentHiddenValue',
    ... )
    True
    >>>
    >>> test_object == TestObject(
    ...     property_='value',
    ...     other_property='differentOtherValue',
    ...     _hidden_property='differentHiddenValue',
    ... )
    False

    An interesting thing to note here is that hidden properties -i.e. properties that begin with an underscore- are not
    taken into account.

    It is also possible to exclude arbitrary properties with the 'exclude_properties' filter :

    >>> from quickbean import exclude_properties
    >>>
    >>> @AutoEq.with_properties_filter(exclude_properties('excluded_property'))
    ... class TestObject(object):
    ...     def __init__(self, property_, other_property, excluded_property):
    ...         self.property_ = property_
    ...         self.other_property = other_property
    ...         self.excluded_property = excluded_property
    >>>
    >>> test_object = TestObject(
    ...     property_='value',
    ...     other_property='otherValue',
    ...     excluded_property='excludedValue',
    ... )
    >>>
    >>> test_object == TestObject(
    ...     property_='value',
    ...     other_property='otherValue',
    ...     excluded_property='differentExcludedValue',
    ... )
    True

    If you prefer, it is even possible to do the opposite, that is to say, specifying the only properties to include
    with the 'only_include_properties' filter :

    >>> from quickbean import only_include_properties
    >>>
    >>> @AutoEq.with_properties_filter(only_include_properties('property_', 'other_property'))
    ... class TestObject(object):
    ...     def __init__(self, property_, other_property, non_included_property):
    ...         self.property_ = property_
    ...         self.other_property = other_property
    ...         self.non_included_property = non_included_property
    >>>
    >>> test_object = TestObject(
    ...     property_='value',
    ...     other_property='otherValue',
    ...     non_included_property='nonIncludedValue',
    ... )
    >>>
    >>> test_object == TestObject(
    ...     property_='value',
    ...     other_property='otherValue',
    ...     non_included_property='differentNonIncludedValue',
    ... )
    True

    """
    def __new__(cls, bean_class):
        return cls.with_properties_filter(exclude_hidden_properties())(bean_class)

    @classmethod
    def with_properties_filter(cls, properties_filter):
        def decorate_bean_class(bean_class):
            # noinspection PyShadowingNames
            def __eq__(self, other):
                visible_properties = {
                    property_: self.__dict__[property_] for property_ in self.__dict__ if properties_filter(property_)
                }
                other_visible_properties = {
                    property_: other.__dict__[property_] for property_ in other.__dict__ if properties_filter(property_)
                }

                return self.__class__ is other.__class__ and visible_properties == other_visible_properties

            # noinspection PyShadowingNames
            def __ne__(self, other):
                return not self.__eq__(other)

            bean_class.__eq__ = __eq__
            bean_class.__ne__ = __ne__

            return bean_class

        return decorate_bean_class


class AutoRepr(object):
    """A decorator used to enhance the given class with an auto-generated representation.

    To use this decorator, you just have to place it in front of your class :

    >>> from quickbean import AutoRepr
    >>>
    >>> @AutoRepr
    ... class TestObject(object):
    ...     def __init__(self, property_, other_property, _hidden_property):
    ...         self.property_ = property_
    ...         self.other_property = other_property
    ...         self._hidden_property = _hidden_property

    You will get an auto-generated representation taking all the properties available from your class :

    >>> test_object = TestObject(
    ...     property_='value',
    ...     other_property='otherValue',
    ...     _hidden_property='hiddenValue',
    ... )
    >>>
    >>> repr(test_object)
    "TestObject(other_property='otherValue', property_='value')"

    An interesting thing to note here is that hidden properties -i.e. properties that begin with an underscore- are not
    taken into account.

    It is also possible to exclude arbitrary properties with the 'exclude_properties' filter :

    >>> from quickbean import exclude_properties
    >>>
    >>> @AutoRepr.with_properties_filter(exclude_properties('excluded_property'))
    ... class TestObject(object):
    ...     def __init__(self, property_, other_property, excluded_property):
    ...         self.property_ = property_
    ...         self.other_property = other_property
    ...         self.excluded_property = excluded_property
    >>>
    >>> test_object = TestObject(
    ...     property_='value',
    ...     other_property='otherValue',
    ...     excluded_property='excludedValue',
    ... )
    >>>
    >>> repr(test_object)
    "TestObject(other_property='otherValue', property_='value')"

    If you prefer, it is even possible to do the opposite, that is to say, specifying the only properties to include
    with the 'only_include_properties' filter :

    >>> from quickbean import only_include_properties
    >>>
    >>> @AutoRepr.with_properties_filter(only_include_properties('property_', 'other_property'))
    ... class TestObject(object):
    ...     def __init__(self, property_, other_property, non_included_property):
    ...         self.property_ = property_
    ...         self.other_property = other_property
    ...         self.non_included_property = non_included_property
    >>>
    >>> test_object = TestObject(
    ...     property_='value',
    ...     other_property='otherValue',
    ...     non_included_property='nonIncludedValue',
    ... )
    >>>
    >>> repr(test_object)
    "TestObject(other_property='otherValue', property_='value')"

    """
    def __new__(cls, bean_class):
        return cls.with_properties_filter(exclude_hidden_properties())(bean_class)

    @classmethod
    def with_properties_filter(cls, properties_filter):
        def decorate_bean_class(bean_class):
            # noinspection PyShadowingNames
            def __repr__(self):
                visible_properties = {
                    property_: self.__dict__[property_] for property_ in self.__dict__ if properties_filter(property_)
                }

                return '%s(%s)' % (self.__class__.__name__, ', '.join([
                    '%s=%r' % (property_, value) for property_, value in sorted(visible_properties.items())
                ]))

            bean_class.__repr__ = __repr__

            return bean_class

        return decorate_bean_class


class AutoToJson(object):
    """A decorator used to enhance the given class with an auto-generated JSON encoder.

    To use this decorator, you just have to place it in front of your class :

    >>> from quickbean import AutoToJson
    >>>
    >>> @AutoToJson
    ... class TestObject(object):
    ...     def __init__(self, property_, other_property, _hidden_property):
    ...         self.property_ = property_
    ...         self.other_property = other_property
    ...         self._hidden_property = _hidden_property

    You will get an auto-generated JSON encoder taking all the properties available from your class :

    >>> test_object = TestObject(
    ...     property_='value',
    ...     other_property='otherValue',
    ...     _hidden_property='hiddenValue',
    ... )
    >>>
    >>> test_object.to_json_str()
    '{"other_property": "otherValue", "property_": "value"}'

    Many frameworks handle JSON with dictionaries, so you never have to parse JSON strings directly. In that case, you
    should use the 'to_json_dict' method instead :

    >>> test_object = TestObject(
    ...     property_='value',
    ...     other_property='otherValue',
    ...     _hidden_property='hiddenValue',
    ... )
    >>>
    >>> list(sorted(test_object.to_json_dict().items()))
    [('other_property', 'otherValue'), ('property_', 'value')]

    An interesting thing to note here is that hidden properties -i.e. properties that begin with an underscore- are not
    taken into account.

    It is also possible to exclude arbitrary properties with the 'exclude_properties' filter :

    >>> from quickbean import exclude_properties
    >>>
    >>> @AutoToJson.with_properties_filter(exclude_properties('excluded_property'))
    ... class TestObject(object):
    ...     def __init__(self, property_, other_property, excluded_property):
    ...         self.property_ = property_
    ...         self.other_property = other_property
    ...         self.excluded_property = excluded_property
    >>>
    >>> test_object = TestObject(
    ...     property_='value',
    ...     other_property='otherValue',
    ...     excluded_property='excludedValue',
    ... )
    >>>
    >>> test_object.to_json_str()
    '{"other_property": "otherValue", "property_": "value"}'

    If you prefer, it is even possible to do the opposite, that is to say, specifying the only properties to include
    with the 'only_include_properties' filter :

    >>> from quickbean import only_include_properties
    >>>
    >>> @AutoToJson.with_properties_filter(only_include_properties('property_', 'other_property'))
    ... class TestObject(object):
    ...     def __init__(self, property_, other_property, non_included_property):
    ...         self.property_ = property_
    ...         self.other_property = other_property
    ...         self.non_included_property = non_included_property
    >>>
    >>> test_object = TestObject(
    ...     property_='value',
    ...     other_property='otherValue',
    ...     non_included_property='nonIncludedValue',
    ... )
    >>>
    >>> test_object.to_json_str()
    '{"other_property": "otherValue", "property_": "value"}'

    This decorator relies on the standard JSON encoder (https://docs.python.org/2/library/json.html). Values are then
    encoded according to this JSON encoder. But sometimes, it may be useful to customize how to encode some particular
    properties. This is done through methods named as the corresponding properties suffixed with '_to_json_str' :

    >>> @AutoToJson
    ... class TestObject(object):
    ...     def __init__(self, property_, other_property, _hidden_property):
    ...         self.property_ = property_
    ...         self.other_property = other_property
    ...         self._hidden_property = _hidden_property
    ...
    ...     def other_property_to_json_str(self):
    ...         return json.dumps('%sToJson' % self.other_property)
    >>>
    >>> test_object = TestObject(
    ...     property_='value',
    ...     other_property='otherValue',
    ...     _hidden_property='hiddenValue',
    ... )
    >>>
    >>> test_object.to_json_str()
    '{"other_property": "otherValueToJson", "property_": "value"}'

    If you prefer handle JSON with dictionaries, use the '_to_json_dict' suffixed method instead :

    >>> @AutoToJson
    ... class TestObject(object):
    ...     def __init__(self, property_, other_property, _hidden_property):
    ...         self.property_ = property_
    ...         self.other_property = other_property
    ...         self._hidden_property = _hidden_property
    ...
    ...     def other_property_to_json_dict(self):
    ...         return '%sToJson' % self.other_property
    >>>
    >>> test_object = TestObject(
    ...     property_='value',
    ...     other_property='otherValue',
    ...     _hidden_property='hiddenValue',
    ... )
    >>>
    >>> test_object.to_json_str()
    '{"other_property": "otherValueToJson", "property_": "value"}'

    This solution is quite simple, but it has a little drawback. Since it directly relies on the object's properties,
    the encoding code cannot be reused somewhere else. If you want your encoding code to be reusable, use types instead.
    Types are entities set to fields named as the corresponding properties suffixed with '_json_type' that encode the
    taken property value through the available 'to_json' method :

    >>> class CustomJsonType(object):
    ...     # noinspection PyMethodMayBeStatic
    ...     def to_json_str(self, value):
    ...         return json.dumps('%sToJson' % value)
    >>>
    >>> @AutoToJson
    ... class TestObject(object):
    ...     def __init__(self, property_, other_property, _hidden_property):
    ...         self.property_ = property_
    ...         self.other_property = other_property
    ...         self._hidden_property = _hidden_property
    ...
    ...     other_property_json_type = CustomJsonType()
    >>>
    >>> test_object = TestObject(
    ...     property_='value',
    ...     other_property='otherValue',
    ...     _hidden_property='hiddenValue',
    ... )
    >>>
    >>> test_object.to_json_str()
    '{"other_property": "otherValueToJson", "property_": "value"}'

    If you prefer handle JSON with dictionaries, use the 'to_json_dict' method instead :

    >>> class CustomJsonType(object):
    ...     # noinspection PyMethodMayBeStatic
    ...     def to_json_dict(self, value):
    ...         return '%sToJson' % value
    >>>
    >>> @AutoToJson
    ... class TestObject(object):
    ...     def __init__(self, property_, other_property, _hidden_property):
    ...         self.property_ = property_
    ...         self.other_property = other_property
    ...         self._hidden_property = _hidden_property
    ...
    ...     other_property_json_type = CustomJsonType()
    >>>
    >>> test_object = TestObject(
    ...     property_='value',
    ...     other_property='otherValue',
    ...     _hidden_property='hiddenValue',
    ... )
    >>>
    >>> test_object.to_json_str()
    '{"other_property": "otherValueToJson", "property_": "value"}'

    """
    def __new__(cls, bean_class):
        return cls.with_properties_filter(exclude_hidden_properties())(bean_class)

    @classmethod
    def with_properties_filter(cls, properties_filter):
        def decorate_bean_class(bean_class):
            # noinspection PyShadowingNames
            def to_json_dict(self):
                # noinspection PyShadowingNames
                def value_to_json_dict(property_, value):
                    if hasattr(self, '%s_to_json_dict' % property_):
                        return getattr(self, '%s_to_json_dict' % property_)()
                    if hasattr(self, '%s_to_json_str' % property_):
                        return json.loads(getattr(self, '%s_to_json_str' % property_)())

                    if hasattr(self, '%s_json_type' % property_):
                        type_ = getattr(self, '%s_json_type' % property_)

                        if hasattr(type_, 'to_json_dict'):
                            return getattr(self, '%s_json_type' % property_).to_json_dict(value)
                        if hasattr(type_, 'to_json_str'):
                            return json.loads(getattr(self, '%s_json_type' % property_).to_json_str(value))

                    if hasattr(value, 'to_json_dict'):
                        return collections.OrderedDict(sorted(value.to_json_dict().items()))
                    if hasattr(value, 'to_json_str'):
                        return collections.OrderedDict(sorted(json.loads(value.to_json_str()).items()))

                    return value

                visible_properties = {
                    property_: self.__dict__[property_] for property_ in self.__dict__ if properties_filter(property_)
                }

                return collections.OrderedDict(
                    (property_, value_to_json_dict(property_, value))
                    for property_, value in sorted(visible_properties.items())
                )

            # noinspection PyShadowingNames
            def to_json_str(self):
                return json.dumps(self.to_json_dict())

            bean_class.to_json_dict = to_json_dict
            bean_class.to_json_str = to_json_str

            return bean_class

        return decorate_bean_class
