from fabric.api import env, prefix, task, roles, run, execute
from quilt import utilities
from . import environ, machine, proxy, app, db, queue, cache


@task
def build():
    utilities.notify(u'Now building out the remote environment.')

    execute(environ.make)
    execute(clone)
    execute(environ.ensure)
    execute(db.create)
    execute(validate)
    execute(migrate)
    execute(db.initial_data)
    execute(collectstatic)
    execute(proxy.ensure)
    execute(app.ensure)
    execute(queue.ensure)


@task
def upgrade():
    utilities.notify(u'Now starting the project upgrade sequence.')

    execute(fetch)
    execute(merge)
    execute(environ.ensure)
    execute(validate)
    execute(migrate)
    execute(collectstatic)
    execute(proxy.ensure)
    execute(app.ensure)
    execute(queue.ensure)


@task
def deploy():
    utilities.notify(u'Now starting the project deploy sequence.')

    execute(fetch)
    execute(merge)
    execute(validate)
    execute(migrate)
    execute(collectstatic)
    execute(app.restart)
    execute(proxy.restart)


@task
def bootstrap(initial='no', environment='no', clear_cache='no'):
    utilities.notify(u'Bootstrapping the project. Hold on tight.')

    if initial == 'yes':
        execute(db.create)
    else:
        execute(db.rebuild)

    execute(migrate)
    execute(db.initial_data)

    if environment == 'yes':
        execute(env.ensure)

    if clear_cache == 'yes':
        execute(cache.flush)

    execute(app.restart)
    execute(proxy.restart)


@roles('app')
@task
def clone():
    utilities.notify(u'Now cloning from the remote repository.')

    with prefix(env.workon):
        run('git clone ' + env.repository_location + ' .')
        run('git checkout ' + env.repository_deploy_branch)
        run(env.deactivate)


@roles('app')
@task
def fetch():
    utilities.notify(u'Now fetching from the remote repository.')

    with prefix(env.workon):
        run('git fetch')
        run(env.deactivate)


@roles('app')
@task
def merge():
    utilities.notify(u'Now merging from the remote repository.')

    with prefix(env.workon):
        run('git merge ' + env.repository_deploy_branch + ' origin/' + env.repository_deploy_branch)
        run('git checkout ' + env.repository_deploy_branch)
        run(env.deactivate)


@roles('app')
@task
def validate():
    utilities.notify(u'Now running Django validations.')

    with prefix(env.workon):
        run('python manage.py validate')
        run(env.deactivate)


@roles('app')
@task
def migrate():
    utilities.notify(u'Now running Django migrations.')

    with prefix(env.workon):
        run('python manage.py syncdb --noinput --migrate')
        run(env.deactivate)


@roles('app')
@task
def collectstatic():
    utilities.notify(u'Now running Django static asset collector.')

    with prefix(env.workon):
        run('python manage.py collectstatic')
        run(env.deactivate)


@roles('app')
@task
def test():
    utilities.notify(u'Running the project test suite.')

    project_namespace = env.project_name + '.apps.'
    project_apps = []
    declared_apps = env.django_settings.INSTALLED_APPS

    for a in declared_apps:
       if a.startswith(project_namespace):
           project_apps.append(a[len(project_namespace):])

    run('python manage.py test ' + ' '.join(project_apps))


@roles('app')
@task
def sanity():
    utilities.notify(u'Starting the project sanity check. Here come the notifications:\n')

    utilities.sanity_check()


@roles('app')
@task
def command(cmd, activate='no'):
    utilities.notify(u'Now executing the command you passed.')

    if activate == 'yes':

        with prefix(env.workon):
            run(cmd)
            run(env.deactivate)
    else:
        run(cmd)
