from __future__ import with_statement, absolute_import
from weakref import ref
from warnings import warn
from mext.reaction.stm import *

__all__ = [
    'InputConflict', 'CellKey',
    'AbstractCell', 'HasListenersBase',
    'ConstantMixin', 'Constant',
    'Value',
]



class CellKey(object):
    """
        A key to be used in snapshots, supports garbage collection
        assisted by controller. A more naive strategy would not work
        because snapshots are copied and keys would never get dropped.
        Also, there's more than one concurrent snapshot possible, which
        necessitates the snapshot tracking.
    """
    __slots__ = ('cell')
    def __init__(self, cell):
        self.cell = ref(cell, self.discard)

    def discard(self, ref):
        discard = getattr(ctrl, 'discard_key', None)
        if discard is not None:
            discard(self)

    @property
    def dead(self):
        return self.cell() is None

    def __repr__(self):
        cell = self.cell()
        if cell is None:
            cell_repr = '<dead cell>'
        else:
            cell_repr = repr(cell)
        return '%s(%s)' % (self.__class__.__name__, cell_repr)





class AbstractCell(AbstractSubject):
    """
        Base class for cells. Useful for its default write and .value property.
        Also good for its support of old get_value / set_value methods
        Also used for a check in CellAttribute.__set__ to replace a cell
    """
    __slots__ = ()

    def read(self):
        raise NotImplementedError(self)
    def write(self, value):
        raise AttributeError("%s is read-only" % self.__class__.__name__)

    value = property(
        lambda self: self.read(),
        lambda self, value: self.write(value)
    )



    def get_value(self):
        warn("cell.get_value() is deprecated, use cell.read()", DeprecationWarning, 2)
        return self.read()
    def set_value(self, value):
        warn("cell.set_value(val) is deprecated, use cell.write(val)", DeprecationWarning, 2)
        self.write(value)










class ConstantMixin(AbstractCell):
    """
        A read-only abstract cell that can't change
    """
    __slots__ = ()
    has_listeners = False
    #was_set = False

    def __setattr__(self, name, value):
        raise AttributeError("Constants can't be changed", self)

    def __repr__(self):
        return "Constant(%r)" % (self.read(),)


class Constant(ConstantMixin):
    """
        A pure read-only value
    """
    __slots__ = 'read'

    def __init__(self, value):
        Constant.read.__set__(self, lambda: value)








class HasListenersBase(AbstractCell):
    __slots__ = '_has_listeners_cell', '_next_listener'

    def __init__(self):
        self._has_listeners_cell = None
        self._next_listener = None
        super(HasListenersBase, self).__init__()

    @property
    def has_listeners(self):
        if ctrl.txn and ctrl.txn.crule.is_listener:
            cell = self._has_listeners_cell
            if cell is None:
                cell = self._has_listeners_cell = Value(self._next_listener is not None)
            return cell.read()
        else:
            return self._next_listener is not None

    def _set_listener(self, listener):
        self._next_listener = listener
        if self._has_listeners_cell is not None:
            if self._has_listeners_cell.next_listener is None:
                self._has_listeners_cell = None
            elif ctrl:
                with ctrl.need_txn() as txn:
                    if not txn.undoing:
                        txn.effect(self._has_listeners_cell.write, listener is not None)
    next_listener = property(lambda self: self._next_listener, _set_listener)

    def become_constant(self):
        if self.next_subject:
            raise RuntimeError("%s instance is trying to become a constant, but has subjects: %r"
                % (self.__class__.__name__, list(self.iter_subjects()))
            )
        link = next_listener = self.next_listener
        self.next_listener = None
        if self._has_listeners_cell is not None:
            ctrl.txn.changed(self._has_listeners_cell)
            self._has_listeners_cell = None

        while link is not None:
            listener = link()
            if listener is not None:
                next_subj = listener.next_subject = link.next_subject
                link.prev_listener = None
                if next_subj is None and isinstance(listener, HasListenersBase):
                    listener.become_constant()
            link = link.next_listener
        self.rule = None
        self.__class__ = self._const_class()





class Value(HasListenersBase, AbstractCell):
    """
        A read-write value
    """
    __slots__ = 'key', '__weakref__'

    def __init__(self, value=None):
        self.key = CellKey(self)
        with ctrl.init_txn() as txn:
            super(Value, self).__init__()
            txn.set_key(txn.writes, self.key, value)

    def run_reset(self):
        pass

    def read(self, for_check=False):
        txn = ctrl.txn
        key = self.key
        if for_check:
            pass
        elif txn:
            txn.used(self)
        else:
            return ctrl.state[key]
        if key in txn.writes:
            return txn.writes[key]
        else:
            txn.note_read(key)
            return txn.snapshot[key]

    def write(self, value):
        with ctrl.need_txn() as txn:
            if txn.resetting:
                if value != self.read(True):
                    raise InputConflict(
                        "During reset Value cannot be changed (%r != %r)"
                        % (value, self.read(True))
                    )
            elif self in txn.set_by:
                # this value was created or set in this txn
                set_by = txn.set_by[self]
                if set_by is not txn.crule:
                    # we didn't process setting from current rule yet
                    curval = txn.writes[self.key]
                    # make sure rules can't set different values
                    # @@ in fact, we shoudn't allow multiple sets at all
                    if value != curval:
                        raise InputConflict(curval, value, set_by, txn.crule)
                else:
                    #@@xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
                    # This is here only because w/o this
                    # if a @maintain sets an attr while initializing
                    # and once again later in the same txn, we'd want to
                    # call .changed again for that one, as there's no way to detect
                    # if the rule is running for the second time in the txn,
                    # we call this.
                    # With new intialization code we will be able to assume
                    # rule only runs once per txn
                    txn.changed(self)
                txn.change_key(txn.writes, self.key, value)
            else:
                # first write of this cell in this txn
                key = self.key
                txn.set_key(txn.writes, key, value)
                txn.set_key(txn.set_by, self, txn.crule)
                txn.note_read(key)
                if value != txn.snapshot[key]:
                    txn.changed(self)
                    # what if a rule changes value to something and back
                    # wouldn't we want to ignore that?

    def __repr__(self):
        if isinstance(ctrl.txn, DisabledTransaction):
            val = '<in disabled-txn>'
        elif ctrl.txn and self.key in ctrl.txn.writes:
            val = repr(ctrl.txn.writes[self.key])
        else:
            val = repr(ctrl.state[self.key])
        return "%s(%s)" % (self.__class__.__name__, val)

    @property
    def was_set(self):
        ctrl.txn.used(self)
        return self in ctrl.txn.set_by




