from __future__ import absolute_import
import sys

from mext.reaction.celltypes import *
from mext.reaction.component.base import *

from mext.const import *

NOT_GIVEN = const()




def named_lambda(func, name):
    if getattr(func, '__name__', None) == '<lambda>':
        func.__name__ = name
    return func

class classmethod_func(object):
    def __init__(self, func):
        self.func = func

    def __get__(self, ob, type):
        name = '%s.%s' % (type.__name__, self.func.__name__)
        return named_lambda(lambda **kw: self.func(type, **kw), name)


def bind(rule, ob):
    if hasattr(rule, '__get__'):
        return rule.__get__(ob, type(ob))
    return rule









class CellAttribute(object):
    optional = True
    __name__ = None

    def __get__(self, ob, typ=None):
        if ob is None:
            return self
        return get_cell(ob, self.__name__, attr=self).read()

    def __repr__(self):
        return "%s(%r)" % (self.__class__.__name__, self.__name__)

    def set_name(self, name):
        assert self.__name__ is None or self.__name__ == name, self.__name__
        self.__name__ = name

    def __set__(self, ob, value):
        raise AttributeError("%r.%s is not settable" % (ob, self.__name__))

    @classmethod
    def _make_multi(cls, frame, kw, arg=None, **opts):
        for k, v in kw.items():
            if k in frame.f_locals:
                raise TypeError("%s is already defined in this class" % (k,))
            v = named_lambda(v, k)
            if arg:
                opts[arg] = v
                attr = cls(**opts)
            else:
                attr = cls(v, **opts)
            attr.set_name(k)
            frame.f_locals[k] = attr

    @classmethod
    def attrs(cls, **attrs):
        cls._make_multi(sys._getframe(1), attrs)









class ValueAttribute(CellAttribute):
    default_value = None

    def make_cell(self, ob):
        return self.factory(self.initial_value(ob))

    def initial_value(self, ob):
        if self.value is NOT_GIVEN:
            return self.default_value
        return self.value

    def __set__(self, ob, value):
        try:
            cells = ob.__cells__
        except AttributeError:
            cells = Cells(ob)

        try:
            cell = cells[self.__name__]
        except KeyError:
            cell = self.make_cell(ob)
            if isinstance(cell, ConstantMixin):
                #@@ add .set_init method to the interface and move this code there
                cells.setdefault(self.__name__, Constant(value))
                return
            cell = cells.setdefault(self.__name__, cell)
        cell.write(value)


class MakeAttribute(ValueAttribute):
    def check_make(self):
        if self.value is not NOT_GIVEN and self.make is not None:
            raise TypeError("Can't specify both a value and 'make'")

    def initial_value(self, ob):
        if self.value is NOT_GIVEN:
            if self.make is not None:
                make = bind(self.make, ob)
                with ctrl.need_txn() as txn:
                    with txn.new_root() as root:
                        r = make()
                        if txn.rule_reads or txn.rule_writes:
                            msg = ("make rule (%r) may not read or write any cells, has read %s; writes: %s"
                                % (make, list(txn.rule_reads), txn.rule_writes)
                            )
                            raise RuntimeError(msg)
                        return r
            return self.default_value
        return self.value



class ResetAttribute(ValueAttribute):
    def init_reset(self, init=NOT_GIVEN, reset=NOT_GIVEN):
        if init is not NOT_GIVEN and reset is not NOT_GIVEN:
            raise TypeError("Can't specify both 'init' and 'reset'")
        if reset is not NOT_GIVEN:
            self.value = reset
            self.resetting = True
            self.factory = self.resetting_cell_cls
        else:
            self.value = init
            self.resetting = False
            self.factory = self.cell_cls



class RuleDecorator(CellAttribute):
    def __init__(self, rule):
        self.rule = rule

    def make_cell(self, ob):
        rule = bind(self.rule, ob)
        return self.factory(rule, self.initial_value(ob))

    def set_name(self, name):
        super(RuleDecorator, self).set_name(name)
        if self.rule is not None:
            self.rule = named_lambda(self.rule, name)




class ListenerAttribute(RuleDecorator):
    optional = False
    def __set__(self, ob, value):
        raise AttributeError

    def make_cell(self, ob):
        rule = bind(self.rule, ob)
        return self.factory(rule)



class Parametrized(object):
    """
        A baseclass that allows to defer instance creation until
        it gets a non-keyword argument (specifying the rule).

        For example here's what it does to `compute`. A trivial case:

            >>> compute(lambda:None)
            compute(None)

        `compute` can also take additional arguments:

            >>> compute(lambda:None, reset=0)
            compute(None)

        But we want to be able to use it as a decorator and still pass
        additional arguments, so the following should work:

            >>> compute(reset=0)(lambda:None)
            compute(None)

        It worked because the first call was intercepted in
        Parametrized.__new__ and returned the lambda that acts as a
        real decorator:

            >>> compute(reset=0)
            <function <lambda> at 0x00C343B0>

    """
    def __new__(cls, *args, **kw):
        if not args:
            return lambda _: cls(_, **kw)
        return object.__new__(cls)




class attr(ResetAttribute, MakeAttribute):
    cell_cls = Value
    resetting_cell_cls = ResetValue
    def __init__(self, init=NOT_GIVEN, reset=NOT_GIVEN, make=None):
        self.init_reset(init, reset)
        self.make = make
        self.check_make()

    @classmethod_func
    def multi(cls, **attrs):
        cls._make_multi(sys._getframe(2), attrs)


class writer(RuleDecorator, ValueAttribute):
    optional = False #@@ this needs consideration / explanation
    def make_cell(self, ob):
        write = bind(self.rule, ob)
        return WriterRule(write)

    def __get__(self, ob, type):
        if type is None:
            return self
        raise AttributeError("Don't read the writer values from anywhere but the rule")

def attrs_reset(**attrs):
    attr._make_multi(sys._getframe(1), attrs, arg='reset')



attr.writer = writer
attrs = attr.multi
attrs.reset = attrs_reset



class compute(RuleDecorator, ResetAttribute, Parametrized):
    cell_cls = ComputeRule
    #cell_cls = NewComputeRule
    resetting_cell_cls = ResetComputeRule
    def __init__(self, rule, reset=NOT_GIVEN, force_reset=NOT_GIVEN, write_to=None):
        self.rule = rule
        self.write_to = write_to
        if force_reset is not NOT_GIVEN:
            if reset is not NOT_GIVEN:
                raise TypeError("Don't use both reset= and force_reset= arguments")
            reset = force_reset
            self.force_reset = True
        else:
            self.force_reset = False
        self.init_reset(NOT_GIVEN, reset)

    def make_cell(self, ob):
        rule = bind(self.rule, ob)
        if self.force_reset:
            return self.factory(rule, self.initial_value(ob), force_reset=True)
        else:
            return self.factory(rule, self.initial_value(ob))

    def __set__(self, ob, value):
        if self.write_to is None:
            super(compute, self).__set__(ob, value)
        else:
            setattr(ob, self.write_to, value)







#@@ const(make=)

class make(MakeAttribute, Parametrized):
    factory = Constant
    value = NOT_GIVEN

    def __init__(self, make):#, optional=True):
        self.make = make
        #self.optional = optional





class track(RuleDecorator, MakeAttribute, Parametrized):
    optional = False
    def __init__(self, rule, make=None, init=NOT_GIVEN, writable=False):
        self.rule = rule
        self.factory = WritableCyclicRule if writable else CyclicRule
        self.make = make
        self.value = init
        self.check_make()

    def set_name(self, name):
        super(track, self).set_name(name)
        if self.make is not None:
            self.make = named_lambda(self.make, name)

    @classmethod_func
    def multi(cls, **attrs):
        cls._make_multi(sys._getframe(2), attrs)

_tm = track.multi
track.attrs = staticmethod(_tm)
_tm.writable = lambda **attrs: track._make_multi(sys._getframe(1), attrs, writable=True)






class todo(CellAttribute):
    def __init__(self, rule):
        self.rule = rule

    def make_cell(self, ob):
        return TodoValue(bind(self.rule, ob))

    @property
    def future(self):
        return property(lambda ob: get_cell(ob, self.__name__).get_future())




class maintain(ListenerAttribute):
    factory = MaintainRule


#class perform(ListenerAttribute):
#    factory = Performer



