from __future__ import absolute_import
import sys
from mext.reaction import *
from mext.reaction.tasks import *

__all__ = [
    'Result', 'Failure', 'get_result',
    'Deferred', 'task_method', 'call_deferred',
]




def get_result(__f, *args, **kw):
    """
        Return a result of calling `__f(*args, **kw)`
        as a `Result` or `Failure`. This never throws exceptions.
    """
    try:
        return Result(__f(*args, **kw))
    except:
        return Failure()

# Result and Failure are callables -- call to get the
# pending result / exception.
# Both have .exc_info attr.

def Result(value):
    r = lambda: value
    r.exc_info = None
    return r

def Failure(exc=None):
    if exc is None:
        exc = sys.exc_info()
    if isinstance(exc, tuple):
        def reraise():
            raise exc[0], exc[1], exc[2]
        reraise.exc_info = exc
    else:
        assert isinstance(exc, Exception)
        def reraise():
            raise exc
        reraise.exc_info = type(exc), exc, None
    return reraise






class Deferred(Component):
    """
        Deferred result.
        To get from @task, do `val = yield dfd`.
        To set, do `dfd.set(val)` or `.set_error(exc)`

        To wait for multiple deferreds, do:

            v1,v2,v3 = yield (d1|d2|d3)

        If any of them will raise an error, it will be propagated immediately,
        w/o waiting for the rest of them.
    """
    __slots__ = ('__cells__', '__weakref__')
    result = attr()
    @atomic
    def set(self, result):
        assert self.result is None
        self.result = Result(result)

    @atomic
    def set_error(self, exc=None):
        assert self.result is None
        self.result = Failure(exc)

    @atomic
    def set_result(self, result):
        assert self.result is None
        self.result = result

    def next(self):
        if self.result is None:
            return PAUSE
        return Return(self.result())



    def __or__(self, other):
        if isinstance(other, Deferred):
            return wait_all([self, other])
        elif isinstance(other, wait_all):
            return wait_all([self] + other.dfds)
        else:
            raise TypeError




class wait_all(object):
    """
        Helper object created when doing `(deferred1 | deferred2)`
        Acts like a Deferred, but returning a list of results of wrapped deferreds.
    """
    def __init__(self, dfds):
        for dfd in dfds:
            assert isinstance(dfd, Deferred)
        self.dfds = dfds

    #@@ can't just use dfd.next(), beacuse we need to be
    # able to pull the result more than once
    def next(self):
        ready = True
        r = []
        for deferred in self.dfds:
            if deferred.result is not None:
                # this can throw an exc, which is intended
                r.append(deferred.result())
            else:
                ready = False
        return r if ready else PAUSE

    def __repr__(self):
        return self.__class__.__name__ + repr(tuple(self.dfds))

    def __or__(self, other):
        if isinstance(other, Deferred):
            return wait_all(self.dfds + [other])
        elif isinstance(other, wait_all):
            return wait_all(self.dfds + other.dfds)
        else:
            raise TypeError

    def __ior__(self, other):
        if isinstance(other, Deferred):
            self.dfds.append(other)
            return self
        elif isinstance(other, wait_all):
            self.dfds += other.dfds
            return self
        else:
            raise TypeError


## def _flatten(items, r=None):
##     if r is None:
##         r = []
##     for item in items:
##         if isinstance(item, (tuple, list)):
##             _flatten(item, r)
##         else:
##             r.append(item)
##     return r
##





class task_method(object):
    """
        Decorate generator to return a Deferred subclass.
        The generator is treated as @task, with the difference that
        it can have arguments, will start only when called,
        and will have as many instances as many times it was called.

            (callable: (..) -> generator) ->
                callable: (..) -> Deferred
    """

    def __init__(self, f):
        self.f = f

    def __repr__(self):
        return '<%s(%r)>' % (self.__class__.__name__, self.f)

    def __get__(self, ob, type):
        if ob is None or not hasattr(self.f, '__get__'):
            return self
        return self.__class__(self.f.__get__(ob, type))

    def call_deferred(self, *args, **kw):
        return StandaloneTask(self.f(*args, **kw))

    __call__ = call_deferred




class StandaloneTask(Deferred):#, AddOn):
    __slots__ = 'gen'
    cache = set() # this is intentionally global

    def __init__(self, gen):
        self.gen = gen
        self.cache.add(self)

    @task
    def run(self):
        #assert is_main_thread()
        try:
            r = yield self.gen
            self.set(r)
        except:
            self.set_error()
            if not has_listeners(self, 'result'):
                #import traceback; traceback.print_stack()
                # we don't want errors to pass silently
                # maybe we should have an option for cases when
                # we actually do want to silence them
                raise
        finally:
            #self.cache.discard(self)
            self.cache.remove(self)









def call_deferred(__f, *args, **kw):
    """
        (__f, *args, **kw) -> Deferred

        If `__f` has support for deferred calls (.call_deferred),
        (like task_methods do) it will be used, if not, it will
        be called immediately.
    """
    if hasattr(__f, 'call_deferred'):
        return __f.call_deferred(*args, **kw)
    else:
        return Deferred(result=get_result(__f, *args, **kw))
