from __future__ import absolute_import
from weakref import ref

__all__ = ['AbstractSubject', 'AbstractListener', 'Link']


class AbstractSubject(object):
    """
        Abstract base for objects that can be linked via ``Link`` objects
    """
    __slots__ = ('_gc_value')
    layer = 0
    can_reset = False # cells don't reset by default

    def __init__(self):
        self.next_listener = None
        super(AbstractSubject, self).__init__()

    def iter_listeners(self):
        """
            Yield the listeners of this subject
        """
        link = self.next_listener
        while link is not None:
            nxt = link.next_listener   # avoid unlinks breaking iteration
            ob = link()
            if ob is not None:
                yield ob
            link = nxt

class AbstractListener(object):
    """
        Abstract base for objects that can be linked via ``Link`` objects
    """
    __slots__ = ()
    layer = 0
    readonly = True # can a rule change anything but its own value?
    can_rerun = True # is it possible to retry this rule if necessary?
    is_listener = True

    def __init__(self):
        self.next_subject = None
        super(AbstractListener, self).__init__()

    def iter_subjects(self):
        """Yield the listeners of this subject"""
        link = self.next_subject
        while link is not None:
            nxt = link.next_subject   # avoid unlinks breaking iteration
            if link.subject is not None:
                yield link.subject
            link = nxt

    def dirty(self):
        """
            Mark the listener dirty and query whether it should be scheduled

            If a true value is returned, the listener should be scheduled.  Note
            that this method is allowed to have side-effects, but must be
            idempotent.
        """
        return True

    @classmethod
    def run(cls):
        """Take whatever action the listener is supposed to take"""
        raise NotImplementedError(cls)



class Link(ref):
    """Dependency link"""
    __slots__ = (
        'subject',
        'next_subject', 'prev_subject',
        'next_listener', 'prev_listener'
    )

    def __new__(cls, subject, listener):
        self = ref.__new__(Link, listener, _unlink_fn)
        self.subject = self.prev_listener = subject
        self.prev_subject = None    # listener link is via weak ref
        nxt = self.next_subject = listener.next_subject
        if nxt is not None:
            nxt.prev_subject = self
        nxt = self.next_listener = subject.next_listener
        if nxt is not None:
            nxt.prev_listener = self
        listener.next_subject = self
        subject.next_listener = self
        return self

    def unlink(self):
        """Deactivate the link and remove it from its lists"""
        nxt = self.next_listener
        prev = self.prev_listener
        if nxt is not None:
            nxt.prev_listener = prev
        if prev is not None and prev.next_listener is self:
            prev.next_listener = nxt
        prev = self.prev_subject
        nxt = self.next_subject
        if nxt is not None:
            nxt.prev_subject = prev
        if prev is None:
            prev = self()   # get head of list
        if prev is not None and prev.next_subject is self:
            prev.next_subject = nxt
        self.subject = self.next_subject = self.prev_subject = None
        self.next_listener = self.prev_listener = None

_unlink_fn = Link.unlink
