from __future__ import with_statement, absolute_import
from copy import copy

from mext.reaction.stm import *
from mext.reaction.celltypes.base import *
from mext.reaction.celltypes.compute import *
from mext.reaction.celltypes.maintain import *

from mext.const import const

__all__ = ['ResetValue', 'ResetComputeRule', 'TodoValue', 'WriterRule']




class ResetBase(HasListenersBase):
    __slots__ = 'reset', '__weakref__'
    can_reset = True # this cell resets

    def __init__(self, value=None):
        super(ResetBase, self).__init__()
        self.reset = value


class ResetValue(ResetBase): #WritableSubject
    __slots__ = ()
    def read(self):
        txn = ctrl.txn
        if txn is None:
            return self.reset
        txn.used(self)
        if txn.resetting:
            return self.reset
        else:
            return txn.transient.get(self, self.reset)

    def write(self, value):
        with ctrl.need_txn() as txn:
            if txn.resetting:
                if value != self.reset:
                    raise RuntimeError(
                        "During reset ResetValue (%r) can only be set to its reset value (%r), crule=%r"
                        % (self, self.reset, txn.crule)
                    )
            elif self in txn.set_by:
                # this value was already set in this txn
                set_by = txn.set_by[self]
                curval = txn.transient[self]
                if value != curval and set_by is not txn.crule:
                    raise InputConflict(curval, value, set_by, txn.crule)
                txn.change_key(txn.transient, self, value)
            else:
                # value wasn't set in this txn
                txn.set_key(txn.set_by, self, txn.crule)
                txn.set_key(txn.transient, self, value)
                if value != self.reset:
                    txn.changed(self)

    def run_reset(self):
        pass

    def __repr__(self):
        extra = ''
        if ctrl.txn and self in ctrl.txn.transient:
            val = ctrl.txn.transient[self]
            if val != self.reset:
                extra = ', reset=%r' % (self.reset,)
        else:
            val = self.reset
        return "%s(%r%s)" % (self.__class__.__name__, val, extra)


class TodoValue(ResetBase):
    """
        Value that logs changes for mutable data structures.
        Can be changed by multiple rules in the same txn unlike Value.
    """
    __slots__ = 'make', '_copy'

    def __init__(self, make, copy=copy):
        self.make = make
        self._copy = copy
        super(TodoValue, self).__init__(make())

    def read(self):
        txn = ctrl.txn
        assert txn, "TodoValue can only be read from txn"
        txn.used(self)
        if txn.resetting:
            return self.reset
        else:
            return txn.transient.get(self, self.reset)


    def write(self, value):
        raise AttributeError

    def get_future(self):
        """
            Get the 'future' (changeable) value
        """
        txn = ctrl.txn
        if not txn:
            raise RuntimeError("future can only be accessed from a transaction")
        if txn.resetting:
            raise RuntimeError("future cannot be accessed during reset")
        if txn.crule.readonly:
            raise RuntimeError("Readonly rules may not access future values")

        txn.changed(self)
        if self in txn.transient:
            value = txn.transient[self]
            last_reader = txn.set_by[self]
            if txn.crule is not last_reader:
                value = self._copy(value)
                txn.change_key(txn.transient, self, value)
                txn.change_key(txn.set_by, self, txn.crule)
        else:
            value = self.make()
            txn.set_key(txn.transient, self, value)
            txn.set_key(txn.set_by, self, txn.crule)
        return value


    def run_reset(self):
        pass

    def __repr__(self):
        extra = ', make=%r' % self.make
        if ctrl.txn and self in ctrl.txn.transient:
            val = ctrl.txn.transient[self]
            #if val != self.reset:
            #    extra += ', reset=%r' % (self.reset,)
        else:
            val = self.reset
        return "%s(%r%s)" % (self.__class__.__name__, val, extra)


class ResetComputeRule(ResetBase, AutoDisconnectMixin, AbstractListener):
    """
        @compute(reset=..) implementation.
        Only listens to its subjects if has listeners itself.
        Cannot access its old value.
    """
    __slots__ = 'rule', 'force_reset', 'layer', 'next_subject'

    def __init__(self, rule, value, force_reset=False):
        self.rule = rule
        self.layer = 0
        self.force_reset = force_reset
        super(ResetComputeRule, self).__init__(value)

    #@@
    #def run(self):
    #    super(ResetComputeRule, self).run()
    #    trl.txn.schedule(self.disconnect_cell)

    def read(self):
        txn = ctrl.txn
        if txn is None:
            return self.reset
        elif txn.crule is self:
            raise RuntimeError("@compute rules cannot use their own value")
        else:
            txn.used(self)
            if txn.resetting:
                #@@ ! if first access happens while resetting we'll never
                # learn what we depend on
                return self.reset
            elif self in txn.transient:
                return txn.transient[self]
            elif self.next_listener is None: # @@ 'connected' is not the same as 'has listeners'
                assert self not in txn.queues.get(self.layer, ())
                with txn._replace_crule(self):
                    value = self.rule()
                    txn.set_key(txn.transient, self, value)
                    if value != self.reset:
                        txn.changed(self)
                    if txn.rule_reads:
                        txn._process_reads()
                    else:
                        txn.effect(self.become_constant)
                #if not txn.crule.is_listener:
                #    txn.effect(self.disconnect)
                return value
            else:
                return self.reset

    def run(self):
        txn = ctrl.txn
        value = self.rule()
        txn.set_key(txn.transient, self, value)
        if value != self.reset:
            txn.changed(self)
        if not txn.rule_reads:
            txn.effect(self.become_constant)

    def run_reset(self):
        if self.force_reset:
            return
        txn = ctrl.txn
        with txn._replace_crule(self):
            value = self.rule()
            txn._process_reads()
        if value != self.reset:
            raise InputConflict(
                "When resetting, resetting rule (%r) must change to reset value (%r), got %s"
                % (self, self.reset, value)
            )
        # @@?
        #if not txn.rule_reads:
        #    txn.effect(self.become_constant)


    def become_constant(self):
        assert self not in ctrl.txn.transient
        super(ResetComputeRule, self).become_constant()

    def _const_class(self):
        return ResetComputeConstant

    def __repr__(self):#@@
        return "%s(%r)" % (type(self).__name__, self.rule)





NO_WRITE = const()


class WriterRule(ResetValue):
    """
        A combined ResetValue and MaintainRule that writes it
    """
    __slots__ = 'do_write', 'write_rule'
    def __init__(self, do_write):
        super(WriterRule, self).__init__(NO_WRITE)
        self.do_write = do_write
        self.write_rule = MaintainRule(self.rule) #@@ custom rule w/ empty on_reset

    def rule(self):
        val = self.read()
        if val is not NO_WRITE:
            self.do_write(val)

    @property
    def has_value(self):
        return self.read() is not NO_WRITE



class ResetComputeConstant(ConstantMixin, ResetComputeRule):
    __slots__ = ()
    value = ResetComputeRule.reset
    def read(self):
        return self.reset
