from __future__ import with_statement, absolute_import
import sys
#from mext.reaction import *
from mext.reaction.stm import *
from mext.reaction.eventloop import *
from mext.reaction.time import *
from mext.reaction.celltypes.maintain import PureListener

from mext.context import State
from mext.const import const

from operator import itemgetter

#@@? rename Pause -> PAUSE
__all__ = ['TaskCell', 'PAUSE', 'FLUSH', 'YIELD', 'Return', 'async_sleep']

PAUSE = const() # Pause until a subject change wakes us up
FLUSH = const() # process the changes w/o exiting to eventloop
YIELD = const() # return control to eventloop, but request to wake up again ASAP

class Return(tuple):
    """
        Wrapper for yielding a value from a task.

        Simple `yield value` usually works, but sometimes value has a .next attribute
        and that gets it to be treated as subtask. For that reason unless you know
        for sure that the result value is not an iterator use `yield Return(value)`
    """
    __slots__ = ()
    value = property(itemgetter(0))
    def __new__(cls, value):
        return tuple.__new__(cls, (value,))
    def __repr__(self):
        return 'Return(%s)' % repr(self[0])


scheduled_tasks = set()

class TaskCell(PureListener):
    """
        Cell that manages a generator-based task
    """
    __slots__ = 'run', 'layer', '_queue'
    readonly = False
    can_rerun = False

    def __init__(self, func, queue=None):
        PureListener.__init__(self, func)
        task = func()
        assert hasattr(task, 'next'), "task rule must return an iterable"
        self.run = self._stepper(task).next
        self.layer = 0
        self._queue = queue or EventQueue.get()
        with ctrl.need_txn():
            self.schedule()

    def schedule(self):
        ctrl.txn.effect(self._schedule)

    #@effect_method
    def _schedule(self):
        if self not in scheduled_tasks:
            scheduled_tasks.add(self)
            self._queue.call(self.do_run)

    def dirty(self):
        self.schedule()
        return False

    def run_reset(self):
        raise TypeError("%r depends on transient value even though it can never see it" % self)

    def _stepper(self, task):
        result = None
        error = False
        #state = State.get()
        # this is a hack, required by incorrect check in mext.context
        state = State.child().swap()
        STACK = [task]
        PUSH_TASK = STACK.append
        POP_TASK = STACK.pop
        while STACK:
            try:
                subtask = STACK[-1]
                if error:
                    if hasattr(subtask, 'throw'):
                        error = False
                        result = subtask.throw(*sys.exc_info())
                    else:
                        # try to throw in parent subtask
                        POP_TASK()
                        continue
                elif result is None:
                    result = subtask.next()
                elif hasattr(subtask, 'send'):
                    result = subtask.send(result)
                else:
                    #if subtask returns a value it should not be discarded
                    msg = "Subtask returned %r, but parent task (%r) has no .send()" % (result, subtask)
                    raise AssertionError(msg)
            except StopIteration:
                # subtask is done.
                POP_TASK()
                if STACK:
                    # ensure (yield iter([])) is None
                    result = None
                else:
                    # top-level task is done
                    # this is necessary because if we used return, .run() would raise StopIteration
                    self.run = lambda: None
                    state.swap()
                    yield
                    raise AssertionError("Execution should never get this far")
            except:
                POP_TASK()
                error = True
            else:
                if (result is PAUSE
                    or result is FLUSH
                    or result is YIELD
                ):
                    if result is YIELD:
                        self.schedule()
                    elif result is FLUSH:
                        ctrl.txn.effect(lambda: ctrl.txn.run_rule(self))
                    elif not ctrl.txn.rule_reads:
                        exc = AssertionError("tasks should not try to PAUSE w/o any subjects to wake them up")
                        try:
                            if hasattr(subtask, 'throw'):
                                subtask.throw(AssertionError, exc, None)
                        finally:
                            state.swap()
                        raise exc
                    # ensure (yield PAUSE) is None
                    result = None
                    state = state.swap() # restore state, get task state
                    yield
                    state = state.swap() # get state, restore task state
                elif hasattr(result, 'next'):
                    # yielded value has .next -- use as subtask
                    # and make sure we try .next not .send as first call
                    PUSH_TASK(result)
                    result = None
                else:
                    if isinstance(result, Return):
                        result = result.value
                    POP_TASK()

        state.swap()
        if error:
            # scanned through a stack of subtasks none having .throw,
            # or error happened in top-level task
            raise
        else:
            # top-level task tried to return a value
            raise RuntimeError("TaskCell rules should only yield PAUSE, FLUSH, YIELD and subtasks, got %r" % result)

    def do_run(self):
        scheduled_tasks.remove(self)
        with ctrl.new_txn() as txn:
            txn.run_rule(self)

def async_sleep(wakeup):
    if not isinstance(wakeup, PointInTime):
        wakeup = Time[wakeup]
    while not wakeup:
        yield PAUSE

#@@ add min_pause(secs),
# would work like async_sleep, but NOT wake up until "dirty"
# that is, if scheduled before `secs` passed, it will wake up when they elapse
# otherwise it will sleep for for `secs` and then proceed sleeping in a `yield PAUSE` mode
# as if the clock dependency wasn't there
# Good for rate-limited UI updates
