from __future__ import with_statement, absolute_import
from mext.reaction.celltypes.rulebase import *


__all__ = [
    'Rule',
    'CyclicRule', 'WritableCyclicRule',
    'ConstantRule'
]

#@@ consider requiring rule to be no-op on first run if init= value was given to @track

class Rule(RuleBase):
    __slots__ = ()

    def __init__(self, rule, value=None):
        super(Rule, self).__init__(rule)
        with ctrl.init_txn() as txn:
            with txn._replace_crule(self):
                newval = self.get_newval(value)
                txn.set_key(txn.writes, self.key, newval)
                txn._process_reads()
            if self.next_subject is None:
                txn.effect(self.become_constant)


    def read(self):
        txn = ctrl.txn
        key = self.key
        if not txn:
            return ctrl.state[key]

        assert txn.crule is not self
        txn.used(self)
        if self in txn.set_by:
            return txn.writes[key]
        else:
            txn.note_read(key)
            return txn.snapshot[key]

    def get_newval(self, oldval):
        return self.rule()

    def run(self):
        txn = ctrl.txn
        key = self.key
        if self in txn.set_by:
            oldval = txn.writes[key]
        else:
            oldval = txn.snapshot[key]
            txn.note_read(key)

        newval = self.get_newval(oldval)

        if newval != oldval:
            txn.changed(self)

        txn.change_key(txn.set_by, self, self)
        txn.change_key(txn.writes, key, newval)

        if not txn.rule_reads:
           txn.effect(self.become_constant)

    def run_reset(self):
        txn = ctrl.txn
        oldval = txn.writes[self.key]
        with txn._replace_crule(self):
            newval = self.get_newval(oldval)
            txn._process_reads()
        if newval != oldval:
            raise InputConflict("%r can't change during reset (%r != %r)"
                % (self, newval, oldval)
            )
        #@@?
        #if not txn.rule_reads:
        #   txn.effect(self.become_constant)


class CyclicRule(Rule): #_const_class, __repr__
    """
        Implementation of @track rules.
        Always runs after a subject change, even if there are no listeners.
        Runs in readonly mode, the rule gets its own previous value
        as an additional argument.
    """
    __slots__ = ()
    get_newval = RuleBase.rule






class WritableCyclicRule(CyclicRule):
    """
        @track(writable=True) implementation.
        Acts in the same way CyclicRule does, but can also be set by a @maintain rule
        or from top-level (in @atomic or with new_txn()).
    """
    __slots__ = ()

    def become_constant(self):
        # we can never become Constant
        pass


    def write(self, value):
        with ctrl.need_txn() as txn:
            key = self.key
            if self in txn.set_by:
                if txn.set_by[self] is not txn.crule:
                    # this value has already run or was set by someone else
                    #@@ this doesn't play nice with instantiating a rule and immediately setting it
                    #@@ with new init code it should. test that
                    raise InputConflict(
                        "%s tries to set %s to %s which already was set or ran"
                        % (txn.crule, self, value)
                    )
            elif self in txn.queues.get(self.layer, ()):
                raise InputConflict(
                    "%s tries to set %s to %s which is already scheduled"
                    % (txn.crule, self, value)
                )

            txn.change_key(txn.writes, key, value)
            txn.set_key(txn.set_by, self, txn.crule)

            txn.note_read(key)
            if value != txn.snapshot[key]:
                txn.changed(self)

            # @@ this is almost good enough, but the check needs to see the state as it was pre-reset
            #txn.reset_queue.add(self)
            # @@ the error message could be confusing,
            # make sure the error message is:
##                 raise InputConflict(
##                     "When cyclic cell is written its rule should be a no-op for the value (%r != %r)"
##                     % (processed_value, value)
##                 )
##




#@@ make this a subject
##     @property
##     def was_set(self):
##         """True if value was set during this recalc"""
##         assert ctrl.txn, "was_set can only be accessed inside txn"
##         ctrl.txn.used(self)
##         return ctrl.txn.set_by.get(self, self) is not self

