"""
Collections to be used with `mext.reaction`
Unlike standard Python containers Reaction collections notify
about their changes and make those changes happen atomically --
either all or no changes are visible depending on success or failure of
the transaction.

Some collection classes closely resemble standard ones:
List<=>list, Dist<=>dict, Set<=>set and support most of standard operations
except for those that are not allowed because they conflict with
transactional properties.
"""

from __future__ import with_statement, absolute_import
import sys, UserDict, UserList
from bisect import bisect_left
from types import MethodType
from mext.reaction import *

if sys.version <= '2.5':
    import sets
else:
    import warnings
    _filter = ('ignore', None, DeprecationWarning, None, 27) # lineno for 'import sets'
    warnings.filters.append(_filter)
    try:
        import sets
    finally:
        warnings.filters.remove(_filter)


__all__ = [
    'Pipe', 'Dict', 'List', 'Set',
    'SortedSet', 'Observing', 'Hub'
]

set_like = set, frozenset, sets.BaseSet
dictlike = (dict,) + set_like


class Pipe(Component):
    """
       Allow one or more writers to send data to zero or more readers
    """

    output = todo(list)
    input  = output.future

    @atomic
    def append(self, value):
        self.input.append(value)

    @atomic
    def extend(self, sequence):
        self.input.extend(sequence)

    def __iter__(self):
        return iter(self.output)

    def __contains__(self, value):
        return value in self.output

    def __len__(self):
        return len(self.output)

    def __repr__(self):
        return repr(self.output)




class Dict(UserDict.IterableUserDict, Component):
    """
        Dictionary-like object that recalculates observers when it's changed

        The ``added``, ``changed``, and ``deleted`` attributes are dictionaries
        showing the current added/changed/deleted contents.  Note that ``changed``
        may include items that were set as of this recalc, but in fact have the
        same value as they had in the previous recalc, as no value comparisons are
        done!

        You may observe these attributes directly, but any rule that reads the
        dictionary in any way (e.g. gets items, iterates, checks length, etc.)
        will be recalculated if the dictionary is changed in any way.

        Note that this operations like pop(), popitem(), and setdefault() that both
        read and write in the same operation are NOT supported, since reading must
        always happen in the present, whereas writing is done to the future version
        of the dictionary.
    """

    todo.attrs(
        deleted = set,
        changed = dict,
    )

    to_change = changed.future
    to_delete = deleted.future

    def __init__(self, other=(), **kw):
        Component.__init__(self)
        if other or kw:
            with ctrl.txn.new_root():
                self.data.update(other)
                self.data.update(kw)

    def copy(self):
        return self.__class__(self.data)

    def get(self, key, failobj=None):
        return self.data.get(key, failobj)

    def __hash__(self):
        raise TypeError


    @track(make=dict)
    def data(self, data):
        deleted, changed = self.deleted, self.changed
        if deleted or changed:
            data = data.copy()
            map(data.__delitem__, deleted)
            data.update(changed)
        return data

    @atomic
    def __setitem__(self, key, item):
        self.to_delete.discard(key)
        self.to_change[key] = item

    @atomic
    def __delitem__(self, key): #@@ remove .data dep
        if key in self.to_delete:
            raise KeyError(key)
        if key in self.to_change:
            del self.to_change[key]
            if key in self.data:
                self.to_delete.add(key)
        elif key not in self.data:
            raise KeyError(key)
        else:
            self.to_delete.add(key)




    @atomic
    def clear(self): #@@ remove .data dep
        self.to_change.clear()
        self.to_delete.update(self.data)

    @atomic
    def update(self, d=(), **kw):
        if d:
            if kw:
                d = dict(d);  d.update(kw)
            elif not hasattr(d, 'iteritems'):
                d = dict(d)
        else:
            d = kw
        to_change = self.to_change
        to_delete = self.to_delete
        for k, v in d.iteritems():
            to_delete.discard(k)
            to_change[k] = d[k]

    def setdefault(self, key, failobj=None):
        """setdefault() is disallowed because it 'reads the future'"""
        raise InputConflict("Can't read and write in the same operation")

    def pop(self, key, *args):
        """The pop() method is disallowed because it 'reads the future'"""
        raise InputConflict("Can't read and write in the same operation")

    def popitem(self):
        """The popitem() method is disallowed because it 'reads the future'"""
        raise InputConflict("Can't read and write in the same operation")




def _list_method(name):
    def method(self, *args):
        self.changed = True
        future_method = getattr(self.future, name)
        future_method(*args) # no return value
    method.__name__ = name
    return atomic(method)



class List(UserList.UserList, Component):
    """
        List-like object that recalculates observers when it's changed

        The ``changed`` attribute is True whenever the list has changed as of the
        current recalculation, and any rule that reads the list in any way (e.g.
        gets items, iterates, checks length, etc.) will be recalculated if the
        list is changed in any way.

        Note that this type is not efficient for large lists, as a copy-on-write
        strategy is used in each recalcultion that changes the list.  If what you
        really want is e.g. a sorted read-only view on a set, don't use this.
    """

    if hasattr(UserList.UserList, '__metaclass__'):
        class __metaclass__(Component.__metaclass__, UserList.UserList.__metaclass__):
            pass

    data = attr(make=list)
    updated = todo(lambda self: self.data[:])
    future  = updated.future
    changed = attr(reset=False)

    def __init__(self, other=(), **kw):
        Component.__init__(self, **kw)
        if other:
            with ctrl.txn.new_root():
                self.data[:] = other

    @maintain
    def _update_data(self):
        if self.changed:
            self.data = self.updated[:]

    # hackish but saves a lot of code
    for name in [
        '__setitem__', '__delitem__',
        '__setslice__', '__delslice__',
        'append', 'insert', 'extend',
        'remove', 'reverse'
    ]:
        locals()[name] = _list_method(name)
    del name

    @atomic
    def __iadd__(self, other):
        self.changed = True
        self.future.extend(other)
        return self

    @atomic
    def __imul__(self, n):
        self.changed = True
        self.future[:] = self.future * n
        return self


    @atomic
    def sort(self, *args, **kw): # separate for **kw support
        self.changed = True
        self.future.sort(*args, **kw)

    def pop(self, i=-1):
        """The pop() method isn't supported, because it 'reads the future'"""
        raise InputConflict("Can't read and write in the same operation")

    def __hash__(self):
        raise TypeError


















class Set(sets.Set, Component):
    """
        Mutable set that recalculates observers when it's changed

        The ``added`` and ``removed`` attributes can be watched for changes, but
        any rule that simply uses the set (e.g. iterates over it, checks for
        membership or size, etc.) will be recalculated if the set is changed.
    """
    _added = todo(set)
    _removed = todo(set)
    added = property(lambda self: self._added)
    removed = property(lambda self: self._removed)
    to_add = _added.future
    to_remove = _removed.future

    def __init__(self, iterable=None, **kw):
        """Construct a set from an optional iterable."""
        Component.__init__(self, **kw)
        if iterable is not None:
            # we can update self._data in place, since no-one has seen it yet
            with ctrl.txn.new_root():
                sets.Set._update(self, iterable)

    @track(make=dict)
    def _data(self, data):
        """The dictionary containing the set data."""
        added, removed = self.added, self.removed
        if added or removed:
            data = data.copy()
            assert not (added & removed) # no intersection
            pop = data.pop
            map(data.__delitem__, removed)
            data.update(dict.fromkeys(self.added, True))
        return data

# this does not work because stdlib sets.BaseSet wants ._data to be a dict
##     @track(make=set) #@@ set
##     def _data(self, data):
##         if self.removed:
##             mark_dirty()
##             data -= self.removed
##             ctrl.txn.on_undo(data.update, self.removed)
##         if self.added:
##             mark_dirty()
##             data |= self.added
##             ctrl.txn.on_undo(data.difference_update, self.added)
##         return data
##
##


    def __setstate__(self, data):
        self.__init__(data[0])




    def _binary_sanity_check(self, other):
        # Check that the other argument to a binary operation is also
        # a set, raising a TypeError otherwise.
        if not isinstance(other, set_like):
            raise TypeError, "Binary operation only permitted between sets"

    def pop(self):
        """The pop() method isn't supported, because it 'reads the future'"""
        raise InputConflict("Can't read and write in the same operation")

    #@@ in fact we should allow this and similar methods from non-listeners
    @atomic
    def discard(self):
        raise InputConflict

    @atomic
    def _update(self, iterable):
        to_remove = self.to_remove
        add = self.to_add.add
        for item in iterable:
            if item in to_remove:
                to_remove.remove(item)
            else:
                add(item)

    @atomic
    def add(self, item):
        """Add an element to a set (no-op if already present)"""
        if item in self.to_remove:
            self.to_remove.remove(item)
        elif item not in self._data: # <- this is bs, we should add it anyway
            self.to_add.add(item)

    @atomic
    def remove(self, item):
        """Remove an element from a set (KeyError if not present)"""
        if item in self.to_add:
            self.to_add.remove(item)
        elif item in self._data and item not in self.to_remove:
            self.to_remove.add(item)
        else:
            raise KeyError(item)


    @atomic
    def clear(self):
        """Remove all elements from this set."""
        self.to_remove.update(self)
        self.to_add.clear()

    def __ior__(self, other):
        """Update a set with the union of itself and another."""
        self._binary_sanity_check(other)
        self._update(other)
        return self

    def __iand__(self, other):
        """Update a set with the intersection of itself and another."""
        self._binary_sanity_check(other)
        self.intersection_update(other)
        return self

    @atomic
    def difference_update(self, other):
        """Remove all elements of another set from this set."""
        data = self._data
        to_add, to_remove = self.to_add, self.to_remove
        for item in other:
            if item in to_add: to_add.remove(item)
            elif item in data: to_remove.add(item)

    @atomic
    def intersection_update(self, other):
        """Update a set with the intersection of itself and another."""
        to_remove = self.to_remove
        to_add = self.to_add
        self.to_add.intersection_update(other)
        other = to_dict_or_set(other)
        for item in self._data:
            if item not in other:
                to_remove.add(item)
        return self



    @atomic
    def symmetric_difference_update(self, other):
        """Update a set with the symmetric difference of itself and another."""
        data = self._data
        to_add = self.to_add
        to_remove = self.to_remove
        for elt in to_dict_or_set(other):
            if elt in to_add:
                to_add.remove(elt)      # Got it; get rid of it
            elif elt in to_remove:
                to_remove.remove(elt)   # Don't got it; add it
            elif elt in data:
                to_remove.add(elt)      # Got it; get rid of it
            else:
                to_add.add(elt)         # Don't got it; add it


def to_dict_or_set(ob):
    """Return the most basic set or dict-like object for ob
    If ob is a sets.BaseSet, return its ._data; if it's something we can tell
    is dictlike, return it as-is.  Otherwise, make a dict using .fromkeys()
    """
    if isinstance(ob, sets.BaseSet):
        return ob._data
    elif not isinstance(ob, dictlike):
        return dict.fromkeys(ob)
    return ob









class Hub(Component):
    """Pub/sub messaging"""

    def get(self, *rule):
        """Return messages of same length with matching non-``None`` args"""
        return self._queries[rule].read()

    def put(self, *row):
        """Send a message to any active subscribers"""
        self._inputs.append(row)

    _inputs = make(Pipe)
    _index = make(dict)     # {(pos,value): set([rule])}

    @active_cellcache(reset=())
    def _queries(self, rule, conn):
        if conn:
            key = None
            for pv in enumerate(rule):
                if pv[1] is not None:
                    key = pv
            if key is not None:
                ind = self._index.setdefault(key, set())
                ind.add(rule)
        else:
            index = self._index
            for pv in enumerate(rule):
                if pv in index:
                    index[pv].remove(rule)

    @maintain
    def _notify(self):
        """Send received rows to observers"""

        inputs = self._inputs
        if not inputs:
            return      # nothing to see, move along...

        index = self._index
        matches = {}

        for row in inputs:
            for pv in enumerate(row):
                if pv not in index:
                    continue
                for rule in index[pv]:
                    if len(rule)==len(row):
                        for v1,v2 in zip(rule, row):
                            if v1!=v2 and v1 is not None:
                                break
                        else:
                            matches.setdefault(rule,[]).append(row)
        if matches:
            queries = self._queries
            for rule in matches:
                queries[rule].write(matches[rule])










class Observing(Component):
    """Monitor a set of keys for changes"""

    lookup_func = attr(lambda x:x) #@@ changing lookup_func will not trigger _watching recalc
    keys = None

    def __init__(self, keys=None, **kw):
        if keys is None:
            keys = Set()
        self.keys = keys
        Component.__init__(self, **kw)

    @track(make=dict)
    def _watching(self, cells):
        added, removed = self.keys.added, self.keys.removed
        if added or removed:
            cells = cells.copy()
            map(cells.__delitem__, removed)
            lookup = self.lookup_func
            cells.update(
                (k, Rule(MethodType(lookup, k, type(k))))
                    for k in added
            )
        return cells

    @track(init=({}, {}))
    def watched_values(self, (_, old)):
        return old, dict([(k, v.read()) for k,v in self._watching.iteritems()])

    #@track(reset={})
    @compute(force_reset={})
    def changes(self):
        # we want to have .watched_values as a dependency even during reset
        old, current = self.watched_values
        changes = {}
        if old != current:
            for k,v in current.iteritems():
                if k not in old or v!=old[k]:
                    changes[k] = v, old.get(k, v)
        return changes








class SortedSet(Component):
    """Represent a set as a list sorted by a key"""

    attrs(
        sort_key  = lambda x:x,  # sort on the object
        reverse = False,
        items = None,
        old_key = None,
        old_reverse = None
    )
    def __init__(self, data, **kw):
        with ctrl.txn.new_root():
            super(SortedSet, self).__init__(data=data, **kw)

    data = attr(make=Set)
    changes = attr(reset=[])

    def __getitem__(self, key):
        if self.reverse:
            key = -(key+1)
        return self.items[int(key)][1]

    def __len__(self):
        return len(self.items)

    @maintain
    def state(self):
        key, reverse = self.sort_key, self.reverse
        data = self.items
        if key != self.old_key or reverse != self.old_reverse:
            if data is None or key != self.old_key:
                data = [(key(ob),ob) for ob in self.data]
                data.sort()
                self.items = data
            size = len(self.data)
            self.changes = [(0, size, size)]
            self.old_key = key
            self.old_reverse = reverse
        else:
            self.changes = self.compute_changes(key, data, reverse)

    def __repr__(self):
        return repr(list(self))


    def compute_changes(self, key, items, reverse):
        changes = [
            (key(ob), "+", ob) for ob in self.data.added] + [
            (key(ob), "-", ob) for ob in self.data.removed
        ]
        changes.sort()
        changes.reverse()
        lo = 0
        hi = old_size = len(items)
        regions = []
        for k, op, ob in changes:
            ind = (k, ob)
            if lo<hi and items[hi-1][0]>=ind:
                pos = hi-1    # shortcut
            if lo<hi and ind>=items[hi-1]:
                # shortcut
                if ind==items[hi-1]:
                    pos = hi-1
                else:
                    pos = hi
            else:
                pos = bisect_left(items, ind, lo, hi)

            if op=='-':
                del items[pos]
                if regions and regions[-1][0]==pos+1:
                    regions[-1] = (pos, regions[-1][1], regions[-1][2])
                else:
                    regions.append((pos, pos+1, 0))
            else:
                items.insert(pos, ind)
                if regions and regions[-1][0]==pos:
                    regions[-1] = (pos, regions[-1][1], regions[-1][2]+1)
                else:
                    regions.append((pos, pos, 1))
            hi=pos

        if reverse:
            return [(old_size-e, old_size-s, sz) for (s,e,sz) in regions[::-1]]
        return regions



#raise DeprecationWarning("%s is disabled" % __name__)

