from __future__ import absolute_import
from weakref import WeakValueDictionary
from UserDict import DictMixin
from types import MethodType

from mext.reaction.component.attrs import *

__all__ = ['cellcache', 'active_cellcache', 'lazymap']


class WeakDefaultDict(WeakValueDictionary):
    def __init__(self, missing):
        WeakValueDictionary.__init__(self)
        self.__missing__ = missing

    def __getitem__(self, key):
        try:
            return WeakValueDictionary.__getitem__(self, key)
        except KeyError:
            r = self[key] = self.__missing__(key)
            return r

class LazyMap(DictMixin):
    __slots__ = 'cells', 'factory', 'rule', 'value'
    def __init__(self, factory, rule, value):
        self.factory = factory
        self.rule = rule
        self.value = value
        self.cells = WeakDefaultDict(self.make_item)

    def make_item(self, key):
        key_rule = MethodType(self.rule, key, type(key))
        return self.factory(key_rule, self.value)

    def __getitem__(self, key):
        return self.cells[key].read()




class cellcache(ResetAttribute, MakeAttribute):
    cell_cls = Value
    resetting_cell_cls = ResetValue

    def __init__(self, make=None, init=NOT_GIVEN, reset=NOT_GIVEN):
        self.make = make
        self.init_reset(init, reset)
        self.check_make()

    def initial_value(self, ob, key):
        if self.value is NOT_GIVEN:
            if self.make is not None:
                make = bind(self.make, ob)
                return make(key)
            #elif self.rule is not None:
            return None
        return self.value

    def make_cell(self, ob):
        def make_cell(key):
            value = self.initial_value(ob, key)
            return self.factory(value)
        return WeakDefaultDict(make_cell)

    def __get__(self, ob, typ=None):
        if ob is None:
            return self
        return get_cell(ob, self.__name__, attr=self)


class active_cellcache(cellcache):
    connect = None

    def connector(self, func):
        """
            Decorate a method as providing a connect function for this cell
        """
        # this can't be merged with __call__, because non-resetting cellcaches
        # have separate make and connect rules
        if self.connect is not None:
            raise TypeError("%r already has a connector" % self)
        self.connect = func
        return func

    def __call__(self, func):
        """
            Set resetting cellcache connector. (To be used as a decorator)
        """
        assert self.resetting
        self.connector(func)
        return self

    def set_name(self, name):
        super(active_cellcache, self).set_name(name)
        if self.connect is None:
            raise TypeError("%r doesn't have a connector" % self)

    def make_cell(self, ob):
        connect = bind(self.connect, ob)
        def make_cell(key):
            value = self.initial_value(ob, key)
            key_connect = lambda s, conn: connect(key, conn)
            cell = self.factory(value)
            connector = Connector(cell, key_connect)
            return cell
        return WeakDefaultDict(make_cell)





class lazymap(compute):
    cell_cls = ComputeRule
    resetting_cell_cls = ResetComputeRule
    def __init__(self, rule, reset=NOT_GIVEN):
        self.rule = rule
        self.init_reset(NOT_GIVEN, reset)

    def make_cell(self, ob):
        value = self.initial_value(ob)
        return LazyMap(self.factory, bind(self.rule, ob), value)

    def __get__(self, ob, typ=None):
        if ob is None:
            return self
        return get_cell(ob, self.__name__, attr=self)


