from __future__ import with_statement, absolute_import
from mext.reaction import *
from mext.reaction.time import *
from mext.context import ServiceStub, State

__all__ = ['EventLoop', 'EventQueue']


class EventLoop(Component, ServiceStub):
    """
        Run an application event loop
    """
    _dont_delegate = ('run',)

    attrs(
        running = False,
        stop_requested = False,
    )

    class run(object):
        def __get__(self, ob, type):
            if ob is None:
                return type._run_cls
            else:
                return ob._run
    run = run()

    @classmethod
    def _run_cls(cls):
        if cls is not EventLoop:
            cls.activate()
        EventLoop._run()

    @atomic
    def stop(self):
        """Stop the event loop at the next opportunity"""
        assert self.running, "EventLoop isn't running"
        self.stop_requested = True

    def _run(self):
        """Loop updating the time and invoking requested calls"""
        with ctrl.new_txn():
            assert not self.running, "EventLoop is already running"
            assert not self.stop_requested
            self._setup()
            self.running = True
            EventQueue.set_eventloop(self)
        try:
            self._loop()
        finally:
            with ctrl.new_txn():
                self.running = False
                self.stop_requested = False
            self._teardown()

    def _setup(self):
        pass

    def _teardown(self):
        pass

    def _loop(self):
        """Subclasses should invoke their external loop here"""
        raise NotImplementedError

    def _arrange_wakeup(self):
        """
            Subclasses should get the EventQueue._wakeup() to get called in eventloop thread ASAP.
            Note: Must be safe to call this from a 'foreign' thread.
        """
        raise NotImplementedError







class EventQueue(Component, ServiceStub):
    _add_to_queue = todo(list)
    _to_call = _add_to_queue.future
    _wakeup_pending = attr(False)
    _eventloop = attr()

    @track(init=(), writable=True)
    def _call_queue(self, oldval):
        return oldval + tuple(self._add_to_queue)

    def set_eventloop(self, eventloop):
        assert eventloop is not None
        if self._eventloop is eventloop:
            return
        assert self._eventloop is None, "Can't change eventloops"
        #self._eventloop = eventloop #@@x
        replace_cell(self, '_eventloop', Constant(eventloop))

    @atomic
    def call(self, __f, *args, **kw):
        """
            Call `__f(*args, **kw)` at the next opportunity
        """
        self._to_call.append((__f, args, kw))

    @compute
    def _can_wakeup(self):
        return (self._eventloop is not None
            and self._eventloop.running
        )

    #@compute
    #def _should_wakeup(self):
    #    return self._eventloop.stop_requested or bool(self._call_queue)

    @maintain
    def _autowakeup(self):
        if (self._can_wakeup
            and not self._wakeup_pending
            and self._call_queue
            #and self._should_wakeup
        ):
            self._arrange_wakeup()

    @effect_method
    def _arrange_wakeup(self):
        if self._wakeup_pending:
            return
        self._eventloop._arrange_wakeup()
        self._wakeup_pending = True

    def _wakeup(self):
        self.flush(1, True)

    def flush(self, count=0, on_wakeup=False):
        """
            Execute the specified number of pending calls (0 for all)
        """
        with ctrl.new_txn():
            queue = self._split_queue(count)
            if on_wakeup:
                self._wakeup_pending = False
        for (f, args, kw) in queue:
            f(*args, **kw)

    @atomic
    def _split_queue(self, count):
        queue = self._call_queue
        if not queue:
            return ()
        count = count or len(queue)
        self._call_queue = queue[count:]
        return queue[:count]


EventQueue.activate()
