from __future__ import with_statement, absolute_import
import functools
from mext.reaction.stm.controllers import LocalController, DisabledTransaction
from mext.reaction.stm.transactions import Transaction, InputConflict, CircularityError
from mext.reaction.stm.stm_old import AbstractListener, AbstractSubject #@@x


__all__ = [
    'ctrl', 'atomic', 'per_txn',
    'effect', 'pure_effect', 'after_txn',
    'effect_method',
    'InputConflict', 'CircularityError',
    'AbstractListener', 'AbstractSubject',
    'DisabledTransaction',
]


ctrl = LocalController(Transaction)


def atomic(func):
    """
        Decorate `func` to always run inside a transaction.
    """
    @functools.wraps(func)
    def wrapped(*args, **kw):
        if ctrl.txn:
            return func(*args, **kw)
        else:
            with ctrl.new_txn():
                return func(*args, **kw)
    return wrapped

def effect(func, *args):
    """
        Queue `func(*args)` to be called if this transaction is successful.

        All such queued calls will be performed in new transaction in a context
        of a shared RootRule (that is, effects from different rules can set conflicting
        values without it being detected).
    """
    txn = ctrl.txn
    assert txn, "Effects are only possible from rules"
    _queue(txn.effects, func, args)

def after_txn(func, *args):
    """
        Same as effect(..), but called outside of txn.
        Useful for things that are likely to trigger *new* transactions (via `with ctrl.new_txn()`).
    """
    txn = ctrl.txn
    if txn is None:
        func(*args)
    else:
        _queue(txn.q_notxn, func, args)

def pure_effect(func, *args):
    """
        Same as effect(..) but called w/ DisabledTransaction
    """
    txn = ctrl.txn
    assert txn, "Effects are only possible from rules"
    _queue(txn.pure_effects, func, args)

def _queue(queue, func, args):
    txn = ctrl.txn
    assert not txn.crule.readonly, "Readonly rules can't queue calls %r => %r" % (txn.crule, (func,) + args)
    assert callable(func), "queued `func` should be callable, got %r" % func
    queue.append((func, args))
    txn.on_undo(queue.pop)




def effect_method(func):
    """
        Make the decorated function to be queued as effect if called
        while in txn. No change otherwise.
    """
    @functools.wraps(func)
    def wrapped(*args):
        if ctrl.txn:
            effect(func, *args)
        else:
            with ctrl.new_txn():
                func(*args)
    return wrapped

def per_txn(func):
    """
        Ensure that decorated function only gets called once per txn
        (w/ the same arguments). Cached results are returned while in the
        same transaction, when txn finishes they are discaded.
        Do *NOT* read txn-al state inside wrapped function.
        For ex. see mext.reaction.time:Time.time
    """
    # we can't support **kw because dicts are not hashable and cannot be used in a key
    @functools.wraps(func)
    def wrapped(*args):
        if not ctrl.txn:
            return func(*args)
        key = func, args # make sure args items are hashable
        try:
            return ctrl.txn.transient[key]
        except KeyError:
            r = ctrl.txn.transient[key] = func(*args)
            return r
    return wrapped
