from __future__ import with_statement, absolute_import
import sys
from contextlib import contextmanager
from heapq import heappush, heappop

# deps of old-style txns
from operator import itemgetter
from mext.reaction.stm.stm_old import *


class InputConflict(Exception):
    """
        Attempt to set a cell to two different values during the same pulse
    """

class RootRule(object):
    """
        A type of special object for top-level txn changes:

            with ctrl.new_txn() as txn:
                assert isinstance(txn.crule, RootRule)

    """
    #__slots__ = ()
    can_rerun = False
    is_listener = False
    readonly = False
    layer = 0



####
####    Generic transaction base classes
####

@classmethod
def not_implemented(cls, *args):
    raise NotImplementedError(cls)

@staticmethod
def do_nothing():
    pass

class TransactionBase(object):
    __slots__ = ()
    def __init__(self, ctrl):
        self.ctrl = ctrl
    def __enter__(self): return self
    def __exit__(self, exc_type, exc, tb):
        if exc_type:
            raise exc_type, exc, tb
    on_success = on_failure = do_nothing
    post_txn = do_nothing



class UndoLoggedTransaction(TransactionBase):
    def __init__(self, ctrl):
        super(UndoLoggedTransaction, self).__init__(ctrl)
        self.undoing = False

    on_undo = not_implemented

##     def __exit__(self, *exc_info):
##         # We have to drop undo log here because it contains
##         # a circular reference to ourselves which in turn refers
##         # listeners, so unless we do this, we get ourselves a gc issue
##         self.undo = None
##         self.schedule = None
##         super(UndoLoggedTransaction, self).__exit__(*exc_info)
##



class LoggedChangesMixin(UndoLoggedTransaction):
##     #@@x
##     def change_attr(self, ob, attr, val):
##         """
##             Set `ob.attr` to `val`, w/undo log to restore the previous value
##         """
##         assert not self.undoing
##         self.undo_log.append((setattr, (ob, attr, getattr(ob, attr))))
##         setattr(ob, attr, val)
##

    def change_key(self, dic, key, val):
        try:
            oldval = dic[key]
        except KeyError:
            self.undo_log.append((dic.__delitem__, (key,)))
        else:
            self.undo_log.append((dic.__setitem__, (key, oldval)))
        dic[key] = val

    def set_key(self, dic, key, val):
        assert key not in dic, `dic[key]`
        self.undo_log.append((dic.__delitem__, (key,)))
        dic[key] = val

    def del_key(self, dic, key):
        self.undo_log.append((dic.__setitem__, (key, dic[key])))
        del dic[key]




class CRuleTransaction(TransactionBase):
    def __init__(self, ctrl):
        super(CRuleTransaction, self).__init__(ctrl)
        self.crule = RootRule() # current rule
        self.rule_reads = set()
        self.rule_writes = set()
        self.has_run = {}   # listeners that have run; {listener: snapshot() before it has run}

    def __repr__(self):
        return "<%s crule=%r>" % (self.__class__.__name__, self.crule)

    def used(self, subject):
        self.rule_reads.add(subject)

    def changed(self, cell):
        #@@ move txn.changed to txn.crule.changed(cell) ?
        #@@ even transient cells should not call .changed(self) during reset
        if self.resetting:
            raise InputConflict(
                "Cells can't change during reset (crule=%r, subj=%r)"
                % (self.crule, cell)
            )
        elif cell.can_reset and cell not in self.reset_queue:
            self.undo_log.append((self.reset_queue.remove, (cell,)))
            self.reset_queue.add(cell)
        # subjects can change themselves
        if cell is not self.crule:
            # non-readonly rules can change others
            if self.crule.readonly:
                raise RuntimeError(
                    "Can't change objects from read-only rules (crule=%r, subj=%r)"
                    % (self.crule, cell)
                )
            #elif self.resetting and not isinstance(self.crule, RootRule): #@@
            #    #@@ track/compute rules should not change either
            #    raise RuntimeError("%r tried to write %r during reset" % (self.crule, cell))
        self.rule_writes.add(cell)



    @contextmanager
    def new_root(self):
        root = RootRule()
        with self._replace_crule(root):
            yield root
            #assert not self.rule_reads and parent.can_rerun
            self._process_writes()

    def run_rule(self, rule):
        """
            Run the specified listener
        """
        # if outer rule can undo and rerun, then we don't need to hurry to rerun
        # this rule, let the outer one finish, then roll them both back and rerun
        # in correct order
        assert not self.resetting
        assert not self.crule.can_rerun, "Only un-initialized rules can be nested"
        assert rule not in self.has_run, "Re-run of rule without retry: %s" % rule
        assert rule.readonly or not self.crule.readonly

        self.has_run[rule] = self.savepoint()
        self.on_undo(self.has_run.pop, rule)
        with self._replace_crule(rule):
            rule.run()
            self._process_writes()
            self._process_reads()

    @contextmanager
    def _replace_crule(self, rule):
        #assert rule not in self.has_run, "Re-run of rule without retry"
        #assert rule.readonly or not self.crule.readonly
        parent, reads, writes = self.crule, self.rule_reads, self.rule_writes
        self.crule, self.rule_reads, self.rule_writes = rule, set(), set()
        try:
            yield
        finally:
            self.crule, self.rule_reads, self.rule_writes = parent, reads, writes



    #@@ belongs to old-style
    def _process_reads(self):
        # Remove subjects from self.rule_reads and link them to `listener`
        # Old subjects of the listener are deleted
        rule = self.crule
        subjects = self.rule_reads
        subjects.discard(rule)

        l = rule.layer
        for subject in subjects:
            if subject.layer >= l:
                l = rule.layer = subject.layer + 1
                if l not in self.queues:
                    self.queues[l] = set() # for _reschedule

        link = rule.next_subject
        while link is not None:
            nxt = link.next_subject   # avoid unlinks breaking iteration
            if link.subject in subjects:
                subjects.remove(link.subject)
            else:
                self.undo_log.append((Link, (link.subject, rule)))
                link.unlink()
            link = nxt

        for subject in subjects:
            link = Link(subject, rule)
            self.undo_log.append((link.unlink, ()))

    #_process_reads = not_implemented
    _process_writes = not_implemented





class EffectsMixin(CRuleTransaction):
    def __init__(self, ctrl):
        super(EffectsMixin, self).__init__(ctrl)
        self.effects = []
        self.pure_effects = []
        self.q_notxn = []

    def effect(self, func, *args):
        self.effects.append((func, args))
        self.on_undo(self.effects.pop)

    def run_queue(self, queue):
        for func, args in queue:
            func(*args)

    def post_txn(self):
        self.run_queue(self.pure_effects)
        self.ctrl.txn = None
        self.run_queue(self.q_notxn)
        # at this point ctrl.txn is expected to be a DisabledTransaction()
        if self.effects:
            with self.ctrl.new_txn():
                self.run_queue(self.effects)
        self.effects = None
        super(EffectsMixin, self).post_txn()





class ScheduledTransaction(CRuleTransaction, UndoLoggedTransaction):
    def __init__(self, ctrl):
        super(ScheduledTransaction, self).__init__(ctrl)
        self.layers = []    # heap of layer numbers
        self.queues = {}    # [layer]    -> dict of listeners to be run
        self.scheduled_by = {} # listener -> set(cells that scheduled it)
        # note that if MaintainRule changes a Value that was read by some `rule_x`
        # it will be the MaintainRule that ends up in scheduled_by[rule_x]
        self.to_retry = set()
        self.reset_queue = set()
        self.resetting = False

    def __exit__(self, *exc_info):
        if exc_info[0] is None:
            try:
                self._process_writes()
                self.rule_writes = set()
                self._run_rules()
                if self.reset_queue:
                    self._reset()
            except:
                exc_info = sys.exc_info()
            self.resetting = False
        super(ScheduledTransaction, self).__exit__(*exc_info)


    def _run_rules(self):
        layers = self.layers
        queues = self.queues
        while layers:
            if self.to_retry:
                self._retry()
            q = queues[layers[0]]
            if q:
                listener = q.pop()
                self.on_undo(self._reschedule, listener)
                self.run_rule(listener)
            else:
                #del queues[layers[0]]
                heappop(layers)

    def _reset(self):
        #@@ change_attr
        #self.on_undo(setattr, self, 'resetting', False)
        self.resetting = True
        self.crule = RootRule()
        to_reset = set()
        for cell in self.reset_queue:
            to_reset.add(cell)
            to_reset.update(cell.iter_listeners())
        self.reset_queue = None
        #self.has_run = None
        for cell in to_reset:
            cell.run_reset()

    #@@ make source_layer required
    def schedule(self, listener, source_layer=-1):
        """
            Schedule `listener` to run during an atomic operation

            If an operation is already in progress, it's immediately scheduled, and
            its scheduling is logged in the undo queue (unless it was already
            scheduled).

            If `source_layer` is specified, ensure that the listener belongs to
            a higher layer than the source, moving the listener from an existing
            queue layer if necessary.  (This layer elevation is intentionally
            NOT undo-logged, however.)
        """
        assert not self.resetting and not self.undoing
        new = old = listener.layer
        if source_layer >= old:
            listener.layer = new = source_layer + 1

        if listener in self.has_run:
            self.to_retry.add(listener)
            if new is not old:
                q = self.queues.setdefault(new, set())
                if not q:
                    heappush(self.layers, new)
            return

        if listener in self.scheduled_by:
            scheduled_by = self.scheduled_by[listener]
            scheduled_by.add(self.crule)
            self.on_undo(scheduled_by.remove, self.crule)
            if new is not old:
                self.queues[old].remove(listener)
        else:
            # first schedule of this listener in this txn, undo-log it
            self.on_undo(self.cancel, listener)
            self.set_key(self.scheduled_by, listener, set([self.crule]))

        try:
            q = self.queues[new]
        except KeyError:
            self.queues[new] = set([listener])
            heappush(self.layers, new)
        else:
            if not q:
                # @@ add test for this path
                heappush(self.layers, new)
            q.add(listener)


    # asserts disabled for performance
    def _reschedule(self, listener):
        #assert self.undoing and not self.resetting
        #assert listener not in self.has_run
        layer = listener.layer
        q = self.queues[layer]
        if not q:
            heappush(self.layers, layer)
        q.add(listener)

    def _schedule_retry(self, listener):
        #assert not self.undoing and not self.resetting
        #assert listener not in self.has_run
        #assert listener not in self.queues[listener.layer]
        self.queues[listener.layer].add(listener)
        self.on_undo(self.cancel, listener)


    def cancel(self, listener):
        """
            Remove the `listener` from queues
        """
        # we use discard and not remove because .cancel(rule) is in undo log
        # and that log could be processed after that rule has run already
        # @@ add test for this case
        #self.queues[listener.layer].remove(listener)
        self.queues[listener.layer].discard(listener)



    def _cleanup_queues(self):
        # for tests only
        assert not self.crule.can_rerun
        self.queues = dict((l,q) for (l,q) in self.queues.iteritems() if q)
        self.layers = self.queues.keys()
        self.layers.sort()


    #@@ belongs to old-style
    def _retry(self):
        try:
            to_retry = self.to_retry
            for item in to_retry:
                path = check_circularity(item, self.scheduled_by, item, set())
                if path:
                    #import pprint; pprint.pprint(path)
                    raise CircularityError(path)
            # undo back through listeners, watching to detect cycles
            self.rollback_to(min(map(self.has_run.__getitem__, to_retry)))
            map(self._schedule_retry, to_retry)
        finally:
            self.to_retry = set()

    #@@ belongs to old-style
    def _process_writes(self):
        # Remove changed items from self.rule_writes and notify their listeners
        # and set up an undo action to track this listener's participation
        # in any cyclic dependency that might occur later.
        rule = self.crule
        layer = rule.layer
        for subject in self.rule_writes:
            for dependent in subject.iter_listeners():
                #@@ reading then writing a cell is a bad idea, can we prohibit it?
                #if dependent is rule:
                #    raise CircularityError(rule)
                if dependent is not rule: #@@ see if this check can be removed
                    if dependent.dirty():
                        #print rule, '->', dependent
                        self.schedule(dependent, layer)


class CircularityError(Exception):
    """Rules arranged in an infinite loop"""


def check_circularity(cell, scheduled_by, start, seen):
    """Detect CircularityError, if applicable"""
    try:
        cells = scheduled_by[cell]
    except KeyError:
        return
    for cell in cells - seen: # not in-place!
        if cell is start:
            return [cell]
        seen.add(cell)
        path = check_circularity(cell, scheduled_by, start, seen)
        if path:
            #path.insert(0, cell)
            path.append(cell) # running order
            return path


####
####    Old-style transaction implementation baseclasses
####



class ManageMixin(TransactionBase):
    def __init__(self, ctrl):
        super(ManageMixin, self).__init__(ctrl)
        self.managers = {} # mgr -> seq_no #  (context managers to __exit__ with)
        self.in_cleanup = False

    def __exit__(self, *exc_info):
        managers = self.managers.items()
        managers.sort(key=itemgetter(1), reverse=True)
        self.managers = None
        self.in_cleanup = True
        try:
            for manager, pos in managers:
                try:
                    manager.__exit__(*exc_info)
                except:
                    exc_info = sys.exc_info()
        finally:
            self.in_cleanup = False
        super(ManageMixin, self).__exit__(*exc_info)

    def manage(self, mgr):
        if mgr not in self.managers:
            mgr.__enter__()
            self.managers[mgr] = len(self.managers)



class LinearUndoLogTransaction(UndoLoggedTransaction):
    # @@ put on_reset in a different baseclass
    def __init__(self, ctrl):
        self.undo_log = []
        super(LinearUndoLogTransaction, self).__init__(ctrl)


    def on_undo(self, func, *args):
        """
            Call `func(*args)` if atomic operation is undone
        """
        assert not self.undoing
        self.undo_log.append((func, args))

    def savepoint(self):
        """
            Get a savepoint suitable for calling ``rollback_to()``
        """
        return len(self.undo_log)

    def rollback_to(self, sp=0):
        """
            Rollback to the specified savepoint
        """
        undo = self.undo_log
        self.undoing = True
        rb = self.rollback_to
        try:
            while len(undo) > sp:
                f, a = undo.pop()
                if f==rb and a:
                    sp = min(sp, a[0])
                else:
                    f(*a)
        finally:
            self.undoing = False




    def __exit__(self, *exc_info):
        #@@ TODO:
        # Next block should be deleted, and on_undo renamed to on_retry.
        # All the changes rules make should be stored in txn and
        # be immediately discarded by discarding the txn object,
        # no undo actions could be necessary -- only this way we can
        # make sure concurrent transactions don't communicate with one
        # another. And even if we have global lock, we still want to have
        # nested / overriding transactions for initialization, so this
        # property is important.
        # So, what is now an undo log should be a log of actions for
        # rolling back rules, not discard their changes in case of errors.

        if exc_info[0] is not None:
            try:
                self.rollback_to(0)
            except:
                exc_info = sys.exc_info()

        # We have to drop undo log here because it contains
        # a circular reference to ourselves which in turn refers
        # listeners, so unless we do this, we get ourselves a gc issue
        self.undo_log = None
        self.schedule = None

        super(LinearUndoLogTransaction, self).__exit__(*exc_info)








class OldTransaction(ScheduledTransaction, LoggedChangesMixin, LinearUndoLogTransaction, ManageMixin):
    pass




####
####    New-style transaction class
####



class SnapshotTransaction(TransactionBase):
    #__slots__ = ('ctrl', 'snapshot', 'writes', 'reads', 'transient')
    def __init__(self, ctrl):
        super(SnapshotTransaction, self).__init__(ctrl)
        self.snapshot = ctrl.make_snapshot()
        self.reads = []
        self.note_read = self.reads.append
        self.writes = {}
        self.transient = {}
        self.set_by = {}

    def on_success(self):
        # if the txn has no writes, it can't conflict w/ others
        if self.writes:
            self.commit_writes()
        self.drop_snapshot()
        super(SnapshotTransaction, self).on_success()

    def on_failure(self):
        self.drop_snapshot()
        super(SnapshotTransaction, self).on_failure()

    def commit_writes(self):
        # repeat until upd_state reports success
        while not self.ctrl.upd_state(self):
            # we are here because we used an out-of date snapshot
            self.replace_snapshot(self.ctrl.make_snapshot())
            # some rules might have been rolled back, run them again
            self.run_rules()

    def replace_snapshot(self, snapshot):
        """
            The state has changed since this txn started,
            change our starting point to the passed snapshot
            and roll back as much as necessary
        """
        # check if the read values are still valid
        base = self.snapshot
        for key in frozenset(self.reads):
            # Equality check is not enough here because
            # the cell value could be a component or another cell
            # and even if they are equal the dependencies might get
            # messed up if we aren't strict enough here.
            # On a second thought, components and cells MUST only
            # be comparable by identity, so this comparision might as well be
            # relaxed.
            if snapshot[key] is not base[key]:
                self._invalidate_read(key)
        self.drop_snapshot()
        self.snapshot = snapshot

    def drop_snapshot(self):
        self.ctrl.drop_snapshot(self.snapshot)
        self.snapshot = None # just in case

    def run_rules(self):
        """
            Run all the scheduled rules
        """
        # do nothing, because _invalidate_read is not implemented
        # and there are no rules to rerun at this point
        raise NotImplementedError

    def _invalidate_read(self, cell_key):
        raise NotImplementedError("Rollback not implemented yet (key=%r)" % cell_key)
        # pop from reads (or replace w/ new value)
        # roll back rules that read it
        # (and reschedule them in some cases)





class Transaction(SnapshotTransaction, OldTransaction, EffectsMixin):
    pass
