from __future__ import with_statement, absolute_import
import functools, os
import unittest

from mext.reaction import *
from mext.reaction.time import *
from mext.reaction.eventloop import *
from mext.reaction.celltypes.base import CellKey
from mext.context import *
from nose.tools import assert_raises, assert_equal, eq_, ok_, assert_true, assert_false, raises

eq = eq_
ok = ok_

__all__ = [
    'force_retry', 'assert_raises', 'assert_equal', 'raises',
    'eq_', 'ok_', 'eq', 'ok', 'assert_true', 'assert_false',
    'with_eventloop', 'with_eventqueue',
    'with_paused_time', #@@x
    'DummyError', 'DummyEventLoop', 'PausedTime', 'TestCell',
    'EventLoopTestCase', 'with_new_txn'
]

def make_sleep():
    if os.name == 'nt':
        # On Windows time.sleep restores default signal handlers while sleeping.
        # http://svn.python.org/view/python/branches/release26-maint/Modules/timemodule.c?view=markup
        # This makes signal_utils.Signals.BREAK miss the signal.
        # To work around this, we call kernel32.Sleep() directly
        import ctypes
        Sleep = ctypes.windll.kernel32.Sleep
        def sleep(secs):
            Sleep(ctypes.c_short(int(1000*secs)))
    else:
        from time import sleep
    return sleep



class DummyError(Exception):
    pass

class TestCell(object):
    def __init__(self):
        self.key = CellKey(self)

def force_retry(rule):
    with ctrl.new_txn() as txn:
        cell = Value()
        @MaintainRule
        def rule_1():
            ctrl.txn.used(cell)
            ctrl.txn.changed(cell)
        @MaintainRule
        def rule_2():
            if cell.value:
                rule()

        rule_1.layer = rule_2.layer + 1 # set order that would need to be changed
        cell.value = 1
        #txn.schedule(rule_2)
        #txn.schedule(rule_1)



def with_eventqueue(func):
    @functools.wraps(func)
    def wrapped(*args):
        with State():
            EventQueue.activate()
            return func(*args)
    return wrapped

def with_eventloop(func):
    @functools.wraps(func)
    def wrapped(*args):
        #with DummyEventLoop.new():
        # we actually want an empty state
        with State():
            PausedTime.activate()
            EventQueue.activate()
            DummyEventLoop.activate()
            return func(*args)
    return wrapped

def with_paused_time(func):
    @functools.wraps(func)
    def wrapped(*args):
        with PausedTime.new():
            return func(*args)
    return wrapped


class DummyEventLoop(EventLoop):
    __service__ = EventLoop

    # .dont_stop controls if eventloop should keep going if there are
    #  * no scheduled timers
    #  * no scheduled calls (EventQueue)
    # The default is to stop ASAP.
    dont_stop = attr(False)

    def _loop(self):
        q = EventQueue.get()
        sleep = make_sleep()
        while True:
            if self.stop_requested:
                break
            if q._call_queue:
                q.flush()
            elif Time.next_event_time:
                rel_time = Time.next_event_time - Time.time
                if rel_time <= 0:
                    # time to tick
                    Time.tick()
                elif q._call_queue:
                    continue
                else:
                    # too early to tick, no call_queue, lets sleep
                    sleep(min(rel_time, 0.05))
            elif self.dont_stop:
                sleep(0.05) # sleep a little bit
            else:
                break


    def poll(self):
        self.flush(1)

    def _arrange_wakeup(self):
        pass




class PausedTime(Time):
    __service__ = Time
    time = 0

    def tick(self):
        raise TypeError("PausedTime can't tick")

    def advance(self, interval=None):
        """
            Advance the current time by the given interval
        """
        assert ctrl.txn is None
        if interval is None:
            if self._next_event is not Max:
                self.time = self._next_event
        else:
            self.time += interval
        self.process_schedule()

class EventLoopTestCase(unittest.TestCase):
    def setUp(self):
        self.state = State()
        self.state.__enter__()
        super(EventLoopTestCase, self).setUp()
        self.configure_context()

    def tearDown(self):
        super(EventLoopTestCase, self).tearDown()
        self.state.__exit__(None, None, None)

    def configure_context(self):
        pass


def with_new_txn(f):
    @functools.wraps(f)
    def g(self):
        try:
            with getattr(self, 'ctrl', ctrl).new_txn() as self.txn:
                return f(self)
        finally:
            del self.txn
    return g


