from __future__ import with_statement, absolute_import

from heapq import heappush, heappop
from time import time
from datetime import datetime
from weakref import WeakValueDictionary
from peak.util.extremes import Max

from mext.reaction import *
from mext.reaction.stm import AbstractSubject
from mext.context import ServiceStub



__all__ = ['Time', 'PointInTime', 'Clock']

txn_time = per_txn(time)


class Time(ServiceStub):
    _schedule = None
    # The time of the next event to occur, or ``None`` if none scheduled
    next_event_time = attr()

    def __init__(self):
        self._schedule = [Max]

    @property
    def time(self):
        return txn_time()

    def tick(self):
        self.process_schedule()

    def __getitem__(self, interval):
        return PointInTime(self.time + interval)

    def _update(self):
        time = self._schedule[0]
        self.next_event_time = time if (time is not Max) else None

    def add_event(self, timer):
        heappush(self._schedule, timer.timestamp)
        self._update()

    def process_schedule(self):
        with ctrl.new_txn():
            now = self.time
            #print 'process_schedule [%s]' % now, self._schedule,
            schedule = self._schedule
            while now >= schedule[0]:
                time = heappop(schedule)
                PointInTime.reached(time)
            while (schedule[0] is not Max
                and not PointInTime.has_listener(schedule[0])
            ):
                heappop(schedule)
            #print '->', self._schedule
            self._update()

Time.activate()

#@@ PointInTime instances are global, if point in time is reached
# in one thread it will trigger all dependencies (other threads may have
# different Time services, so it would not change to `True` in those threads,
# the cells will be triggered, and they will execute in the thread where the
# point in time WAS reached, even if they created this dependency in a different thread.

# In other words:
#  * make sure that all threads where PointInTime will be used have the same Time *instance* installed
#    as a service
#  * make sure that the Time is being serviced by some EventLoop
#  * be prepared that cells dependent on some PointInTime will get scheduled in the thread of that eventloop


class PointInTime(AbstractSubject):
    """
        Value representing a moment in time
    """
    __slots__ = 'timestamp', '__weakref__', 'next_listener'
    layer = 0

    class __metaclass__(type):
        _cache = WeakValueDictionary()
        def __call__(cls, when):
            r = cls._cache.get(when)
            #r = EventQueue.get()._times.get(when)
            if r is not None:
                return r
            return cls._cache.setdefault(when, type.__call__(cls, when))

        def reached(cls, time):
            pit = cls._cache.get(time)
            if pit is not None:
                ctrl.txn.changed(pit)

        def has_listener(cls, time):
            pit = cls._cache.get(time)
            return pit is not None and pit.next_listener is not None



    def __init__(self, when):
        self.timestamp = when
        super(PointInTime, self).__init__()

    def __repr__(self):
        return '%s(%r)' % (type(self).__name__, self.timestamp)

    def __getitem__(self, offset):
        return PointInTime(self.timestamp + offset)

    def __sub__(self, other):
        if not isinstance(other, PointInTime):
            raise TypeError("Can't subtract %r from timer" % (other,))
        return self.timestamp - other.timestamp

    def __lt__(self, other):
        if not isinstance(other, PointInTime):
            raise TypeError
        return self.timestamp < other.timestamp

    def __ge__(self, other):
        return not self < other

    def __nonzero__(self):
        if Time.time >= self.timestamp:
            return True
        if ctrl.txn and ctrl.txn.crule.is_listener:
            ctrl.txn.used(self)
            if self.next_listener is None:
                ctrl.txn.effect(Time.add_event, self)
        return False

    passed = property(__nonzero__)

    def as_datetime(self):
        return datetime.fromtimestamp(self.timestamp)

#def attr_timestamp():
#    return make(staticmethod(lambda: Time[0]), optional=False)

class Clock(Component):
    attrs(step=None, align=None)

    def __init__(self, step=1, align=Time):
        self.step = step
        self.align = align[0]

    @compute
    def tick_no(self):
        """
            Tick number. Integer counting from 0.
            (step * tick_no) <= (current_time - align) < (step * (tick_no+1))
        """
        tick_no = int((Time[0] - self.align) / self.step)
        next_tick = self.align[(tick_no+1) * self.step]
        if next_tick.passed:
            # Sometimes due to float inherent rounding
            # next_tick is triggered by the eventloop but
            # tick_no just ends up being the same as during prev recalc
            # this means that next_tick is the same as the one that was triggered
            # i.e. it is passed.
            # We catch this case and make sure tick_no advances.
            next_tick = next_tick[self.step]
            if next_tick.passed:
                # now this is incredible
                raise AssertionError("next_tick has passed. now=%r; next_tick=%r" % (Time[0], next_tick))
            return tick_no + 1
        return tick_no

    @compute
    def time(self):
        """
            Current PointInTime, aligned to align[<int>*step].
            Either current or in the past, but by no more than `step` seconds.
        """
        return self.align[self.tick_no * self.step]

    @compute(force_reset=False)
    def tick(self):
        """
            Transient value signalling tick change.
        """
        self.tick_no
        return True
