# -*- coding: utf-8 -*-

#  Copyright (C) 2013 Hai Bison
#
#  See the file LICENSE at the root directory of this project for copying
#  permission.

"""
Created on Wed Nov 13 02:27:08 2013

@author: Hai Bison
"""

import datetime, re, sys

'''The Redis instance.'''
R = None

'''The main separator for components of a DB key.'''
SEP = ':'

def redis2python(red, cast=None):
    '''Converts ``red`` to Python strings.

        Parameters:

        :cast:
            If provided, the data returned will be casted to the given type. If
            the data is a group (list, set, dict...) then its elements will be
            casted. If the data is a single item then itself will be casted.

        Returns:
            The converted value.
    '''
    _type = type(red)
    if _type == dict:
        result = {}
        for k in red:
            v = red[k].decode()
            result[k.decode()] = cast(v) if cast else v
        return result
    elif _type == set:
        return set(cast(_.decode()) if cast else _.decode() for _ in red)
    elif _type == list:
        return list(cast(_.decode()) if cast else _.decode() for _ in red)

    return cast(red) if cast else red
    #.redis2python()

'''An empty class. It can be used to hold arbitrary variables.'''
class Empty: pass

class Model:
    '''The base model.

    To make it work, you can provide:

    :Override ``serie_name()`` if needed:
        You return the serie name. This value will be used to prefix keys in DB.
        Default is the class' qualified name (``__qualname__``).

    :_AUTO_ID:
        Set to ``True`` to let the model auto assign to itself new UID when
        creating new instance. Default is ``True``.

    Declares all of your variables' data types within class' scope (also means
    class variables). This helps the framework map Python's data types to Redis'
    data types. Note that please don't prefix variables with ``_`` (or more).
    They are reserved. For structure types such as ``dict``, ``list``, ``set``...,
    you have to create new instance of them and put into them at least one value.
    For example:

        class User(rdb.Model):
            email = str
            first_name = str
            last_name = str
            """This is a list of product IDs, each ID is an ``int``."""
            products = [0]

    In constructors, you should use the same keys that you defines within class'
    scope. So the framework can help you build the bridge between Python's and
    Redis' data types. That also means you can use whatever keys you want,
    the framework just don't touch "unknown" keys and use default data types
    for them. It is byte array.

    Currently, the framework supports these data types:

    - If a value is a normal object, it will be put into DB as a byte array.
    - If a value is a dictionary, it will be put into DB as a hash.
    - If a value is a list, it will be put into DB as a list.
    - If a value is a set, it will be put into DB as a set.
    - If a value is a ``bytes``, it will be put into DB as byte array.
    - Currently doesn't support Redis' sorted set yet.

    If the field's type is a group of values (``dict``, ``set``, ``list``...),
    please define one element. This helps determining the element data type.

    If you want to remove a field (which actually remove its key), don't use
    Python ``del``, instead, set its value to ``None``.
    '''

    '''This is reserved key.'''
    _ID = '_id'
    '''This is reserved key.'''
    _COUNT = '_count'
    _AUTO_ID = True

    # SOME CONSTANTS
    _S_YEAR = '_year'
    _S_MONTH = '_month'
    _S_DAY = '_day'
    _S_HOUR = '_hour'
    _S_MINUTE = '_minute'
    _S_SECOND = '_second'
    _S_MICROSECOND = '_microsecond'
    _S_UTC_OFFSET = '_utc_offset'

    @classmethod
    def serie_name(cls):
        '''Returns the serie name of this object in DB.
        '''
        return cls.__qualname__

    @classmethod
    def uid(cls, inc=True):
        '''Returns the current unique ID of the model. If ``inc`` is ``True``,
        increases the ID by ``1`` and returns new value.
        '''

        _ = SEP.join([cls.serie_name(), cls._ID])
        return R.incr(_) if inc else int(R.get(_).decode())
        #.uid()

    @classmethod
    def _redis2python(cls, name, key, obj):
        '''Determines Python type of class' variable ``name``, then reads data
        from Redis at ``key``, converts the data read to corresponding Python
        type and assigns the value to variable ``name`` of object ``obj``.
        '''

        attr = getattr(cls, name)
        if callable(attr) and not isinstance(attr, type): return

        _type = attr if isinstance(attr, type) else type(attr)
        if _type == dict:
            red = R.hgetall(key)
            py = {}
            element_type = type(next(iter(attr.values())))
            for _ in red: py[_.decode()] = element_type(red[_].decode())
            setattr(obj, name, py)
        elif _type == set:
            element_type = type(next(iter(attr)))
            setattr(obj, name,
                    set(element_type(_.decode()) for _ in R.smembers(key)))
        elif _type == list:
            element_type = type(attr[0])
            _ = R.lrange(key, 0, sys.maxsize)
            setattr(obj, name,
                    list(element_type(_.decode()) for _ in _))
        elif _type == datetime.datetime:
            _ = redis2python(R.hgetall(key), int)
            tzinfo = _[cls._S_UTC_OFFSET] if cls._S_UTC_OFFSET in _ else 0
            if tzinfo:
                tzinfo = datetime.timezone(datetime.timedelta(seconds=tzinfo))
            setattr(obj, name, datetime.datetime(year=_[cls._S_YEAR],
                                                 month=_[cls._S_MONTH],
                                                 day=_[cls._S_DAY],
                                                 hour=_[cls._S_HOUR],
                                                 minute=_[cls._S_MINUTE],
                                                 second=_[cls._S_SECOND],
                                                 microsecond=_[cls._S_MICROSECOND],
                                                 tzinfo=tzinfo if tzinfo else None))
        elif _type == datetime.date:
            _ = redis2python(R.hgetall(key), int)
            setattr(obj, name, datetime.date(year=_[cls._S_YEAR],
                                             month=_[cls._S_MONTH],
                                             day=_[cls._S_DAY]))
        elif _type == datetime.time:
            _ = redis2python(R.hgetall(key), int)
            tzinfo = _[cls._S_UTC_OFFSET] if cls._S_UTC_OFFSET in _ else 0
            if tzinfo:
                tzinfo = datetime.timezone(datetime.timedelta(seconds=tzinfo))
            setattr(obj, name, datetime.time(hour=_[cls._S_HOUR],
                                             minute=_[cls._S_MINUTE],
                                             second=_[cls._S_SECOND],
                                             microsecond=_[cls._S_MICROSECOND],
                                             tzinfo=tzinfo if tzinfo else None))
        else:
            _ = R.get(key)
            if _type in [str, int, float, complex, bool]:
                _ = _type(_.decode())
            setattr(obj, name, _)
        #._redis2python()

    @classmethod
    def query(cls, func, *names, id_gt=0, id_lt=0, limit=0):
        '''Queries.

        Parameters:

        :func:
            The callback function which accepts one argument -- which is an
            arbitrary object crarrying all fields provided in ``names``. You
            return ``True`` to accept the result; return ``None`` or ``False``
            otherwise.

        :*names:
            List of variable names of current instance of this model that you
            want to get their values. The values found will be attached to an
            arbitrary object, that object will be passed back to ``func`` for
            you to process.

        :id_gt:
            If set, and the current instance of this model enables auto ID, then
            *try* to only looks for items having ID greater than the given
            value.

        :id_lt:
            If set, and the current instance of this model enables auto ID, then
            *try* to only looks for items having ID lower than the given value.

        :limit:
            To limits the results. ``0`` is unlimited.

        Returns:
            A list of "accepted objects" which are specified by ``func``. If
            current instance of the model support auto ID, the real items will
            be returned. Otherwise the empty objects which carry variables you
            required will be returned.

        Note that if you use both ``id_gt`` and ``id_lt``, they should not be
        conflicted. For example this is wrong: ``id_gt=99` and ``id_lt=9``. In
        such case, an empty list is returned.
        '''

        # BUILD THE KEY PREFIX

        prefix = cls.serie_name() + SEP

        if cls._AUTO_ID:
            if id_gt and id_lt:
                if id_gt >= id_lt: return []
                prefix += '?' * len(str(id_gt)) + '*'
            elif id_gt:
                prefix += '?' * len(str(id_gt)) + '*'
            elif id_lt:
                prefix += '?' * len(str(id_gt))
            else:
                prefix += '*'
            prefix += SEP

        # QUERY

        result = []
        for name in names:
            for key in R.keys(prefix + name):
                key = key.decode()

                obj = Empty()
                cls._redis2python(name, key, obj)

                if func(obj):
                    if cls._AUTO_ID:
                        _ = key[:-len(name)-len(SEP)]
                        _ = int(re.search(r'[0-9]+$', _).group())
                        result.append(cls(_id=_))
                    else:
                        result.append(obj)
                    if limit and len(result) >= limit:
                        return result

        return result
        #.query()

    @classmethod
    def count(cls):
        '''Counts total item of current model.
        '''
        _ = R.get(SEP.join([cls.serie_name(), cls._COUNT]))
        return int(_.decode()) if _ else 0
        #.count()

    def __init__(self, __strict=True, **args):
        '''Creates new instance. ``args`` is a list of key-value pairs.

        Parameters:

        :__strict:
            If you provide ``_id``, that means you are trying to get the item
            with an ID. If *all* of its keys are not available, ``KeyError``
            will be raised, *unless* ``__strict`` is ``False``.

        If you create new instance without an ID, new ID will be generated only
        when you call ``put()``.
        '''

        if len(args) == 1 and Model._ID in args:
            self._id = args[Model._ID]

            # Try to get the item in DB
            name_count = missing_names = 0
            for name in vars(self.__class__):
                if name.startswith('_'): continue

                attr = getattr(self.__class__, name)
                if callable(attr) and not isinstance(attr, type): return

                key = SEP.join([self.__class__.serie_name(), str(self._id), name])
                name_count += 1
                if R.exists(key):
                    self.__class__._redis2python(name, key, self)
                else:
                    missing_names += 1
                #.for

            if __strict and name_count > 0 and missing_names == name_count:
                raise KeyError

            return
            #.if

        self._id = -1
        for _ in args: setattr(self, _, args[_])
        #.__init__()

    def put(self):
        '''Put this item into DB.
        '''

        key = self.__class__.serie_name() + SEP
        if self.__class__._AUTO_ID:
            if self._id < 0: self._id = self.uid()
            key += '{}{}'.format(self._id, SEP)

        key_count = 0
        change_count = 0

        allvars = vars(self)
        for name in allvars:
            if name.startswith('_'): continue

            key_count += 1
            value = allvars[name]
            name = key + name

            _type = type(value)
            if _type == dict:
                R.hmset(name, value)
                change_count += 1
            elif _type == set:
                R.sadd(name, *value)
                change_count += 1
            elif _type == list:
                R.delete(name)
                R.rpush(name, *value)
                change_count += 1
            elif _type == datetime.datetime:
                R.hmset(name, {
                        __class__._S_YEAR: value.year,
                        __class__._S_MONTH: value.month,
                        __class__._S_DAY: value.day,
                        __class__._S_HOUR: value.hour,
                        __class__._S_MINUTE: value.minute,
                        __class__._S_SECOND: value.second,
                        __class__._S_MICROSECOND: value.microsecond })
                if value.tzinfo:
                    R.hset(name, __class__._S_UTC_OFFSET,
                           int(value.tzinfo.utcoffset(None).total_seconds()))
                change_count += 1
            elif _type == datetime.date:
                R.hmset(name, {
                        __class__._S_YEAR: value.year,
                        __class__._S_MONTH: value.month,
                        __class__._S_DAY: value.day })
                change_count += 1
            elif _type == datetime.time:
                R.hmset(name, {
                        __class__._S_HOUR: value.hour,
                        __class__._S_MINUTE: value.minute,
                        __class__._S_SECOND: value.second,
                        __class__._S_MICROSECOND: value.microsecond })
                if value.tzinfo:
                    R.hset(name, __class__._S_UTC_OFFSET,
                           int(value.tzinfo.utcoffset(None).total_seconds()))
                change_count += 1
            elif value != None:
                R.set(name, value)
                change_count += 1
            else:
                R.delete(name)
                change_count -= 1
            #.for

        _ = SEP.join([self.__class__.serie_name(), __class__._COUNT])
        if key_count + change_count == 0:
            R.decr(_)
        else:
            R.incr(_)
        #.put()

    def delete(self):
        '''Deletes item by its ID. Returns ``True`` if OK, ``None`` otherwise.
        '''

        if self._id < 0: return

        keys = []
        for name in vars(self.__class__):
            if name.startswith('_'): continue

            attr = getattr(self.__class__, name)
            if callable(attr) and not isinstance(attr, type): continue

            keys.append(SEP.join([self.__class__.serie_name(), str(self._id), name]))
            #.for

        if keys: R.delete(*keys)
        R.decr(SEP.join([self.__class__.serie_name(), __class__._COUNT]))
        #.delete()
