# -*- coding: utf-8 -*-

#  Copyright (C) 2013 Hai Bison
#
#  See the file LICENSE at the root directory of this project for copying
#  permission.

"""
Created on Wed Nov 13 02:27:08 2013

@author: Hai Bison
"""

import datetime, re, sys

''' The Redis instance.'''
R = None

''' The main separator for components of a DB key.'''
SEP = ':'

def redis2python(red, cast=None):
    ''' Converts ``red`` to Python strings.

        Parameters:

        :cast:
            If provided, the data returned will be casted to the given type. If
            the data is a group (list, set, dict...) then its elements will be
            casted. If the data is a single item then itself will be casted.

        Returns:
            The converted value.
    '''
    _type = type(red)
    if _type == dict:
        result = {}
        for k in red:
            v = red[k].decode()
            result[k.decode()] = cast(v) if cast else v
        return result
    elif _type == set:
        return set(cast(_.decode()) if cast else _.decode() for _ in red)
    elif _type == list:
        return list(cast(_.decode()) if cast else _.decode() for _ in red)

    return cast(red) if cast else red
    #.redis2python()

''' An empty class. It can be used to hold arbitrary variables.'''
class Empty: pass

class Model:
    ''' The base model.

    To make it work, you can provide:

    :Override ``_serie_name()`` if needed:
        You return the serie name. This value will be used to prefix keys in DB.
        Default is the class' qualified name (``__qualname__``).

    :_AUTO_ID:
        Set to ``True`` to let the model auto assign to itself new UID when
        creating new instance. Default is ``True``.

    Declares all of your variables' data types within class' scope (also means
    class variables). This helps the framework map Python's data types to Redis'
    data types. Note that please don't prefix variables with ``_`` (or more).
    They are reserved. For structure types such as ``dict``, ``list``, ``set``...,
    you have to create new instance of them and put into them at least one value.
    For example:

        class User(rdb.Model):
            email = str
            first_name = str
            last_name = str
            """This is a list of product IDs, each ID is an ``int``."""
            products = [0]

    In constructors, you should use the same keys that you defines within class'
    scope. So the framework can help you build the bridge between Python's and
    Redis' data types. That also means you can use whatever keys you want,
    the framework just don't touch "unknown" keys and use default data types
    for them. It is byte array.

    Currently, the framework supports these data types:

    - If a value is a normal object, it will be put into DB as a byte array.
    - If a value is a dictionary, it will be put into DB as a hash.
    - If a value is a list, it will be put into DB as a list.
    - If a value is a set, it will be put into DB as a set.
    - If a value is a ``bytes``, it will be put into DB as byte array.
    - Currently doesn't support Redis' sorted set yet.

    If the field's type is a group of values (``dict``, ``set``, ``list``...),
    please define one element. This helps determining the element data type.

    If you want to remove a field (which actually remove its key), don't use
    Python ``del``, instead, set its value to ``None``.
    '''

    ### SETTINGS

    _AUTO_ID = True

    ''' Default set size that Redis supports. You can change this for your
        needs. However this value does *not* affect indexes (which use sorted
        set).'''
    _SET_SIZE = 2**32 - 1

    ''' Add all fields' *names* that you want them to be indexed to this list.
        If you do, you might consider overriding function ``_score()`` to
        support your values.
    '''
    _INDEXES = []

    ''' This is *not* maximum length of strings that you can store (Redis allows
        up to 512 MiB).

        This is the maximum length of strings that can be *calculated* their
        score. If a string length is larger than this value, then only the first
        x characters are used to calculate the strings' score. Of course ``x``
        is the value of this flag.

        Be careful if you want to change this. All old scores will be invalid
        with new value.
    '''
    _MAX_STR_LEN = 99

    ''' Convenient value for maximum string's score.'''
    _MAX_STR_SCORE = sys.float_info.max

    ''' Sets to ``True`` to compares strings with case. Default is ``False``.'''
    _STR_CASE_SENSITIVE = False

    ### CONSTANTS

    _S_ID = '_id'
    _S_IDS = '_ids'
    _S_INDEXES = '_indexes'

    _S_YEAR = '_year'
    _S_MONTH = '_month'
    _S_DAY = '_day'
    _S_HOUR = '_hour'
    _S_MINUTE = '_minute'
    _S_SECOND = '_second'
    _S_MICROSECOND = '_microsecond'
    _S_UTC_OFFSET = '_utc_offset'

    @classmethod
    def _serie_name(cls):
        ''' Returns the serie name of this object in DB.
        '''
        return cls.__qualname__

    @classmethod
    def _uid(cls, inc=True):
        ''' Returns the current unique ID of the model. If ``inc`` is ``True``,
            increases the ID by ``1`` and returns new value.
        '''

        _ = SEP.join([cls._serie_name(), cls._S_ID])
        return R.incr(_) if inc else int(R.get(_).decode())
        #._uid()

    @classmethod
    def _redis2python(cls, name, key, obj):
        ''' Determines Python type of class' variable ``name``, then reads data
            from Redis at ``key``, converts the data read to corresponding
            Python type and assigns the value to variable ``name`` of object
            ``obj``.
        '''

        attr = getattr(cls, name)
        if callable(attr) and not isinstance(attr, type): return

        _type = attr if isinstance(attr, type) else type(attr)
        if _type == dict:
            red = R.hgetall(key)
            py = {}
            element_type = type(next(iter(attr.values())))
            for _ in red: py[_.decode()] = element_type(red[_].decode())
            setattr(obj, name, py)
        elif _type == set:
            element_type = type(next(iter(attr)))
            setattr(obj, name,
                    set(element_type(_.decode()) for _ in R.smembers(key)))
        elif _type == list:
            element_type = type(attr[0])
            _ = R.lrange(key, 0, sys.maxsize)
            setattr(obj, name,
                    list(element_type(_.decode()) for _ in _))
        elif _type == datetime.datetime:
            _ = redis2python(R.hgetall(key), int)
            tzinfo = _[cls._S_UTC_OFFSET] if cls._S_UTC_OFFSET in _ else 0
            if tzinfo:
                tzinfo = datetime.timezone(datetime.timedelta(seconds=tzinfo))
            setattr(obj, name, datetime.datetime(year=_[cls._S_YEAR],
                                                 month=_[cls._S_MONTH],
                                                 day=_[cls._S_DAY],
                                                 hour=_[cls._S_HOUR],
                                                 minute=_[cls._S_MINUTE],
                                                 second=_[cls._S_SECOND],
                                                 microsecond=_[cls._S_MICROSECOND],
                                                 tzinfo=tzinfo if tzinfo else None))
        elif _type == datetime.date:
            _ = redis2python(R.hgetall(key), int)
            setattr(obj, name, datetime.date(year=_[cls._S_YEAR],
                                             month=_[cls._S_MONTH],
                                             day=_[cls._S_DAY]))
        elif _type == datetime.time:
            _ = redis2python(R.hgetall(key), int)
            tzinfo = _[cls._S_UTC_OFFSET] if cls._S_UTC_OFFSET in _ else 0
            if tzinfo:
                tzinfo = datetime.timezone(datetime.timedelta(seconds=tzinfo))
            setattr(obj, name, datetime.time(hour=_[cls._S_HOUR],
                                             minute=_[cls._S_MINUTE],
                                             second=_[cls._S_SECOND],
                                             microsecond=_[cls._S_MICROSECOND],
                                             tzinfo=tzinfo if tzinfo else None))
        else:
            _ = R.get(key)
            if _type in [str, int, float, complex, bool]:
                _ = _type(_.decode())
            setattr(obj, name, _)
        #._redis2python()

    @classmethod
    def query(cls, func=None, *names, id_gt=None, id_lt=None, limit=None,
              order=None):
        ''' Queries.

        Parameters:

        :func:
            The callback function which accepts one argument -- which is an
            arbitrary object specified by ``names``. You return ``True`` to
            accept the result; return ``None`` or ``False`` otherwise.

            If not provided, all items will be returned.

        :*names:
            List of variable names of current instance of this model that you
            want to get their values. The values found will be attached to an
            arbitrary object, that object will be passed back to ``func`` for
            you to process.

            If not provided, the *real* record will be passed to ``func``.

        :id_gt:
            If set then only looks for items having ID greater than the given
            value.

        :id_lt:
            If set then only looks for items having ID lower than the given
            value.

        :limit:
            To limits the results. ``None`` is unlimited.

        :order:
            This is a tuple of 3, 4 or 6. It can be represented by names like:
            ``order=(name,min,max,offset,count,ascending)``. Where:
            * ``name`` is the name of field you want to order.
            * ``min`` is the minimum score.
            * ``max`` is the maximum score.
            * ``ascending`` (optional), default is ``True``, which will order
              the result from low to high scores. Set to ``False`` or ``None``
              to use descending order.
            * ``offset`` (optional), is the offset. If set, ``count`` must be
              set too.
            * ``count`` (optional), is the count.

        Returns:
            A list of "accepted objects" which are specified by ``func`` and
            ``names``.

        Notes:

            If you use both ``id_gt`` and ``id_lt``, they should not be
            conflicted. For example this is wrong: ``id_gt=99` and ``id_lt=9``.
            In such case, an empty list is returned.

            If current instance does not support auto ID, an empty list is
            returned.
        '''

        if not cls._AUTO_ID or (id_gt != None and id_lt != None and id_gt >= id_lt):
            return []

        # BUILD THE KEY PREFIX

        prefix = cls._serie_name()

        # QUERY

        result = []
        id_set = 0
        while 1:
            if order:
                key_ids = SEP.join([cls._serie_name(), cls._S_INDEXES, order[0]])
                if not R.zcard(key_ids): return result

                offset = None if len(order) <= 4 else order[4]
                count = None if len(order) <= 5 else order[5]

                if len(order) <= 3 or order[3]:
                    keys = R.zrangebyscore(key_ids, order[1], order[2], offset,
                                           count)
                else:
                    keys = R.zrevrangebyscore(key_ids, order[2], order[1],
                                              offset, count)
            else:
                key_ids = SEP.join([cls._serie_name(), cls._S_IDS, str(id_set)])
                id_set += 1
                if not R.scard(key_ids): return result
                keys = R.smembers(key_ids)

            for key in keys:
                key = int(key.decode())
                if id_gt != None and key <= id_gt: continue
                if id_lt != None and key >= id_lt: continue

                if names:
                    obj = Empty()
                    _ = SEP.join([prefix, str(key), ''])
                    for name in names:
                        cls._redis2python(name, _ + name, obj)

                    if func:
                        if func(obj):
                            result.append(cls(_id=key))
                    else:
                        result.append(obj)
                    if limit and len(result) >= limit:
                        return result
                else:
                    obj = cls(_id=key)
                    if func:
                        if func(obj): result.append(obj)
                    else: result.append(obj)
                #.for

            # ``while`` loop is for ID set only. For indexes we just need this
            # loop once.
            if order: return result
            #.while

        return result
        #.query()

    @classmethod
    def _add_id(cls, _id):
        ''' Adds an ID to ID set. This does nothing if ``_AUTO_ID`` is
            ``False``, or ``_id < 0``.
        '''

        if _id < 0: return

        i = 0
        while 1:
            key = SEP.join([cls._serie_name(), cls._S_IDS, str(i)])
            if R.scard(key) < cls._SET_SIZE:
                R.sadd(key, _id)
                return
            i += 1
            #.while
        #._add_id()

    @classmethod
    def _remove_id(cls, _id):
        ''' Remove an ID from ID set. This does nothing if ``_AUTO_ID`` is
            ``False``, or ``_id < 0``.'''

        if _id < 0: return

        i = 0
        while 1:
            key = SEP.join([cls._serie_name(), cls._S_IDS, str(i)])
            if R.sismember(key, _id):
                R.srem(key, _id)
                return
            i += 1
            #.while
        #._remove_id()

    @classmethod
    def count(cls):
        ''' Counts total item of current model.
        '''

        result = i = 0
        while 1:
            size = R.scard(SEP.join([cls._serie_name(), cls._S_IDS, str(i)]))
            if size > 0:
                result += size
                i += 1
            else: break

        return result
        #.count()

    @classmethod
    def _score(cls, obj=None, **kwarg):
        ''' Override this function to "score" your values. By default, this
            function supports some numeric objects such as ``int``, ``float``,
            ``complex``, ``bool``; ``datetime``, ``date`` and ``time`` in module
            ``datetime``.

        You provide either a single value via ``obj`` or a map of variable names
        pointing to their value via ``kwarg``. If you provide ``obj`` then
        ``kwarg`` is ignored.

        For ``str``s, this function turns them to lower case, then calculates
        their value by summing up all characters.

        For groups (``list``, ``set``, ``dict``, ``bytes``...), this function
        calculates their value by their size.

        For module ``datetime``'s types: ``datetime`` and ``date``'s values are
        total seconds from ``datetime.min``. ``time``'s value is total seconds
        it holds.

        For unsupported types, their value is ``0``.

        Returns:
            If ``obj`` is not ``None``, returns its score. If ``kwarg`` is not
            ``None``, returns a map of original variable names to their score.
            If both parameters were not provided, returns ``0``.
        '''

        def __score(value):
            '''Calculates score of ``value``.'''

            _type = type(value)
            if _type in [dict, set, list, bytes]:
                return len(value)
            elif _type == datetime.datetime:
                return (value - datetime.datetime.min).total_seconds()
            elif _type == datetime.date:
                return (value - datetime.date.min).total_seconds()
            elif _type == datetime.time:
                _ = value.hour * 60 * 60 + value.minute * 60 + value.second + \
                    value.microsecond / 1000
                if value.tzinfo:
                    _ += value.tzinfo.utcoffset(None).total_seconds()
                return _
            elif _type in [int, float]:
                return value
            elif _type == bool:
                return int(value)
            elif _type == str:
                # We assume all strings take the same _MAX_STR_LEN length. So we
                # append all strings with "virtual" characters to fit
                # _MAX_STR_LEN. Every "virtual" character has a value of 0.
                if not cls._STR_CASE_SENSITIVE:
                    value = value.lower()
                result = 0
                for i in range(0, len(value)):
                    result += ord(value[i]) * 10**(cls._MAX_STR_LEN - i)
                return result

            return 0
            #.__score()

        if obj != None: return __score(obj)
        if not kwarg: return 0

        for name, value in kwarg.items():
            kwarg[name] = __score(value)
        return kwarg
        #._score()

    def __init__(self, __strict=True, **args):
        ''' Creates new instance. ``args`` is a list of key-value pairs.

        Parameters:

        :__strict:
            If you provide ``_id``, that means you are trying to get the item
            with an ID. If *all* of its keys are not available, ``KeyError``
            will be raised, *unless* ``__strict`` is ``False``.

        If you create new instance without an ID, new ID will be generated only
        when you call ``put()``.
        '''

        if len(args) == 1 and __class__._S_ID in args:
            self._id = args[__class__._S_ID]

            # Try to get the item in DB
            name_count = missing_names = 0
            # Always use dir() (instead of vars()) to get the variables from
            # parent classes.
            for name in dir(self.__class__):
                if name.startswith('_'): continue

                attr = getattr(self.__class__, name)
                if callable(attr) and not isinstance(attr, type): continue

                key = SEP.join([self.__class__._serie_name(), str(self._id), name])
                name_count += 1
                if R.exists(key):
                    self.__class__._redis2python(name, key, self)
                else:
                    missing_names += 1
                #.for

            if __strict and name_count > 0 and missing_names == name_count:
                # Be careful with format(), don't use '{:}' because ``_id`` is
                # not always a number.
                raise KeyError('{}'.format(self._id))

            return
            #.if

        self._id = -1
        for _ in args: setattr(self, _, args[_])
        #.__init__()

    def put(self):
        ''' Put this item into DB.
        '''

        pipe = R.pipeline()

        key = self.__class__._serie_name() + SEP
        if self.__class__._AUTO_ID:
            if self._id < 0: self._id = self.__class__._uid()
            # Be careful with format(), don't use '{:}' because ``_id`` is not
            # always a number.
            key += '{}{}'.format(self._id, SEP)
            key_indexes = SEP.join([self.__class__._serie_name(),
                                    __class__._S_INDEXES, ''])

        key_count = 0
        change_count = 0

        allvars = vars(self)
        for name in allvars:
            if name.startswith('_'): continue

            key_count += 1
            value = allvars[name]

            if self._id >= 0 and hasattr(self.__class__, name) and \
                name in self.__class__._INDEXES:
                key_index = key_indexes + name
                if value != None:
                    # A bug with redis-py. Some time the form ``member, score``
                    # is reversed. So we use a ``dict`` here for the form
                    # ``member=score``.
                    pipe.zadd(key_index,
                              **{str(self._id): self.__class__._score(value)})
                else:
                    pipe.zrem(key_index, str(self._id))

            name = key + name

            _type = type(value)
            if _type == dict:
                pipe.hmset(name, value)
                change_count += 1
            elif _type == set:
                pipe.sadd(name, *value)
                change_count += 1
            elif _type == list:
                pipe.delete(name)
                pipe.rpush(name, *value)
                change_count += 1
            elif _type == datetime.datetime:
                pipe.hmset(name, {
                           __class__._S_YEAR: value.year,
                           __class__._S_MONTH: value.month,
                           __class__._S_DAY: value.day,
                           __class__._S_HOUR: value.hour,
                           __class__._S_MINUTE: value.minute,
                           __class__._S_SECOND: value.second,
                           __class__._S_MICROSECOND: value.microsecond })
                if value.tzinfo:
                    pipe.hset(name, __class__._S_UTC_OFFSET,
                              int(value.tzinfo.utcoffset(None).total_seconds()))
                change_count += 1
            elif _type == datetime.date:
                pipe.hmset(name, {
                           __class__._S_YEAR: value.year,
                           __class__._S_MONTH: value.month,
                           __class__._S_DAY: value.day })
                change_count += 1
            elif _type == datetime.time:
                pipe.hmset(name, {
                           __class__._S_HOUR: value.hour,
                           __class__._S_MINUTE: value.minute,
                           __class__._S_SECOND: value.second,
                           __class__._S_MICROSECOND: value.microsecond })
                if value.tzinfo:
                    pipe.hset(name, __class__._S_UTC_OFFSET,
                              int(value.tzinfo.utcoffset(None).total_seconds()))
                change_count += 1
            elif value != None:
                pipe.set(name, value)
                change_count += 1
            else:
                pipe.delete(name)
                change_count -= 1
            #.for

        # ADD OR REMOVE _id to ID SET

        if key_count + change_count == 0:
            self.__class__._remove_id(self._id)
        else:
            self.__class__._add_id(self._id)

        pipe.execute()
        #.put()

    def delete(self):
        ''' Deletes this item. Returns ``True`` if OK, ``None`` otherwise.
        '''

        keys = []
        keys_indexes = None
        if self.__class__._AUTO_ID and self._id >= 0 and self.__class__._INDEXES:
            keys_indexes = []

        # Always use dir() (instead of vars()) to get the variables from parent
        # classes.
        for name in dir(self.__class__):
            if name.startswith('_'): continue

            attr = getattr(self.__class__, name)
            if callable(attr) and not isinstance(attr, type): continue

            if self._id >= 0:
                _ = SEP.join([self.__class__._serie_name(), str(self._id), name])
                if keys_indexes != None and name in self.__class__._INDEXES:
                    keys_indexes.append(name)
            else:
                _ = SEP.join([self.__class__._serie_name(), name])
            keys.append(_)
            #.for

        if keys:
            R.delete(*keys)
            self.__class__._remove_id(self._id)
        if keys_indexes:
            _ = SEP.join([self.__class__._serie_name(), __class__._S_INDEXES, ''])
            for name in keys_indexes:
                R.zrem(_ + name, str(self._id))
        #.delete()
