"""
    requires.chain
    ~~~~~~~~~~
    Represents a series of methods to be called
    against the object and the return object of
    the resultant calls before the needed value
    is requested. Example::

        >>> from requires.chain import chain
        >>> stack = []
        >>> chain([]).append(1).value()
        >>> stack
        [1]

    It is somewhat like the ``chain`` function
    is underscore.js. You can also use use this
    when constructing tests or just practically
    anything in general.
"""

class _Method(object):
    def __init__(self, instance, name):
        self.name = name
        self.instance = instance

    def __call__(self, *args, **kwargs):
        self.instance.stack.append((self.name, (args, kwargs)))
        return self.instance

class chain(object):
    """
    Represents a series of methods to be
    called upon the given object. Usage::

        >>> from requires.chain import chain
        >>> chain([]).append(1).value()
        [1]

    :param obj: The target object.
    :param continuous: Whether to continuously
        use the wrapped object only or to
        use the results of the previous
        method call.
    """
    def __init__(self, obj, continuous=False):
        self.obj    = obj
        self.stack  = []
        self.continuous = continuous

    def _cont(self):
        for method, arguments in self.stack:
            args, kwargs = arguments
            getattr(self.obj, method)(*args, **kwargs)
        return self.obj

    def value(self):
        """
        Gets the value of the chained
        object by calling all of the
        chained functions and then
        calling them. For example::

            >>> chain([]).append(1).value()
            [1]
        """
        if self.continuous:
            return self._cont()
        memo = self.obj
        for method, arguments in self.stack:
            args, kwargs = arguments
            memo = getattr(memo, method)(*args, **kwargs)
        return memo

    def __getattr__(self, value):
        if value in self.__dict__:
            return self.__dict__[value]
        return _Method(self, value)

