"""
    requires.expect
    ~~~~~~~~~~~

    Helps construct the assertion tests,
    BDD-style in a very readable and
    natural way. Easily construct tests
    in a few function calls::

        >>> from requires.expect import expect
        >>> expect([]).to.have.length(0)

    There are a few quirks however to
    achieve readbility- the keywords
    in the following list just return
    the expect object itself:

     - to
     - be
     - been
     - Is
     - that
     - And
     - have
     - With
     - at
     - of
     - same
"""

from requires import _NATURAL_KEYWORDS
from requires.assertion import Assertion, get, getkey

__all__ = ['expect']

class _Method(object):
    def __init__(self, instance):
        self.__ins = instance

    def __setattr__(self, key, value):
        self.__dict__[key] = value

    def __getattr__(self, value):
        if value in self.__dict__:
            return self.__dict__[value]
        return getattr(self.__ins, value)

    def __call__(self, value):
        self.__ins.suite.lengthIs(value)
        return self.__ins


class expect(object):
    """
    The object to help create the assertion
    functions around a given object, i.e.::

        >>> from requires.expect import expect
        >>> expect([])

    :param obj: The subject of the assertion
        tests, i.e. the object to test your
        declarations against.
    """
    def __init__(self, obj):
        self.suite = Assertion(obj)

    @property
    def root(self):
        """
        Explicitly gets the current ``expect``
        instance, i.e. the object itself. Used
        for assigning to the `natural` keywords.
        A simple demonstration::

            >>> e = expect([])
            >>> e.root == e
        """
        return self

    def Return(self, value, *args, **kwargs):
        self.suite.Return(value, *args, **kwargs)
        return self

    def method(self, name):
        """
        Asserts that the wrapped object has a
        given method (any attribute that is
        callable, even classes/instances).
        Usage::

            >>> expect([]).to.have.method('append')

        :param name: The name of the method.
        """
        self.suite.method(name)
        return self

    @property
    def Not(self):
        """
        Toggles the internal ``not`` flag,
        which is a flag used to determine
        the success of tests. For example::

            >>> expect([]).to.Not.have.method('x')
        """
        value = True
        if self.suite.Not:
            value = False
        self.suite.Not = value
        return self

    def a(self, type_):
        """
        Asserts that the object is of the
        given type using the built in
        ``isinstance`` function. Usage::

            >>> expect([]).to.be.a(list)

        :param type_: The type which you
            think it should be.
        """
        self.suite.istype(type_)
        return self

    an = a

    def include(self, item):
        """
        Asserts that the wrapped object
        contains the given item using
        a ``x in y`` expression. For
        example::

            >>> expect({'a':True}).to.include('a')

        :param item: The item which
            you want to assert your
            wrapped object contains.
        """
        self.suite.include([item])
        return self

    def ok(self):
        """
        Asserts that the wrapped object
        is of a truthy value, i.e. not
        an empty list, dictionary, or
        string. Example::

            >>> expect(True).to.be.ok()

        An alias to this function is
        ``exists``, which is made to
        increase readability when working
        with asynchronous code::

            >>> res, err = async_func(0)
            >>> expect(err).to.Not.exist()
        """
        self.suite.ok()
        return self

    exist = ok

    def element(self, value):
        """
        Asserts that the wrapped object
        has an element within it available
        under the given index and returns
        a new ``expect`` object with the
        element. For example::

            >>> expect([1,2,3]).to.have.element(0).that.equal(1)

        Deep elements (recursive lookups)
        are also supported, for example::

            >>> expect([[0]]).to.have.deep.element('0.0')

        :param value: An integer or string
            that can be used to index the
            wrapped object.
        """
        value = self.suite.element(value)
        return expect(value)

    def empty(self):
        """
        Asserts that the length of the
        object is zero. The following
        are essentially equivalent::

            >>> expect([]).to.be.empty()
            >>> expect([]).to.have.length(0)
        """
        self.suite.length = True
        self.suite.lengthIs(0)
        self.suite.length = False
        return self

    def key(self, string):
        """
        Asserts that the wrapped object has
        an object that can be indexed under
        the given key and returns a new
        ``expect`` object containing the
        particular object. Example::

            >>> expect({'a':1}).to.have.key('a').that.equal(1)

        Nested lookups are also available if
        you allow the ``deep`` flag, for
        example::

            >>> expect({'a':{'b':1}}).to.have.deep.key('a.b')

        :param string: The key that can be
            used to access the object. It
            does not have to be an instance
            of ``str``.
        """
        if not self.suite.deep:
            subject = self.suite.obj[string]
        else:
            subject = getkey(self.suite.obj, string)
        instance = expect(subject)
        return instance

    @property
    def itself(self):
        """
        Sets the context to the wrapped
        object itself, i.e.::

            >>> expect([]).length(0).And\
                    .itself.have.method('append')
        """
        if hasattr(self, 'main'):
            return self.main
        self.suite.length = False
        return self

    def equal(self, value):
        """
        Asserts that the current context
        (either length or object itself)
        is equal to the given value. An
        example::

            >>> expect(1+1).equal(2)
            >>> expect([]).length.to.be.equal(0)

        :param value: The value to assert
            equality against.
        """
        self.suite.equals(value)
        return self

    def above(self, value):
        """
        Asserts that the current context
        is greater than the given value::

            >>> expect(1+1).above(1)
        :param value: The value to use
            during the assertion.
        """
        self.suite.gt(value)
        return self

    def least(self, value):
        """
        Expects that the current context
        is greater than or equal to the
        given value, for example::

            >>> expect(1+1).at.least(2)

        :param value: The value to use
            during the assertion.
        """
        self.suite.ge(value)
        return self

    def below(self, value):
        """
        Asserts that the context is lesser
        than the given value. For example::

            >>> expect(0).to.be.below(1)
        
        :param value: The value to use
            during the assertion.
        """
        self.suite.lt(value)
        return self

    def most(self, value):
        """
        Asserts that the context is either
        lesser or equal to the supplied
        value. Example usage::

            >>> expect(0).to.be.at.most(0)
            >>> expect([1]).length.to.be.at.most(1)

        :param value: The value to use
            during the assertion.
        """
        self.suite.le(value)
        return self

    def within(self, minimum, maximum):
        """
        Assert that the current context
        is within the given maximum
        bounds, i.e. ``1 < x < 3``.
        Example usage would be something
        of the following::

            >>> expect(1).to.be.within(0,2)

        :param minimum: The lower bound.
        :param maximum: The upper bound.
        """
        self.suite.within(minimum, maximum)
        return self

    def attribute(self, *params):
        """
        Assert that either the object has
        a given attribute or that the
        object's given attribute equals
        to the value. For example::

            >>> expect(list).to.have.attribute('append')
            >>> expect(human).to.have.attribute("age", 13)

        You must either call this function
        with two arguments or one, any greater
        would not be accepted and raise an
        error.
        """
        if len(params) > 1:
            self.suite.attribute(*params)
            return self
        self.suite.have(*params)
        return self

    @property
    def length(self):
        """
        Returns an object which you can
        call to immediately assert the
        length of the wrapped object,
        or chain to more functions in
        order to set the context of
        assertions to that of length.
        Example::

            >>> expect([]).to.have.length(0)
            >>> expect([]).to.have.length.within(-1,1)
        """
        self.suite.length = True
        return _Method(self)

    def keys(self, *args):
        """
        Asserts that the wrapped object,
        which must implement a ``keys``
        attribute, has the given keys,
        i.e. includes the given keys.
        Example usage::

            >>> expect({'a':1}).to.have.keys('a')

        :param args: Positional arguments
            consisting of the list of keys
            that you want the object to
            contain.
        """
        self.suite.keys(*args)
        return self

    def throw(self, exception, *args, **kwargs):
        """
        Asserts that the wrapped object
        must throw a given exception
        when called with the given
        positional and keyword arguments.
        For example::

            >>> def func(x):
            ...     return 1/x
            ...
            >>> expect(func).to.throw(ValueError, 0)

        :param exception: The exception
            to assert that it raises.
        """
        self.suite.throw(exception, *args, **kwargs)
        return self

    def satisfy(self, function):
        """
        Asserts that the given function
        must return a truthy value when
        the wrapped object is passed to
        it. For example::

            >>> expect(1).to.satisfy(lambda x:x>0)

        :param function: The function to
            call with the wrapped object.
        """
        self.suite.satisfy(function)
        return self

    def members(self, mem):
        """
        Asserts that the object is a
        superset of the passed in object,
        i.e. you can have::

            >>> expect({1,2}).to.have.members({1})

        :param mem: The iterable of
            objects that the wrapped
            object should contain.
        """
        self.suite.include(mem)
        return self

    @property
    def deep(self):
        """
        Sets the internal ``deep`` flag so
        nested queries/assertions are made
        possible::

            >>> expect([[0]]).to.have.deep.element('0.0')
        """
        value = True
        if self.suite.deep:
            value = False
        self.suite.deep = value
        return self

    # Only define this function at the
    # end of the class to not override
    # any chainable stuff
    def property(self, string):
        """
        Asserts that the object would have
        a given property and returns a new
        ``expect`` object containing the
        property. For example::

            >>> class Person(object):
            ...     def __init__(self, name):
            ...         self.name = name
            ...
            >>> person = Person('Maker')
            >>> expect(person).to.have.property('name')

        When the ``deep`` flag is set, nested
        properties are also possible::

            >>> expect(person).to.have.deep.property('name.strip')

        :param string: The name of the
            expected property.
        """
        if not self.suite.deep:
            subject = getattr(self.suite.obj, string)
        else:
            subject = get(self.suite.obj, string)
        return expect(subject)



# assign the natural keywords to
# the object only once to minimize
# instantiation time.
for item in _NATURAL_KEYWORDS:
    setattr(expect, item, expect.root)

