# -*- coding: utf-8 -*-
#
# This file is part of RestAuthCommon (https://common.restauth.net).
#
# RestAuthCommon is free software: you can redistribute it and/or modify it under the terms of the
# GNU General Public License as published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# RestAuthCommon is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
# without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
# the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with RestAuthCommon. If
# not, see <http://www.gnu.org/licenses/>.

"""Classes and methods related to content handling.

.. moduleauthor:: Mathias Ertl <mati@restauth.net>
"""

from __future__ import unicode_literals

import json as libjson
import pickle
import sys

from RestAuthCommon import error

PY2 = sys.version_info[0] == 2
PY3 = sys.version_info[0] == 3

if PY2:  # pragma: py2
    from urlparse import parse_qs
    from urllib import urlencode

    string_types = basestring
else:  # pragma: py3
    from urllib.parse import parse_qs
    from urllib.parse import urlencode

    string_types = (str, bytes, )


class ContentHandler(object):
    """A common base class for all content handlers.

    If you want to implement your own content handler, you must subclass this class and implement
    all marshal_* and unmarshal_* methods.

    **Never use this class directly.** It does not marshal or unmarshal any content itself.

    Any keyword arguments will be set as instance attributes. This means that you can instantiate
    a handler with different settings, for example, this would instantiate a
    :py:class:`PickleContentHandler`, that uses pickle protocol version 1:

        >>> h = PickleContentHandler().PROTOCOL
        2
        >>> PickleContentHandler(PROTOCOL=1).PROTOCOL
        1
    """

    mime = None
    """Override this with the MIME type handled by your handler."""

    librarypath = None
    """Override ``librarypath`` to lazily load named library upon first use.

    This may be a toplevel module (e.g. ``"json"``) or a submodule (e.g.  ``"lxml.etree"``). The
    named library is accessable via ``self.library``.

    Example::

        class XMLContentHandler(ContentHandler):
            librarypath = 'lxml.etree'

            def unmarshal_str(self, data):
                tree = self.library.Element(data)
                # ...
    """

    SUPPORT_NESTED_DICTS = True
    """Set to False if your content handler does not support nested dictionaries as used e.g.
    during user-creation."""

    _library = None

    @property
    def library(self):
        """Library configured with the ``librarypath`` class variable."""
        if self._library is None:
            if '.' in self.librarypath:
                mod, lib = self.librarypath.rsplit('.', 1)
                _temp = __import__(mod, fromlist=[str(lib)])
                self._library = getattr(_temp, lib)
            else:
                self._library = __import__(self.librarypath)
        return self._library

    def __init__(self, **kwargs):
        for k, w in kwargs.items():
            setattr(self, k, w)

    def _normalize_list3(self, l):  # pragma: py3
        """Converts any byte objects of l to str objects."""
        return [e.decode('utf-8') if isinstance(e, bytes) else e for e in l]

    def _normalize_list2(self, l):  # pragma: py2
        """Converts any str objects of l to unicode objects."""
        return [e.decode('utf-8') if isinstance(e, str) else e for e in l]

    def _normalize_dict3(self, d):  # pragma: py3
        """Converts any keys or values of d that are bytes to str."""
        def conv(v):
            if isinstance(v, bytes):
                return v.decode('utf-8')
            elif isinstance(v, dict):
                return self._normalize_dict3(v)
            return v

        return dict((conv(k), conv(v)) for k, v in d.items())

    def _normalize_dict2(self, d):  # pragma: py2
        """Converts any keys or values of d that are str to unicode."""
        def conv(v):
            if isinstance(v, str):
                return v.decode('utf-8')
            elif isinstance(v, dict):
                return self._normalize_dict2(v)
            return v

        return dict((conv(k), conv(v)) for k, v in d.iteritems())

    def _normalize_str3(self, s):  # pragma: py3
        """Converts byte objects to str."""
        return s.decode('utf-8') if isinstance(s, bytes) else s

    def _normalize_str2(self, s):  # pragma: py2
        """Converts str objects to unicode."""
        return s.decode('utf-8') if isinstance(s, str) else s

    def marshal(self, obj):
        """Shortcut for marshalling just any object.

        .. NOTE:: If you know the type of **obj** in advance, you should use the marshal_* methods
        directly for improved speed.

        :param obj: The object to marshall.
        :return: The marshalled representation of the object.
        :rtype: str
        :raise error.MarshalError: If marshalling goes wrong in any way.
        """
        if isinstance(obj, string_types):
            func_name = 'marshal_str'
        else:
            func_name = 'marshal_%s' % (obj.__class__.__name__)

        try:
            func = getattr(self, func_name)
            return func(obj)
        except error.MarshalError as e:
            raise e
        except Exception as e:
            raise error.MarshalError(e)

    def unmarshal_str(self, data):  # pragma: no cover
        """Unmarshal a string.

        :param data: Data to unmarshal.
        :type  data: bytes in python3, str in python2
        :rtype: str in python3, unicode in python2
        """
        pass

    def unmarshal_dict(self, body):  # pragma: no cover
        """Unmarshal a dictionary.

        :param data: Data to unmarshal.
        :type  data: bytes in python3, str in python2
        :rtype: dict
        """
        pass

    def unmarshal_list(self, body):  # pragma: no cover
        """Unmarshal a list.

        :param data: Data to unmarshal.
        :type  data: bytes in python3, str in python2
        :rtype: list
        """
        pass

    def marshal_str(self, obj):  # pragma: no cover
        """Marshal a string.

        :param obj: Data to marshal.
        :type  obj: str, bytes, unicode
        :rtype: bytes in python3, str in python2
        """
        pass

    def marshal_list(self, obj):  # pragma: no cover
        """Marshal a list.

        :param obj: Data to marshal.
        :type  obj: list
        :rtype: bytes in python3, str in python2
        """
        pass

    def marshal_dict(self, obj):  # pragma: no cover
        """Marshal a dictionary.

        :param obj: Data to marshal.
        :type  obj: dict
        :rtype: bytes in python3, str in python2
        """
        pass

    if PY3:  # pragma: py3
        normalize_str = _normalize_str3
        normalize_list = _normalize_list3
        normalize_dict = _normalize_dict3
    else:  # pragma: py2
        normalize_str = _normalize_str2
        normalize_list = _normalize_list2
        normalize_dict = _normalize_dict2


class JSONContentHandler(ContentHandler):
    """Handler for JSON encoded content.

    .. seealso:: `Specification <http://www.json.org>`_, `WikiPedia
       <http://en.wikipedia.org/wiki/JSON>`_
    """

    mime = 'application/json'
    """The mime-type used by this content handler is 'application/json'."""

    SEPARATORS = (str(','), str(':'))

    class ByteEncoder(libjson.JSONEncoder):
        if PY3:  # pragma: py3
            def decode_dict(self, d):
                def conv(v):
                    if isinstance(v, bytes):
                        return v.decode('utf-8')
                    elif isinstance(v, dict):
                        return self.decode_dict(v)
                    return v

                return dict((conv(k), conv(v)) for k, v in d.items())

            def encode(self, obj):
                if isinstance(obj, bytes):
                    obj = obj.decode('utf-8')
                elif isinstance(obj, dict):
                    obj = self.decode_dict(obj)

                try:
                    return libjson.JSONEncoder.encode(self, obj)
                except TypeError:
                    raise

            def default(self, obj):
                if isinstance(obj, bytes):
                    return obj.decode('utf-8')
                return libjson.JSONEncoder.default(self, obj)

    def unmarshal_str(self, body):
        try:
            pure = libjson.loads(self.normalize_str(body))
            if not isinstance(pure, list) or len(pure) != 1:
                raise error.UnmarshalError("Could not parse body as string")

            string = pure[0]

            # In python 2.7.1 (not 2.7.2) json.loads("") returns a str and
            # not unicode.
            return self.normalize_str(string)
        except ValueError as e:
            raise error.UnmarshalError(e)

    def unmarshal_dict(self, body):
        try:
            return libjson.loads(self.normalize_str(body))
        except ValueError as e:
            raise error.UnmarshalError(e)

    def unmarshal_list(self, body):
        try:
            return libjson.loads(self.normalize_str(body))
        except ValueError as e:
            raise error.UnmarshalError(e)

    def marshal_str(self, obj):
        try:
            dumped = libjson.dumps([obj], separators=self.SEPARATORS, cls=self.ByteEncoder)
            return dumped.encode('utf-8')
        except Exception as e:
            raise error.MarshalError(e)

    def marshal_list(self, obj):
        try:
            dumped = libjson.dumps(obj, separators=self.SEPARATORS, cls=self.ByteEncoder)
            return dumped.encode('utf-8')
        except Exception as e:
            raise error.MarshalError(e)

    def marshal_dict(self, obj):
        try:
            dumped = libjson.dumps(obj, separators=self.SEPARATORS, cls=self.ByteEncoder)
            return dumped.encode('utf-8')
        except Exception as e:
            raise error.MarshalError(e)


class BSONContentHandler(ContentHandler):
    """Handler for BSON ("Binary JSON") encoded content.

    .. NOTE:: This contant handler requires either the ``pymongo`` or the ``bson`` library to be
       installed.

    .. seealso:: `Specification <http://bsonspec.org/>`_, `WikiPedia
       <http://en.wikipedia.org/wiki/BSON>`_,
       `pymongo <https://pypi.python.org/pypi/pymongo>` on PyPi,
       `bson <https://pypi.python.org/pypi/bson>` on PyPi
    """
    mime = 'application/bson'
    """The mime-type used by this content handler is 'application/json'."""

    librarypath = 'bson'

    def __init__(self, **kwargs):
        super(BSONContentHandler, self).__init__(**kwargs)

        if hasattr(self.library, 'BSON'):
            self.dumps = self.library.BSON.encode
            self.loads = self.library.BSON.decode
        else:
            self.dumps = self.library.dumps
            self.loads = self.library.loads

    def marshal_dict(self, obj):
        return self.dumps({'d': self.normalize_dict(obj), })

    def marshal_list(self, obj):
        return self.dumps({'l': self.normalize_list(obj), })

    def marshal_str(self, obj):
        return self.dumps({'s': self.normalize_str(obj), })

    def _unmarshal_dict2(self, body):
        if isinstance(body, unicode):
            body = body.encode('utf-8')
        return self.loads(body)['d']

    def _unmarshal_list2(self, body):
        if isinstance(body, unicode):
            body = body.encode('utf-8')
        return self.loads(body)['l']

    def _unmarshal_str2(self, body):
        if isinstance(body, unicode):
            body = body.encode('utf-8')
        return self.loads(body)['s']

    def _unmarshal_dict3(self, body):
        return self.loads(body)['d']

    def _unmarshal_list3(self, body):
        return self.loads(body)['l']

    def _unmarshal_str3(self, body):
        return self.loads(body)['s']

    if PY3:
        unmarshal_dict = _unmarshal_dict3
        unmarshal_list = _unmarshal_list3
        unmarshal_str = _unmarshal_str3
    else:
        unmarshal_dict = _unmarshal_dict2
        unmarshal_list = _unmarshal_list2
        unmarshal_str = _unmarshal_str2


class MessagePackContentHandler(ContentHandler):
    """Handler for MessagePack encoded content.

    .. NOTE:: This content handler requires the ``msgpack-python`` library to be installed.

    .. seealso:: `Specification <http://msgpack.org/>`_, `WikiPedia
       <http://en.wikipedia.org/wiki/MessagePack>`_, `msgpack-python
       <https://pypi.python.org/pypi/msgpack-python/>`_ on PyPI.
    """
    mime = 'application/messagepack'
    """The mime-type used by this content handler is 'application/messagepack'."""

    librarypath = 'msgpack'

    def marshal_dict(self, obj):
        try:
            return self.library.packb(self.normalize_dict(obj))
        except Exception as e:
            raise error.MarshalError(e)

    def marshal_list(self, obj):
        try:
            return self.library.packb(self.normalize_list(obj))
        except Exception as e:
            raise error.MarshalError(e)

    def marshal_str(self, obj):
        try:
            return self.library.packb(self.normalize_str(obj))
        except Exception as e:
            raise error.MarshalError(e)

    def unmarshal_dict(self, body):
        return self.normalize_dict(self.library.unpackb(body))

    def unmarshal_list(self, body):
        return self.normalize_list(self.library.unpackb(body))

    def unmarshal_str(self, body):
        return self.normalize_str(self.library.unpackb(body))


class FormContentHandler(ContentHandler):
    """Handler for HTML Form urlencoded content.

    .. WARNING:: Because of the limitations of urlencoded forms, this handler does not support
       nested dictionaries. The primary use of this content handler is to enable you to manually
       add data with e.g. the curl command line utility, where you don't want to serialize posted
       data manually. Do not use this class in production.
    """

    mime = 'application/x-www-form-urlencoded'
    """The mime-type used by this content handler is 'application/x-www-form-urlencoded'."""

    SUPPORT_NESTED_DICTS = False

    def _decode_dict(self, d):  # pragma: py2
        decoded = {}
        for key, value in d.items():
            key = key.decode('utf-8')
            if isinstance(value, (str, unicode)):
                decoded[key] = value.decode('utf-8')
            elif isinstance(value, list):  # pragma: no cover
                decoded[key] = [e.decode('utf-8') for e in value]
            elif isinstance(value, dict):  # pragma: no cover
                decoded[key] = self._decode_dict(value)

        return decoded

    def unmarshal_dict(self, body):
        if PY3:  # pragma: py3
            body = body.decode('utf-8')

        parsed_dict = parse_qs(body, True)
        ret_dict = {}
        for key, value in parsed_dict.items():
            if isinstance(value, list) and len(value) == 1:
                ret_dict[key] = value[0]
            else:
                ret_dict[key] = value

        if PY2:  # pragma: py2
            ret_dict = self._decode_dict(ret_dict)

        return ret_dict

    def unmarshal_list(self, body):
        if PY3:  # pragma: py3
            body = body.decode('utf-8')

        if body == '':
            return []

        parsed = parse_qs(body, True)['list']

        if PY2:  # pragma: py2
            parsed = [e.decode('utf-8') for e in parsed]
        return parsed

    def unmarshal_str(self, body):
        if PY3:  # pragma: py3
            body = body.decode('utf-8')

        parsed = parse_qs(body, True)['str'][0]
        return self.normalize_str(parsed)

    def marshal_str(self, obj):
        try:
            if PY2:  # pragma: py2
                obj = obj.encode('utf-8')
                return urlencode({'str': obj})
            else:  # pragma: py3
                return urlencode({'str': obj}).encode('utf-8')
        except Exception as e:
            raise error.MarshalError(e)

    def _encode_dict(self, d):  # pragma: py2
        encoded = {}
        for key, value in d.items():
            key = key.encode('utf-8')
            if isinstance(value, (str, unicode)):
                encoded[key] = value.encode('utf-8')
            elif isinstance(value, list):  # pragma: no cover
                encoded[key] = [e.encode('utf-8') for e in value]
            elif isinstance(value, dict):  # pragma: no branch
                encoded[key] = self._encode_dict(value)

        return encoded

    def marshal_dict(self, obj):
        try:
            if PY2:  # pragma: py2
                obj = self._encode_dict(obj)

            # verify that no value is a dictionary, because the unmarshalling for
            # that doesn't work:
            for v in obj.values():
                if isinstance(v, dict):
                    raise error.MarshalError(
                        "FormContentHandler doesn't support nested dictionaries.")
            if PY3:  # pragma: py3
                return urlencode(obj, doseq=True).encode('utf-8')
            else:  # pragma: py2
                return urlencode(obj, doseq=True)
        except Exception as e:
            raise error.MarshalError(e)

    def marshal_list(self, obj):
        try:
            if PY2:  # pragma: py2
                obj = [e.encode('utf-8') for e in obj]
                return urlencode({'list': obj}, doseq=True)
            else:  # pragma: py3
                return urlencode({'list': obj}, doseq=True).encode('utf-8')
        except Exception as e:
            raise error.MarshalError(e)


class PickleContentHandler(ContentHandler):
    """Handler for pickle-encoded content.

    .. seealso:: `module documentation
       <http://docs.python.org/2/library/pickle.html>`_,
       `WikiPedia <http://en.wikipedia.org/wiki/Pickle_(Python)>`_
    """

    mime = 'application/pickle'
    """The mime-type used by this content handler is 'application/pickle'."""

    PROTOCOL = 2

    def marshal_str(self, obj):
        try:
            return pickle.dumps(self.normalize_str(obj), protocol=self.PROTOCOL)
        except Exception as e:
            raise error.MarshalError(str(e))

    def marshal_dict(self, obj):
        try:
            return pickle.dumps(self.normalize_dict(obj), protocol=self.PROTOCOL)
        except Exception as e:
            raise error.MarshalError(str(e))

    def marshal_list(self, obj):
        try:
            return pickle.dumps(self.normalize_list(obj), protocol=self.PROTOCOL)
        except Exception as e:
            raise error.MarshalError(str(e))

    def unmarshal_str(self, data):
        try:
            return self.normalize_str(pickle.loads(data))
        except Exception as e:
            raise error.UnmarshalError(str(e))

    def unmarshal_list(self, data):
        try:
            return self.normalize_list(pickle.loads(data))
        except Exception as e:
            raise error.UnmarshalError(str(e))

    def unmarshal_dict(self, data):
        try:
            return self.normalize_dict(pickle.loads(data))
        except Exception as e:
            raise error.UnmarshalError(str(e))


class Pickle3ContentHandler(PickleContentHandler):
    """Handler for pickle-encoded content, protocol level version 3.

    This version is only supported by the Python3 version the pickle module, this ContentHandler is
    only usable in Python3.

    .. seealso:: `module documentation <http://docs.python.org/3/library/pickle.html>`_,
       `WikiPedia <http://en.wikipedia.org/wiki/Pickle_(Python)>`_
    """

    mime = 'application/pickle3'
    """The mime-type used by this content handler is 'application/pickle3'."""

    PROTOCOL = 3


class YAMLContentHandler(ContentHandler):
    """Handler for YAML encoded content.

    .. NOTE:: This ContentHandler requires the ``PyYAML`` library to be installed.

    .. seealso:: `Specification <http://www.yaml.org/>`_, `WikiPedia
      <http://en.wikipedia.org/wiki/YAML>`_, `PyYAML <https://pypi.python.org/pypi/PyYAML>`_ on
      PyPI
    """
    mime = 'application/yaml'
    """The mime-type used by this content handler is 'application/yaml'."""

    librarypath = 'yaml'

    def _marshal_str3(self, obj):  # pragma: py3
        return self.library.dump(obj).encode('utf-8')

    def _marshal_str2(self, obj):  # pragma: py2
        return self.library.dump(obj)

    def marshal_str(self, obj):
        try:
            return self._marshal_str(self.normalize_str(obj))
        except Exception as e:
            raise error.MarshalError(e)

    def _marshal_dict3(self, obj):  # pragma: py3
        return self.library.dump(obj).encode('utf-8')

    def _marshal_dict2(self, obj):  # pragma: py2
        return self.library.dump(obj)

    def marshal_dict(self, obj):
        try:
            return self._marshal_dict(self.normalize_dict(obj))
        except Exception as e:
            raise error.MarshalError(e)

    def _marshal_list3(self, obj):  # pragma: py3
        return self.library.dump(obj).encode('utf-8')

    def _marshal_list2(self, obj):  # pragma: py2
        return self.library.dump(obj)

    def marshal_list(self, obj):
        try:
            return self._marshal_list(self.normalize_list(obj))
        except Exception as e:
            raise error.MarshalError(e)

    def unmarshal_str(self, data):
        try:
            unmarshalled = self.library.load(data)
            return self.normalize_str(unmarshalled)
        except self.library.YAMLError as e:  # pragma: no cover
            raise error.UnmarshalError(e)

    def unmarshal_list(self, data):
        try:
            return self.normalize_list(self.library.load(data))
        except self.library.YAMLError as e:
            raise error.UnmarshalError(e)

    def unmarshal_dict(self, data):
        try:
            return self.normalize_dict(self.library.load(data))
        except self.library.YAMLError as e:
            raise error.UnmarshalError(e)

    if PY3:  # pragma: py3
        _marshal_str = _marshal_str3
        _marshal_dict = _marshal_dict3
        _marshal_list = _marshal_list3
    else:  # pragma: py2
        _marshal_str = _marshal_str2
        _marshal_dict = _marshal_dict2
        _marshal_list = _marshal_list2


class XMLContentHandler(ContentHandler):
    """Handler for XML encoded data.

    .. NOTE:: This ContentHandler requires the ``lxml`` library to be installed.

    .. seealso:: `lxml <https://pypi.python.org/pypi/lxml>`_ on PyPI
    """

    mime = 'application/xml'
    """The mime-type used by this content handler is 'application/xml'."""

    librarypath = 'lxml.etree'

    def unmarshal_str(self, data):
        text = self.library.fromstring(data).text
        if text is None:
            text = ''

        return self.normalize_str(text)

    def _unmarshal_dict(self, tree):
        d = {}

        # find all strings
        for e in tree.iterfind('str'):
            if e.text is None:
                d[e.attrib['key']] = ''
            else:
                d[e.attrib['key']] = e.text

        # parse subdictionaries
        for subdict in tree.iterfind('dict'):
            d[subdict.attrib['key']] = self._unmarshal_dict(subdict)

        return d

    def unmarshal_dict(self, body):
        d = self._unmarshal_dict(self.library.fromstring(body))
        return self.normalize_dict(d)

    def unmarshal_list(self, body):
        l = []
        for elem in self.library.fromstring(body).iterfind('str'):
            if elem.text is None:
                l.append('')
            else:
                l.append(elem.text)
        return self.normalize_list(l)

    def marshal_str(self, obj):
        try:
            obj = self.normalize_str(obj)
            root = self.library.Element('str')
            root.text = obj
            return self.library.tostring(root)
        except Exception as e:
            raise error.MarshalError(e)

    def marshal_list(self, obj):
        try:
            obj = self.normalize_list(obj)

            root = self.library.Element('list')
            for value in obj:
                elem = self.library.Element('str')
                elem.text = value
                root.append(elem)
            return self.library.tostring(root)
        except Exception as e:
            raise error.MarshalError(e)

    def _marshal_dict(self, obj, key=None):
        root = self.library.Element('dict')
        if key is not None:
            root.attrib['key'] = key

        obj = self.normalize_dict(obj)

        for key, value in obj.items():
            if isinstance(value, dict):
                root.append(self._marshal_dict(value, key=key))
            else:
                elem = self.library.Element('str', attrib={'key': key})
                elem.text = value
                root.append(elem)
        return root

    def marshal_dict(self, obj):
        try:
            return self.library.tostring(self._marshal_dict(obj))
        except Exception as e:
            raise error.MarshalError(e)


CONTENT_HANDLERS = {
    'application/bson': BSONContentHandler,
    'application/json': JSONContentHandler,
    'application/messagepack': MessagePackContentHandler,
    'application/pickle': PickleContentHandler,
    'application/pickle3': Pickle3ContentHandler,
    'application/x-www-form-urlencoded': FormContentHandler,
    'application/xml': XMLContentHandler,
    'application/yaml': YAMLContentHandler,
}
"""
Mapping of MIME types to their respective handler implemenation. You can use this dictionary to
dynamically look up a content handler if you do not know the requested content type in advance.

================================= ===============================================
MIME type                         Handler
================================= ===============================================
application/json                  :py:class:`.handlers.JSONContentHandler`
application/x-www-form-urlencoded :py:class:`.handlers.FormContentHandler`
application/pickle                :py:class:`.handlers.PickleContentHandler`
application/pickle3               :py:class:`.handlers.Pickle3ContentHandler`
application/xml                   :py:class:`.handlers.XMLContentHandler`
application/yaml                  :py:class:`.handlers.YAMLContentHandler`
application/bson                  :py:class:`.handlers.BSONContentHandler`
application/messagepack           :py:class:`.handlers.MessagePackContentHandler`
================================= ===============================================

If you want to provide your own implementation of a :py:class:`.ContentHandler`, you can add it to
this dictionary with the appropriate MIME type as the key.
"""

# old names, for compatability:
content_handler = ContentHandler
json = JSONContentHandler
xml = XMLContentHandler
form = FormContentHandler

# 'YamlContentHandler' was introduced in 0.6.1 and renamed for consistency to
# 'YAMLContentHandler' in 0.6.2
YamlContentHandler = YAMLContentHandler
