# -*- coding: UTF-8 -*-

from unittest import TestCase

class TestCtypesFortran(TestCase):
    """
    Test calling FORTRAN subroutine with scalars and arrays through ctypes.

    _ctypes is the extension module generated by f2py.  Without using the
    interface provided by f2py, the compiled FORTRAN subroutine can be accessed
    with ctypes.

    It should be noted that FORTRAN subroutines are called by reference, i.e.,
    all the arguements passed in are pointers.  ctypes.byref should be used to
    access the subroutines.

    Also, FORTRAN arrays are column-major while C arrays are row-major.  The
    order of indices of arrays should be transposed while passed into
    subroutine.

    Usually the symbol for FORTRAN subroutines in shared object files generated 
    by compiler/linker is appended with an underscore.

    @ivar lib_c_ctypes: loaded shared object file.
    @itype lib_c_ctypes: ctypes.CDLL
    @ivar a: one dimensional integer array.
    @itype a: numpy.ndarray
    @ivar b: two dimensional double array.
    @itype b: numpy.ndarray
    @ivar args: list of arguements for test FORTRAN subroutine.
    @itype args: list
    """

    def setUp(self):
        """
        Set up shared objects generated by f2py.

        f2py generates standard .pyd shared object file on Windows while .so
        under Linux.
        """
        import sys, os
        from ctypes import CDLL, c_int, c_double, POINTER, byref
        from numpy import arange
        from solvcon.dependency import guess_dllname
        from solvcon.conf import env
        # load the dll created by f2py and bind the subroutine.
        libpath = os.path.join(env.libdir, guess_dllname('sc_solvcontest'))
        self.lib_c_ctypes = CDLL(libpath)
        # create test arrays.
        self.a = arange(10, dtype='int32')
        self.b = arange(20, dtype='float64').reshape((4,5))
        # prepare arguments.
        self.args = [
            byref(c_int(0)),        # placeholder.
            byref(c_double(0.0)),   # placeholder.
            byref(c_int(self.a.shape[0])),
            self.a.ctypes._as_parameter_,
            byref(c_int(self.b.shape[1])),
            byref(c_int(self.b.shape[0])),
            self.b.ctypes._as_parameter_,
        ]

    def test_int(self):
        """
        Test for the 1D integer array.

        FORTRAN subroutine adds each element with a random value, and after that
        the value is substracted in the following Python code.
        """
        from random import randint
        from ctypes import byref, c_int
        # back up array.
        a_orig = self.a.copy()
        # run FORTRAN subroutine.
        tval = randint(0,10000000)
        self.args[0] = byref(c_int(tval))
        self.lib_c_ctypes.ctypes_test(*self.args)
        # revert in Python and test.
        self.a -= tval
        for i in range(len(self.a)):
            self.assertEqual(self.a[i], a_orig[i])

    def test_double(self):
        """
        Test for 2D double array.  Shape is also taken care.

        FORTRAN subroutine adds each element with a random value and subtracts 
        it from the first column.  In the following Python code, the value is
        substracted from the remaining rows.
        """
        from random import randint
        from ctypes import byref, c_double
        # back up array.
        b_orig = self.b.flatten()
        # run FORTRAN subroutine.
        tval = float(randint(0,10000000))
        self.args[1] = byref(c_double(tval))
        self.lib_c_ctypes.ctypes_test(*self.args)
        # revert in Python and test.
        self.b[1:,:] -= tval
        b = self.b.flatten()
        for i in range(len(b)):
            self.assertEqual(b[i], b_orig[i])

    def test_struct(self):
        from random import randint
        from ctypes import POINTER, byref, c_int, c_double, Structure, c_void_p
        from numpy import arange
        # build struct/type.
        class Record(Structure):
            _fields_ = [
                ('idx', c_int),
                ('arr', c_double * 10),
                ('arr_ptr', c_void_p),
            ]
        # fill data into struct/type.
        arr = arange(10, dtype='float64')
        arr1 = arr.copy()
        r1 = Record()
        r1.idx = 10
        r1.arr[:] = arr[:]
        r1.arr_ptr = arr1.ctypes._as_parameter_
        r2 = Record()
        r2.idx = 10
        r2.arr[:] = arr[:]
        # run foreign function.
        self.lib_c_ctypes.ctypes_type_test(byref(r1), byref(r2))
        # verify result.
        self.assertEqual(r1.idx, -r2.idx)
        for i in range(10):
            self.assertEqual(r1.arr[i], -r2.arr[i])
        for i in range(10):
            self.assertEqual(arr1[i], arr[i]+200)
