import sqlalchemy as sa
from sqlalchemy.ext.hybrid import hybrid_property
from sqlalchemy.orm.collections import attribute_mapped_collection
from sqlalchemy_utils.functions import get_primary_keys

from .comparators import TranslationComparator
from .exc import ImproperlyConfigured
from .utils import option


class HybridPropertyBuilder(object):
    def __init__(self, manager, translation_model):
        self.manager = manager
        self.translation_model = translation_model
        self.model = self.translation_model.__parent_class__

    def getter_factory(self, property_name):
        def attribute_getter(obj):
            value = getattr(obj.current_translation, property_name)
            if value:
                return value

            default_locale = self.manager.option(obj, 'fallback_locale')
            if callable(default_locale):
                default_locale = default_locale(obj)
            return getattr(
                obj.translations[default_locale],
                property_name
            )

        return attribute_getter

    def setter_factory(self, property_name):
        """
        Returns a hybrid property setter for given property name.

        :param property_name: Name of the property to generate a setter for
        """
        return (
            lambda obj, value:
            setattr(obj.current_translation, property_name, value)
        )

    def generate_hybrid(self, property_name):
        """
        Generates a SQLAlchemy hybrid property for given translation model
        property.

        :param property_name:
            Name of the translation model property to generate hybrid property
            for.
        """
        setattr(
            self.model,
            property_name,
            hybrid_property(
                fget=self.getter_factory(property_name),
                fset=self.setter_factory(property_name),
                expr=lambda cls: getattr(
                    cls.__translatable__['class'], property_name
                )
            )
        )

    def detect_collisions(self, property_name):
        """
        Detects possible naming collisions for given property name.

        :raises sqlalchemy_i18n.exc.ImproperlyConfigured: if the model already
            has a property with given name
        """
        mapper = sa.inspect(self.model)
        if mapper.has_property(property_name):
            raise ImproperlyConfigured(
                "Attribute name collision detected. Could not create "
                "hybrid property for translated attribute '%s'. "
                "An attribute with the same already exists in parent "
                "class '%s'." % (
                    property_name,
                    self.model.__name__
                )
            )

    def __call__(self):
        for column in self.translation_model.__table__.c:
            exclude = self.manager.option(
                self.model, 'exclude_hybrid_properties'
            )

            if column.key in exclude or column.primary_key:
                continue

            self.detect_collisions(column.key)
            self.generate_hybrid(column.key)


class RelationshipBuilder(object):
    def __init__(self, translation_cls):
        self.translation_cls = translation_cls
        self.parent_cls = self.translation_cls.__parent_class__

    @property
    def primary_key_conditions(self):
        conditions = []
        for key in get_primary_keys(self.parent_cls).keys():
            conditions.append(
                getattr(self.parent_cls, key) ==
                getattr(self.translation_cls, key)
            )
        return conditions

    def assign_single_translations(self):
        for locale in option(self.parent_cls, 'locales'):
            key = '_translation_%s' % locale
            if key in self.parent_cls.__dict__:
                continue

            conditions = self.primary_key_conditions
            conditions.append(self.translation_cls.locale == locale)
            setattr(
                self.parent_cls,
                key,
                sa.orm.relationship(
                    self.translation_cls,
                    primaryjoin=sa.and_(*conditions),
                    foreign_keys=list(
                        get_primary_keys(self.parent_cls).values()
                    ),
                    uselist=False,
                    viewonly=True
                )
            )

    def assign_current_translation(self):
        """
        Assigns the current translation relationship for translatable parent
        class.
        """
        try:
            current_locale = self.parent_cls.locale
        except NotImplementedError:
            pass
        else:
            if '_current_translation' not in self.parent_cls.__dict__:
                conditions = self.primary_key_conditions
                conditions.append(
                    self.translation_cls.locale == current_locale
                )

                self.parent_cls._current_translation = sa.orm.relationship(
                    self.translation_cls,
                    primaryjoin=sa.and_(*conditions),
                    foreign_keys=list(
                        get_primary_keys(self.parent_cls).values()
                    ),
                    viewonly=True,
                    uselist=False
                )

    def assign_translations(self):
        """
        Assigns translations relationship for translatable model. The assigned
        attribute is a relationship to all translation locales.
        """
        if not hasattr(self.parent_cls, '_translations'):
            foreign_keys = [
                getattr(self.translation_cls, column_key)
                for column_key in get_primary_keys(self.parent_cls).keys()
            ]

            self.parent_cls._translations = sa.orm.relationship(
                self.translation_cls,
                primaryjoin=sa.and_(*self.primary_key_conditions),
                foreign_keys=foreign_keys,
                collection_class=attribute_mapped_collection('locale'),
                comparator_factory=TranslationComparator,
                cascade='all, delete-orphan',
                passive_deletes=True,
            )

    def assign_translation_parent(self):
        if 'translation_parent' not in self.translation_cls.__dict__:
            setattr(
                self.translation_cls,
                'translation_parent',
                sa.orm.relationship(
                    self.parent_cls,
                    uselist=False,
                    viewonly=True
                )
            )

    def __call__(self):
        self.assign_single_translations()
        self.assign_current_translation()
        self.assign_translations()
        self.assign_translation_parent()
